package com.zoyi.channel.plugin.android.util;

import android.content.Context;
import android.support.annotation.Nullable;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity2;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.TimeRange;
import com.zoyi.channel.plugin.android.model.rest.WorkingTime;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.SimpleTimeZone;

public class ChannelUtils {

  private final static long MINUTES_TO_MILLISECONDS = 60L * 1000L;

  private final static int DAY_IN_MINUTES = 24 * 60;

  public static boolean isChannelPluginActivity(Context context) {
    return context instanceof BaseActivity2;
  }

  public static boolean isSameActivity(@Nullable Context context1, @Nullable Context context2) {
    return context1 != null && context2 != null && context1.hashCode() == context2.hashCode();
  }

  /**
   * Get remaining minutes to working time for channel
   *
   * @param channel   channel information
   * @param timestamp timestamp in millseconds for now without timezone
   * @return {@code -1} if not working and can not calculate remaining time {@code 0}  if channel is now working else, remaining time in
   * minutes to working time
   */
  public static int getMinutesLeftToWork(@Nullable Channel channel, @Nullable Long timestamp) {
    if (channel == null || timestamp == null || timestamp == 0L) {
      return -1;
    }
    if (Const.WORKING_TYPE_ALWAYS.equals(channel.getWorkingType())) {
      return 0;
    }
    if (Const.WORKING_TYPE_NEVER.equals(channel.getWorkingType())) {
      return -1;
    }

    WorkingTime workingTime = channel.getWorkingTime();

    if (workingTime == null || !workingTime.isValid()) {
      return 0;
    }

    long utcOffsetMilliseconds = channel.getUtcOffsetMinutes() * MINUTES_TO_MILLISECONDS;
    long now = timestamp + utcOffsetMilliseconds;

    Calendar calendar = Calendar.getInstance(new SimpleTimeZone(0, "UTC"));
    calendar.setTimeInMillis(now);

    int weekday = calendar.get(Calendar.DAY_OF_WEEK); // sunday is 1, saturday is 7
    int hours = calendar.get(Calendar.HOUR_OF_DAY);
    int minutes = calendar.get(Calendar.MINUTE);

    int totalRemainMinutes = 0;

    for (int daysDelta = 0; daysDelta <= 7; daysDelta++) {
      int nowMinutes = daysDelta == 0 ? hours * 60 + minutes : 0;
      int targetWeekday = (weekday - 1 + daysDelta) % 7 + 1;

      int remainMinutes = getRemainingMinuteForWeekday(
          workingTime.getWeekdayTimeRange(targetWeekday),
          channel.getLunchTime(),
          nowMinutes
      );

      if (remainMinutes == -1) {
        totalRemainMinutes += (DAY_IN_MINUTES - nowMinutes);
      } else {
        totalRemainMinutes += remainMinutes;
        break;
      }
    }
    return totalRemainMinutes;
  }

  /**
   * Get remaining minutes to working time for channel
   *
   * @param weekday    time range for weekday
   * @param lunch      channel lunch time
   * @param nowMinutes target timestamp
   * @return {@code -1} If this weekday did not working and must see next weekday. Return 0 or positive number for remaining minutes. That
   * weekday is working.
   */
  private static int getRemainingMinuteForWeekday(@Nullable TimeRange weekday, @Nullable TimeRange lunch, int nowMinutes) {
    if (weekday == null || !weekday.isValid()) {
      return -1;
    }

    List<TimeRange> workingRanges = new ArrayList<>();

    if (lunch != null && lunch.isValid()) {
      TimeRange restTime = new TimeRange(Math.max(weekday.getFrom(), lunch.getFrom()), Math.min(weekday.getTo(), lunch.getTo()));

      if (restTime.isValid()) {
        if (weekday.getFrom() < restTime.getFrom()) {
          workingRanges.add(new TimeRange(weekday.getFrom(), restTime.getFrom()));
        }
        if (restTime.getTo() < weekday.getTo()) {
          workingRanges.add(new TimeRange(restTime.getTo(), weekday.getTo()));
        }
      } else {
        workingRanges.add(weekday);
      }
    } else {
      workingRanges.add(weekday);
    }

    for (TimeRange timeRange : workingRanges) {
      if (nowMinutes < timeRange.getFrom()) {
        return timeRange.getFrom() - nowMinutes;
      }
      if (nowMinutes <= timeRange.getTo()) {
        return 0;
      }
    }

    return -1;
  }
}
