package com.zoyi.channel.plugin.android;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.FrameLayout;

import com.zoyi.channel.plugin.android.activity.userchat_list.UserChatListActivity;
import com.zoyi.channel.plugin.android.enumerate.StoreType;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.event.ChannelViewBus;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.event.StoreBus;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.PushBotItem;
import com.zoyi.channel.plugin.android.model.entity.Guest;
import com.zoyi.channel.plugin.android.model.etc.InAppPushItem;
import com.zoyi.channel.plugin.android.push_bot.OnPushBotClickListener;
import com.zoyi.channel.plugin.android.selector.ChannelIOSelector;
import com.zoyi.channel.plugin.android.selector.CounterSelector;
import com.zoyi.channel.plugin.android.selector.PluginSelector;
import com.zoyi.channel.plugin.android.selector.UiStateSelector;
import com.zoyi.channel.plugin.android.store.PushBotItemStore;
import com.zoyi.channel.plugin.android.store.Store;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.Executor;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.util.IntentUtils;
import com.zoyi.channel.plugin.android.util.UIUtils;
import com.zoyi.channel.plugin.android.view.popup.InAppPushView;
import com.zoyi.channel.plugin.android.view.popup.PushBotPopupView;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.functions.Action1;

/**
 * Control launcher, popup
 */
class ChannelView extends FrameLayout implements OnPushBotClickListener {

  private Context context;

  private ChannelLauncherView channelLauncherView;
  private PushBotPopupView pushBotPopupView;
  private InAppPushView inAppPushView;

  @Nullable
  private Subscription subscription;

  public ChannelView(Context context) {
    super(context);
    init(context);
  }

  public ChannelView(Context context, AttributeSet attrs) {
    super(context, attrs);
    init(context);
  }

  public ChannelView(Context context, AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context);
  }

  @Initializer
  private void init(final Context context) {
    this.context = context;

    View view = LayoutInflater.from(context).inflate(R.layout.ch_plugin_layout_channel_view, this, true);

    channelLauncherView = view.findViewById(R.id.ch_view_launcher);
    pushBotPopupView = view.findViewById(R.id.ch_view_push_bot_popup);
    inAppPushView = view.findViewById(R.id.ch_view_in_app_push);

    pushBotPopupView.setPushBotClickListener(this);
  }

  @Override
  protected void onAttachedToWindow() {
    super.onAttachedToWindow();

    if (ChannelIO.isBooted()) {
      channelLauncherView.initialize();
    }

    subscription = RxBus.observable()
        .subscribe(new Action1<Object>() {
          @Override
          public void call(Object o) {

            if (o instanceof CommandBus) {
              CommandBus commandBus = (CommandBus) o;

              switch (commandBus.getCommand()) {
                case BOOT:
                  setVisibility(VISIBLE);

                  channelLauncherView.initialize();
                  break;

                case SHUTDOWN:
                  setVisibility(GONE);

                  inAppPushView.hide();
                  pushBotPopupView.hide();
                  channelLauncherView.hide();
                  break;
              }
            }

            if (o instanceof ChannelViewBus) {
              ChannelViewBus channelViewBus = (ChannelViewBus) o;

              switch (channelViewBus.getChannelViewEvent()) {
                case UPDATE_LAUNCHER_VISIBILITY:
                  if (UiStateSelector.canShowLauncher()) {
                    if (!channelLauncherView.isVisible() && CompareUtils.isSameActivity(context, ChannelIO.getTopActivity())) {
                      trackPageViewEvent();
                    }
                    channelLauncherView.show();
                  } else {
                    channelLauncherView.hide();
                  }
                  break;

                case SHOW_PUSH_BOT:
                  if (CompareUtils.isSameActivity(context, ChannelIO.getTopActivity()) &&
                      channelViewBus.getData() != null &&
                      channelViewBus.getData() instanceof PushBotItem) {

                    PushBotItem pushBotItem = (PushBotItem) channelViewBus.getData();
                    inAppPushView.hide();
                    pushBotPopupView.show(pushBotItem);
                  }
                  break;

                case SHOW_IN_APP_PUSH:
                  if (UiStateSelector.canShowInAppPush(context) &&
                      channelViewBus.getData() != null &&
                      channelViewBus.getData() instanceof InAppPushItem) {
                    pushBotPopupView.hide();
                    inAppPushView.show((InAppPushItem) channelViewBus.getData());
                  }
                  break;
              }
            }

            if (o instanceof StoreBus) {
              StoreBus storeBus = (StoreBus) o;

              if (CompareUtils.exists(storeBus.getStoreType(), StoreType.USER, StoreType.VEIL) &&
                  storeBus.getEntity() != null &&
                  storeBus.getEntity() instanceof Guest) {
                Guest guest = (Guest) storeBus.getEntity();
                channelLauncherView.updateLauncherBadgeCount(guest.getAlert());
              }

              if (storeBus.getStoreType() == StoreType.PLUGIN) {
                channelLauncherView.setLauncherStyle();
              }

              if (storeBus.getStoreType() == StoreType.PUSH_BOT_ITEM) {
                channelLauncherView.updateLauncherBadgeCount(CounterSelector.getTotalCounter());
              }
            }
          }
        });
  }

  /**
   * PageView event 를 tracking 하는 조건
   * <p>
   * 1. Launcher 의 visibility 가 invisible 인 상태에서 visible 로 전환될 때
   * 2. 1번의 조건을 만족하고, 현재 Launcher 가 attach 되어있는 activity 가 topActivity 인 경우
   */
  private void trackPageViewEvent() {
    String pluginId = PluginSelector.getPluginId();

    if (pluginId != null && ChannelIOSelector.isEnabledTrackDefaultEvent()) {
      ChannelIOManager.trackPageView(context, pluginId);
    }
  }

  @Override
  protected void onDetachedFromWindow() {
    super.onDetachedFromWindow();
    if (subscription != null && !subscription.isUnsubscribed()) {
      subscription.unsubscribe();
    }
  }

  @Override
  public void onPushBotClick(@NonNull String pushBotId) {
    startPushBotChat(pushBotId);
  }

  @Override
  public void onPushBotContentsClick(@NonNull String pushBotId, @Nullable String redirectUrl) {
    if (redirectUrl != null) {
      Executor.executeRedirectUrlAction(context, redirectUrl);
    } else {
      startPushBotChat(pushBotId);
    }
  }

  private void startPushBotChat(@NonNull String pushBotId) {
    IntentUtils.setNextActivity(context, UserChatListActivity.class)
        .setTransition(Transition.SLIDE_FROM_BOTTOM)
        .putExtra(Const.EXTRA_PUSH_BOT_ID, pushBotId)
        .startActivity();
  }
}
