package com.zoyi.channel.plugin.android.activity.chat.view;

import android.content.Context;
import android.support.annotation.NonNull;
import android.text.InputType;
import android.text.TextUtils;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.inputmethod.EditorInfo;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnProfileEventListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnChangeInputStateListener;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.UIUtils;
import com.zoyi.channel.plugin.android.util.io.Keyboard;
import com.zoyi.channel.plugin.android.view.handler.EditTextChangedListener;
import com.zoyi.channel.plugin.android.view.layout.CHTextView;
import com.zoyi.channel.plugin.android.view.layout.WatchedEditText;

import java.math.BigDecimal;

import static android.text.InputType.TYPE_CLASS_NUMBER;
import static android.text.InputType.TYPE_NUMBER_FLAG_DECIMAL;
import static android.text.InputType.TYPE_NUMBER_FLAG_SIGNED;

/**
 * Created by mika on 2018. 4. 17..
 */

public class ProfileMessageView extends LinearLayout {

  private Context context;
  protected OnProfileEventListener onProfileEventListener;
  protected OnChangeInputStateListener onChangeInputStateListener;

  private CHTextView textName;
  private CHTextView textError;
  private CHTextView textPage;
  private CHTextView textTotalPage;

  private LinearLayout inputLayout;
  private FrameLayout inputItems;
  private WatchedEditText editText;
  private ImageButton sendButton;
  private View loadingView;

  private View completeLayout;
  private CHTextView textComplete;

  protected String key;
  protected String type;
  protected boolean hasError;
  protected boolean isLoading;
  protected boolean hasEditTextFocus = false;

  public ProfileMessageView(
      Context context,
      OnProfileEventListener onProfileEventListener,
      OnChangeInputStateListener onChangeInputStateListener) {
    super(context);
    init(context, onProfileEventListener, onChangeInputStateListener);
  }

  private void init(Context context,
                    final OnProfileEventListener onProfileEventListener,
                    final OnChangeInputStateListener onChangeInputStateListener) {
    this.context = context;
    View view = LayoutInflater.from(context).inflate(R.layout.ch_plugin_layout_profile_item, this);

    textName = view.findViewById(R.id.ch_text_profile_item_name);
    textError = view.findViewById(R.id.ch_text_profile_item_error);
    textPage = view.findViewById(R.id.ch_text_profile_item_page);
    textTotalPage = view.findViewById(R.id.ch_text_profile_item_total_page);

    inputLayout = (LinearLayout) view.findViewById(R.id.ch_layout_profile_item_input);
    inputItems = (FrameLayout) view.findViewById(R.id.ch_layout_profile_item_input_items);
    editText = (WatchedEditText) view.findViewById(R.id.ch_edit_profile_item_input);
    sendButton = (ImageButton) view.findViewById(R.id.ch_button_profile_item_input);
    loadingView = view.findViewById(R.id.ch_view_profile_item_input_loading);

    completeLayout = view.findViewById(R.id.ch_layout_profile_item_complete);
    textComplete = view.findViewById(R.id.ch_text_profile_item_complete);

    this.onProfileEventListener = onProfileEventListener;
    this.onChangeInputStateListener = onChangeInputStateListener;

    initInputView(context);
    initInputViewEventListener();

    sendButton.setOnClickListener(new OnClickListener() {
      @Override
      public void onClick(View v) {
        if (onProfileEventListener != null) {
          onProfileEventListener.onSendButtonClick(key, getValue(), getInputState());
        }
      }
    });
  }

  protected void initInputView(Context context) {
  }

  protected void initInputViewEventListener() {
    editText.setWatchedTextChangedListener(new EditTextChangedListener() {
      @Override
      public void onWatchedTextChanged(String text) {
        if (CompareUtils.isSame(" ", text)) {
          editText.setText("");
        } else {
          if (hasEditTextFocus) {
            onChangeInputStateListener.onUpdateInputState(key, text, !text.isEmpty());
          }
          setStyle(false, editText.isFocused(), isValidInputState(text), isLoading);
        }
      }
    });
    editText.setOnFocusChangeListener(new OnFocusChangeListener() {
      @Override
      public void onFocusChange(View v, boolean hasFocus) {
        hasEditTextFocus = hasFocus;
        setStyle(hasError, hasFocus, isValidInputState(editText.getText().toString()), isLoading);
        onProfileEventListener.onChangeFocus(hasFocus);
      }
    });
    editText.setOnEditorActionListener(new TextView.OnEditorActionListener() {
      @Override
      public boolean onEditorAction(TextView v, int actionId, KeyEvent event) {
        if (actionId == EditorInfo.IME_ACTION_DONE || actionId == EditorInfo.IME_ACTION_NEXT) {
          if (isValidInputState(getValue())) {
            onProfileEventListener.onSendButtonClick(key, getValue(), getInputState());
          }
          return true;
        }
        return false;
      }
    });

  }

  public void setKey(String key) {
    this.key = key;
  }

  public void setType(String type) {
    if (type != null) {
      this.type = type;

      switch (type) {
        case "Number":
          editText.setInputType(TYPE_CLASS_NUMBER | TYPE_NUMBER_FLAG_DECIMAL);
          break;

        default:
          editText.setInputType(InputType.TYPE_CLASS_TEXT);
          break;
      }
    }
  }

  private void initLayout() {
    textName.setVisibility(GONE);
    textError.setVisibility(GONE);
    textPage.setVisibility(GONE);
    textTotalPage.setVisibility(GONE);
    inputLayout.setVisibility(GONE);
    completeLayout.setVisibility(GONE);
    textComplete.setVisibility(GONE);
  }

  public void setState(
      String name,
      String error,
      int page,
      int totalPage,
      Object value,
      Object inputState,
      boolean isLoading,
      boolean hasCurrentViewFocus,
      boolean isLastProfile) {
    initLayout();

    if (isValueExists(value)) {
      textName.setText(name);

      completeLayout.setVisibility(VISIBLE);
      textComplete.setVisibility(VISIBLE);
      textComplete.setText(getFormattedString(value));

      setCompleteStyle();
    } else {
      textName.setText(name);
      textError.setText(error);

      textPage.setVisibility(VISIBLE);
      textPage.setText(String.valueOf(page));
      textTotalPage.setVisibility(VISIBLE);
      textTotalPage.setText(String.format("/%d", totalPage));

      inputLayout.setVisibility(VISIBLE);

      setInputState(inputState, isLoading, isLastProfile);
      setStyle(!TextUtils.isEmpty(error), isInputViewFocused(), isValidInputState(inputState), isLoading);

      if (hasCurrentViewFocus) {
        requestFocusInputView();
      }
    }
  }

  private void setCompleteStyle() {
    this.hasError = false;
    this.isLoading = false;

    textError.setVisibility(GONE);
    textName.setVisibility(VISIBLE);
  }

  protected void setStyle(boolean hasError, boolean hasFocus, boolean inputStateValid, boolean isLoading) {
    this.hasError = hasError;
    this.isLoading = isLoading;

    if (hasError) {
      textError.setVisibility(VISIBLE);
      textName.setVisibility(GONE);

      inputLayout.setBackgroundResource(R.drawable.ch_plugin_profile_message_input_wrong);
      sendButton.setImageResource(R.drawable.ch_send_error);
    } else if (hasFocus) {
      textError.setVisibility(GONE);
      textName.setVisibility(VISIBLE);

      inputLayout.setBackgroundResource(R.drawable.ch_plugin_profile_message_input);
      sendButton.setImageResource(R.drawable.ch_send);
    } else {
      textError.setVisibility(GONE);
      textName.setVisibility(VISIBLE);

      inputLayout.setBackgroundResource(R.drawable.ch_plugin_profile_message_input_out_focus);
      sendButton.setImageResource(R.drawable.ch_send);
    }

    loadingView.setVisibility(UIUtils.getVisible(isLoading, true));
    sendButton.setVisibility(UIUtils.getVisible(inputStateValid && !isLoading, true));
  }

  protected void overrideInputView(View view) {
    editText.setVisibility(GONE);
    inputItems.addView(view);
  }

  protected boolean isInputViewFocused() {
    return editText.isFocused();
  }

  protected void requestFocusInputView() {
    editText.requestFocus();
    Keyboard.open(context, editText);
  }

  protected Object getInputState() {
    return editText.getText().toString();
  }

  protected void setInputState(Object inputState, boolean isLoading, boolean isLastProfile) {
    editText.setText(inputState == null ? "" : getFormattedString(inputState));
    editText.setSelection(inputState == null ? 0 : getFormattedString(inputState).length());
    editText.setEnabled(!isLoading);
    editText.setImeOptions(isLastProfile ? EditorInfo.IME_ACTION_DONE : EditorInfo.IME_ACTION_NEXT);
  }

  protected Object getValue() {
    String inputValue = editText.getText().toString();
    if (CompareUtils.isSame(Const.PROFILE_TYPE_NUMBER, getType())) {
      try {
        return Double.valueOf(inputValue);
      } catch (NumberFormatException exception) {
        L.e(exception.getMessage());
      }
    }
    return inputValue;
  }

  protected String getType() {
    return type;
  }

  protected boolean isValidInputState(Object value) {
    return value != null && !TextUtils.isEmpty(value.toString());
  }

  protected String getFormattedString(@NonNull Object object) {
    if (object instanceof Double) {
      BigDecimal bigDecimal = new BigDecimal(object.toString());
      String numberValue = bigDecimal.toString();

      if (numberValue.endsWith(".0")) {
        return numberValue.replace(".0", "");
      } else {
        return numberValue;
      }
    } else {
      return object.toString();
    }
  }

  private boolean isValueExists(Object value) {
    return value != null && !TextUtils.isEmpty(value.toString());
  }
}
