package com.zoyi.channel.plugin.android.presenter.chat;

import android.content.Context;
import android.text.TextUtils;

import com.zoyi.channel.plugin.android.ChannelIO;
import com.zoyi.channel.plugin.android.ChannelStore;
import com.zoyi.channel.plugin.android.activity.chat.ChatDataDictionary;
import com.zoyi.channel.plugin.android.activity.chat.ChatManager;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageClickListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageReviewListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageSendListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnSendingMessageClickListener;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnUserInfoListener;
import com.zoyi.channel.plugin.android.activity.chat.model.ActionMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.MobileNumberItem;
import com.zoyi.channel.plugin.android.activity.chat.model.NewMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.ResolveMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.SendingMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.model.TypingItem;
import com.zoyi.channel.plugin.android.activity.chat.model.UserInfoItem;
import com.zoyi.channel.plugin.android.activity.chat.type.UserInfoType;
import com.zoyi.channel.plugin.android.activity.userchat_list.UserChatListDataDictionary;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.enumerate.chat.ChatState;
import com.zoyi.channel.plugin.android.event.ChannelModelBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.lib.CountryCodeDialog;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.Bot;
import com.zoyi.channel.plugin.android.model.interfaces.ChannelModel;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.Country;
import com.zoyi.channel.plugin.android.model.rest.File;
import com.zoyi.channel.plugin.android.model.rest.Manager;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.model.interfaces.ProfileEntity;
import com.zoyi.channel.plugin.android.model.rest.Session;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.model.rest.Veil;
import com.zoyi.channel.plugin.android.model.rest.WebPage;
import com.zoyi.channel.plugin.android.model.wrapper.ChatPreloadWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.CountryWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.ManagersWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.MessagesWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.ScriptsWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.UserVeilWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.RequestUtils;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.channel.plugin.android.util.TimeUtils;
import com.zoyi.rx.Observable;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.functions.Action1;
import com.zoyi.rx.functions.Func3;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * Created by mika on 2017. 2. 27..
 */
public class ChatPresenter implements
    ChatContract.Presenter,
    OnSendingMessageClickListener,
    OnMessageClickListener,
    OnUserInfoListener,
    OnMessageSendListener,
    OnMessageReviewListener {

  private ChatContract.View view;
  private ChatAdapterContract.View adapterView;
  private ChatAdapterContract.Model adapterModel;

  private Context context;

  private UserChat userChat;
  private Session session;

  private ChatDataDictionary dictionary;

  private String chatId;

  private ChatState state = ChatState.IDLE;

  private String backwardId = Const.MESSAGE_ID_MAX;
  private String forwardId = Const.MESSAGE_ID_MIN;
  private String forwardTempId = Const.MESSAGE_ID_MIN;
  private Long forwardTimestamp = Long.MIN_VALUE;

  private int fetchCounter = 0;
  private boolean backwardLoading = false;
  private boolean firstMessage = true;
  private boolean isStateCompleted = false;

  private Long lastReadAt;
  private long welcomedAt;

  private List<Country> countries;

  private List<SendingMessageItem> tempQueue;

  private final Object messageObject = new Object();

  public ChatPresenter(Context context) {
    this.context = context;

    TimeUtils.refreshOffset();

    dictionary = ChatDataDictionary.getInstance();
    tempQueue = new ArrayList<>();
    countries = new ArrayList<>();
  }

  @Override
  public void setView(ChatContract.View view) {
    this.view = view;
  }

  @Override
  public void release() {
    if (dictionary != null) {
      dictionary.release();
    }
    dictionary = null;
  }

  @Override
  public UserChat getUserChat() {
    return userChat;
  }

  @Override
  public String getChatId() {
    return chatId;
  }

  @Override
  public boolean isStateCompleted() {
    return isStateCompleted;
  }

  @Override
  public void setChatId(String chatId) {
    this.chatId = chatId;
  }

  @Override
  public void setAdapterView(ChatAdapterContract.View adapterView) {
    this.adapterView = adapterView;
    this.adapterView.setOnMessageClickListener(this);
    this.adapterView.setOnSendingMessageClickListener(this);
    this.adapterView.setOnUserInfoListener(this);
    this.adapterView.setOnMessageReviewListener(this);
  }

  @Override
  public void setAdapterModel(ChatAdapterContract.Model adapterModel) {
    this.adapterModel = adapterModel;
  }

  @Override
  public void requestPreloadData(final String chatId) {
    Plugin plugin = ChannelStore.getPlugin();
    if (plugin == null) {
      return;
    }
    view.showProgress(ResUtils.getString(context, "ch.loading_information"));

    Observable.zip(
        ChannelIO.getApi().getFollowingManagers(),
        ChannelIO.getApi().getPlugin(plugin.getId()),
        ChannelIO.getApi().getScripts(plugin.getId()),
        new Func3<ManagersWrapper, PluginWrapper, ScriptsWrapper, ChatPreloadWrapper>() {
          @Override
          public ChatPreloadWrapper call(ManagersWrapper managersWrapper, PluginWrapper pluginWrapper, ScriptsWrapper scriptsWrapper) {
            return new ChatPreloadWrapper(managersWrapper, pluginWrapper, scriptsWrapper);
          }
        })
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<ChatPreloadWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            L.e(error.getMessage());
            view.hideProgress();
          }

          @Override
          public void onNext(ChatPreloadWrapper chatPreloadWrapper) {
            view.hideProgress();

            if (chatPreloadWrapper != null
                && handleFollowingManagers(chatPreloadWrapper.getManagersWrapper())
                && handlePlugin(chatPreloadWrapper.getPluginWrapper())
                && handleScripts(chatPreloadWrapper.getScriptsWrapper())) {
              fetchUserChat(chatId);
            }
          }
        });
  }

  private boolean handleFollowingManagers(ManagersWrapper managersWrapper) {
    if (managersWrapper == null) {
      return false;
    }
    dictionary.setFollowingManagers(managersWrapper.getManagers());
    return true;
  }

  private boolean handlePlugin(PluginWrapper pluginWrapper) {
    if (pluginWrapper == null || pluginWrapper.getBot() == null) {
      return false;
    }

    ChannelStore.setBot(pluginWrapper.getBot());
    UserChatListDataDictionary.getInstance().add(pluginWrapper.getBot());
    return true;
  }

  private boolean handleScripts(ScriptsWrapper scriptsWrapper) {
    if (scriptsWrapper == null) {
      return false;
    }
    dictionary.setScripts(scriptsWrapper.getScripts());
    return true;
  }

  private void fetchUserChat(String chatId) {
    if (TextUtils.isEmpty(chatId)) {
      this.chatId = null;
      addWelcomeMessage();

      view.onLoadUserChat(null, false);
      view.updateBackwardId(null);
    } else {
      this.chatId = chatId;
      adapterModel.addMessageItems(getChatManager().getFailedItems(chatId));

      fetchUserChat();
    }
  }

  private void addWelcomeMessage() {
    if (adapterModel.getWelcomeMessage() == null) {
      String message;
      if (ChannelStore.hasName()) {
        message = dictionary.getScript(context, "welcome", "ch.scripts.welcome.default");
        if (message != null) {
          message = message.replace("${name}", ChannelStore.getName());
        }
      } else {
        message = dictionary.getScript(context, "welcome_ghost", "ch.scripts.welcome_ghost.default");
      }

      if (!TextUtils.isEmpty(message)) {
        welcomedAt = System.currentTimeMillis();
        adapterModel.setWelcomeMessage(new ActionMessageItem(null, message, welcomedAt, 1));
      }
    }
  }

  @Override
  public void fetchUserChat() {
    if (userChat == null) {
      view.showProgress(ResUtils.getString(context, "ch.loading_information"));
    }
    setState(ChatState.USER_CHAT_LOADING, false);
    ChannelIO.getApi().getUserChat(chatId)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<UserChatWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            view.hideProgress();
            handleUserChatFetchError(error);
          }

          @Override
          public void onNext(UserChatWrapper userChatWrapper) {
            setUserChat(userChatWrapper, false);
          }
        });
  }

  private void createUserChat() {
    view.showProgress(ResUtils.getString(context, "ch.loading_information"));

    setState(ChatState.USER_CHAT_LOADING, false);

    Plugin plugin = ChannelStore.getPlugin();
    if (ChannelStore.getPluginKey() != null && plugin != null) {
      ChannelIO.getApi().createUserChat(ChannelStore.getPluginKey(), plugin.getId(), welcomedAt)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<UserChatWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              view.hideProgress();
              handleUserChatFetchError(error);
            }

            @Override
            public void onNext(UserChatWrapper userChatWrapper) {
              RxBus.post(new ChannelModelBus(userChatWrapper.getUserChat(), true));
              RxBus.post(new ChannelModelBus(userChatWrapper.getSession(), true));

              setUserChat(userChatWrapper, true);
            }
          });
    } else {
      view.hideProgress();
    }
  }

  private void setUserChat(UserChatWrapper wrapper, boolean create) {
    view.hideProgress();

    if (isStateEquals(ChatState.USER_CHAT_LOADING)) {
      if (wrapper == null) {
        handleUserChatFetchError(new Exception("UserChatWrapper cannot be null"));
      } else {
        userChat = wrapper.getUserChat();
        chatId = userChat.getId();
        isStateCompleted = userChat.isStateCompleted();
        session = wrapper.getSession();
        dictionary.add(wrapper.getManagers());

        adapterModel.setUserChat(userChat);

        view.onLoadUserChat(userChat, create);
        view.setInputLayoutVisibility();
        view.setReconnectVisibility(false);

        if (lastReadAt == null) {
          lastReadAt = create ? Long.MAX_VALUE : session.getLastReadAt();
        }

        if (tempQueue.size() > 0) {
          getChatManager().send(chatId, tempQueue, this);
          tempQueue.clear();
        }

        joinChat();
      }
    }
  }

  private void handleUserChatFetchError(Throwable throwable) {
    if (throwable != null) {
      L.e(throwable.getMessage());
    }
    if (isStateEquals(ChatState.USER_CHAT_LOADING)) {
      setState(ChatState.USER_CHAT_NOT_LOADED, true);
    }
  }

  private void joinChat() {
    if (SocketManager.isReady()) {
      setState(ChatState.CHAT_JOINING, false);
      SocketManager.joinChat(chatId);
    } else {
      setState(ChatState.WAITING_SOCKET, false);
      SocketManager.reconnect();
    }
  }

  private boolean isAvailableRequest(int count) {
    return isStateEquals(ChatState.MESSAGES_LOADING) && fetchCounter == count;
  }

  @Override
  public void fetchMessages() {
    if (SocketManager.isReady()) {
      fetchCounter++;
      setState(ChatState.MESSAGES_LOADING, false);
      if (!isMessageExists()) {
        fetchInitMessages(fetchCounter);
      } else {
        fetchForwardMessages(forwardId, fetchCounter);
      }
    } else {
      setState(ChatState.WAITING_SOCKET, false);
      SocketManager.reconnect();
    }
  }

  @Override
  public boolean isMessageExists() {
    return CompareUtils.compare(backwardId, forwardId) < 0;
  }

  @Override
  public void fetchInitMessages(final int counter) {
    ChannelIO.getApi()
        .getMessages(
            chatId,
            Const.MESSAGE_ID_MAX,
            Const.MESSAGE_FETCH_LIMIT,
            Const.DESC)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<MessagesWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            if (isAvailableRequest(counter)) {
              handleMessagesFetchError(error, counter);
            }
          }

          @Override
          public void onNext(MessagesWrapper wrapper) {
            if (isAvailableRequest(counter)) {
              if (wrapper == null) {
                handleMessagesFetchError(new Exception("MessagesWrapper cannot be null"), counter);
              } else {
                backwardId = wrapper.getNext();
                view.updateBackwardId(backwardId);

                dictionary.add(wrapper.getManagers());
                dictionary.add(wrapper.getBots());

                addMessages(wrapper.getMessages());
                calculateForwardId(wrapper.getMessages());
                calculateForwardId();

                if (lastReadAt >= forwardTimestamp) {
                  lastReadAt = Long.MAX_VALUE;
                }

                addNewMessageItem(wrapper.getMessages());

                setState(ChatState.CHAT_READY, false);
                readAll();

                view.scrollToBottom(true);
              }
            }
          }
        });
  }

  @Override
  public void fetchBackwardMessages() {
    if (backwardLoading || !isMessageExists()) {
      return;
    }
    backwardLoading = true;

    ChannelIO.getApi()
        .getMessages(
            chatId,
            backwardId,
            Const.MESSAGE_FETCH_LIMIT,
            Const.DESC)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<MessagesWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            backwardLoading = false;

            handleMessagesFetchError(error, fetchCounter);
          }

          @Override
          public void onNext(MessagesWrapper wrapper) {
            if (wrapper == null) {
              handleMessagesFetchError(new Exception("MessagesWrapper cannot be null"), fetchCounter);
            } else {
              backwardId = wrapper.getNext();
              view.updateBackwardId(backwardId);

              dictionary.add(wrapper.getManagers());
              dictionary.add(wrapper.getBots());

              adapterModel.addMessages(wrapper.getMessages(), chatId);

              addNewMessageItem(wrapper.getMessages());
            }
            backwardLoading = false;
          }
        });
  }

  @Override
  public void fetchForwardMessages(String since, final int counter) {
    ChannelIO.getApi()
        .getMessages(
            chatId,
            since,
            Const.MESSAGE_FETCH_LIMIT,
            Const.ASC)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<MessagesWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            if (isAvailableRequest(counter)) {
              handleMessagesFetchError(error, counter);
            }
          }

          @Override
          public void onNext(MessagesWrapper wrapper) {
            if (isAvailableRequest(counter)) {
              if (wrapper == null) {
                handleMessagesFetchError(new Exception("MessagesWrapper cannot be null"), counter);
              } else {
                dictionary.add(wrapper.getManagers());
                dictionary.add(wrapper.getBots());

                addMessages(wrapper.getMessages());
                calculateForwardId(wrapper.getMessages());

                if (wrapper.getNext() == null) {
                  calculateForwardId();
                  setState(ChatState.CHAT_READY, false);
                  readAll();
                } else {
                  fetchForwardMessages(forwardId, counter);
                }
              }
            }
          }
        });
  }

  private void handleMessagesFetchError(Throwable throwable, int counter) {
    if (throwable != null) {
      L.e(throwable.getMessage());
    }
    if (isStateEquals(ChatState.MESSAGES_LOADING) && fetchCounter == counter) {
      setState(ChatState.MESSAGES_NOT_LOADED, true);
    }
  }

  private void addNewMessageItem(List<Message> messages) {
    if (lastReadAt == null || isStateCompleted) {
      return;
    }

    long min = Long.MAX_VALUE;
    long max = Long.MIN_VALUE;

    if (messages != null) {
      for (Message message : messages) {
        if (message.getLog() != null) {
          continue;
        }
        min = Math.min(min, message.getCreatedAt());
        max = Math.max(max, message.getCreatedAt());
      }
    }

    if (min <= lastReadAt && lastReadAt < max) {
      adapterModel.addMessageItem(new NewMessageItem(lastReadAt));
    }
  }

  private void addMessage(Message message) {
    if (CompareUtils.isSame(message.getChatId(), chatId)) {
      adapterModel.addMessage(message, chatId);

      boolean isMine = CompareUtils.exists(message.getPersonType(), User.CLASSNAME, Veil.CLASSNAME);

      if (isStateEquals(ChatState.CHAT_READY)) {
        calculateForwardId(message);
        readAll();
      } else {
        forwardTempId = CompareUtils.max(forwardTempId, message.getId());
      }

      forwardTimestamp = Math.max(forwardTimestamp, message.getCreatedAt());

      if (!isStateCompleted && firstMessage && isMine) {
        firstMessage = false;

        if (ChannelStore.isRequestGuestInfo()) {
          checkRequireUserInfo(forwardTimestamp, false);
        }
      }

      view.scrollToBottom(isMine);
    }
  }

  private void addMessages(List<Message> messages) {
    if (messages != null) {
      for (Message message : messages) {
        forwardTimestamp = Math.max(forwardTimestamp, message.getCreatedAt());
        forwardId = CompareUtils.max(forwardId, message.getId());
      }
    }
    adapterModel.addMessages(messages, chatId);
  }

  private void calculateForwardId(Message message) {
    if (message != null) {
      forwardId = CompareUtils.max(forwardId, message.getId());
    }
  }

  private void calculateForwardId(List<Message> messages) {
    if (messages != null) {
      for (Message message : messages) {
        calculateForwardId(message);
      }
    }
  }

  private void calculateForwardId() {
    forwardId = CompareUtils.max(forwardId, forwardTempId);
  }

  @Override
  public void sendTextMessage(String message) {
    if (TextUtils.isEmpty(message)) {
      return;
    }
    SendingMessageItem item = new SendingMessageItem(chatId, message, false);
    adapterModel.addMessageItem(item);

    sendMessages(Collections.singletonList(item));
  }

  @Override
  public void sendImageFiles(ArrayList<String> imagePaths) {
    List<SendingMessageItem> items = new ArrayList<>();
    if (imagePaths != null) {
      for (String p : imagePaths) {
        SendingMessageItem item = new SendingMessageItem(chatId, p, true);
        adapterModel.addMessageItem(item);
        items.add(item);
      }
    }
    sendMessages(items);
  }

  private void sendMessages(List<SendingMessageItem> items) {
    view.scrollToBottom(true);

    if (chatId == null) {
      tempQueue.addAll(items);
      createUserChat();
    } else {
      getChatManager().send(items, this);
    }
  }

  @Override
  public void receiveData(ChannelModel channelModel, boolean upsert) {
    if (channelModel != null) {
      if (!TextUtils.isEmpty(chatId)) {
        synchronized (messageObject) {
          switch (channelModel.getClass().getSimpleName()) {
            case Message.CLASSNAME:
              Message message = (Message) channelModel;

              if (message.getBotOption() != null &&
                  message.getBotOption().isWelcome() &&
                  adapterModel.getWelcomeMessage() != null) {
                return;
              }

              handleMessageLog(message);
              addMessage(message);

              view.processNewMessage(
                dictionary.getProfile(message.getPersonType(), message.getPersonId()),
                message);
              break;

            case File.CLASSNAME:
            case WebPage.CLASSNAME:
              dictionary.add(channelModel);
              break;

            case Bot.CLASSNAME:
            case Manager.CLASSNAME:
              dictionary.add(channelModel);

              if (userChat != null && !(userChat.isStateReady() || userChat.isStateOpen())) {
                view.setProfileEntity((ProfileEntity) channelModel);
              }
              break;

            case UserChat.CLASSNAME:
              handleUserChatData((UserChat) channelModel);
              view.changeBigBar();
              break;
          }
        }
      }

      if (Channel.CLASSNAME.equals(channelModel.getClass().getSimpleName())) {
        // Exception for Channel model
        view.setChannel((Channel) channelModel);
      }
    }
  }

  private void handleUserChatData(UserChat userChat) {
    if (userChat == null || !chatId.equals(userChat.getId())) {
      return;
    }

    this.userChat = userChat;

    if (userChat.isStateRemoved()) {
      leaveChat();
      view.onRemovedChat();
    } else if (userChat.isStateCompleted()) {
      adapterModel.addMessageItem(userChat);
      view.scrollToBottom(false);
    } else if (userChat.isStateFollowing()) {
      view.setProfileEntity(dictionary.getProfile(userChat.getHostType(), userChat.getHostId()));
    }
  }

  private void handleMessageLog(Message message) {
    if (CompareUtils.isSame(message.getChatId(), chatId)) {
      if (message.getLog() == null) {
        return;
      }
      if (message.getLog().isResolve()) {
        isStateCompleted = true;
        userChat.setState(Const.USER_CHAT_STATE_RESOLVED);
      } else if (message.getLog().isClose()) {
        isStateCompleted = true;
        userChat.setState(Const.USER_CHAT_STATE_CLOSED);
      } else {
        isStateCompleted = false;
      }
      adapterModel.setUserChat(userChat);

      view.setInputLayoutVisibility();
    }
  }

  private void readAll() {
    ChannelIO.getApi().readAll(chatId)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<Void>());
  }

  @Override
  public void receiveCommand(Command command, Object object) {
    switch (command) {
      case APP_STARTED:
        refresh();
        break;

      case SOCKET_DISCONNECTED:
        fetchCounter++;
        if (!isStateEquals(ChatState.IDLE)) {
          setState(ChatState.USER_CHAT_NOT_LOADED, true);
        }
        break;

      case SOCKET_ERROR:
        view.setReconnectVisibility(true);
        break;

      case READY:
        view.setReconnectVisibility(false);

        if (isStateEquals(ChatState.WAITING_SOCKET)) {
          joinChat();
        } else {
          refresh();
        }
        break;

      case LEAVED:
        if (object != null
            && CompareUtils.compare(chatId, (String) object) == 0
            && isStateEquals(ChatState.CHAT_READY)) {
          setState(ChatState.NOT_JOINED, false);  // true?
        }
        break;

      case JOINED:
        if (object != null
            && CompareUtils.compare(chatId, (String) object) == 0
            && isStateEquals(ChatState.CHAT_JOINING)) {

          view.setReconnectVisibility(false);

          fetchMessages();
        }
        break;
    }
  }

  @Override
  public void receiveTyping(Typing typing) {
    synchronized (messageObject) {
//      boolean isBottom = view.isBottomPosition();

      TypingItem item = new TypingItem();
      int position = adapterModel.getIndex(item);
      if (position >= 0) {
        item = (TypingItem) adapterModel.getItem(position);
      }
      if (item == null) {
        return;
      }

      switch (typing.getAction()) {
        case "start":
          item.addOrUpdateTyping(typing);
          break;

        case "stop":
          item.removeTyping(typing);
          break;
      }

      adapterModel.addOrUpdateMessageItem(position, item);
//      if (item.hasTyping()) {
//        adapterModel.addOrUpdateMessageItem(position, item);
//      } else {
//        adapterModel.removeMessageItem(item);
//      }
//
//      if (isBottom) {
//        view.scrollToBottom(true);
//      }
    }
  }

  @Override
  public void messageClicked(Message message) {
    if (message.getFile() != null) {
      if (message.getFile().getPreviewThumb() == null || !message.getFile().isImage()) {
        view.onFileDownload(message);
      } else {
        adapterModel.setImageFilesToStorage();
        view.onShowPhotoAlbum(message.getFile());
      }
    } else if (message.getWebPage() != null) {
      urlClicked(message.getWebPage().getUrl());
    }
  }

  @Override
  public void urlClicked(String url) {
    view.onUrlClicked(url);
  }

  @Override
  public void sendingMessageClicked(SendingMessageItem item) {
    view.sendingMessageClicked(item);
  }

  @Override
  public void resend(SendingMessageItem item) {
    if (item != null) {
      removeFailedItem(item);

      item.refreshCreatedAt();

      adapterModel.addMessageItem(item);
      getChatManager().send(item, this);
    }
  }

  @Override
  public void removeFailedItem(SendingMessageItem item) {
    if (item != null) {
      adapterModel.removeMessageItem(item);
      getChatManager().removeFailedItem(item.getCreatedAt());
    }
  }

  @Override
  public void sendSuccessed(SendingMessageItem item, Message message) {
    adapterModel.removeMessageItem(item);
    receiveData(message, true);
  }

  @Override
  public void sendFailed(SendingMessageItem item) {
    adapterModel.addMessageItem(item);
  }

  @Override
  public void refresh() {
    switch (state) {
      case USER_CHAT_NOT_LOADED:
        if (chatId != null) {
          fetchUserChat();
        } else {
          createUserChat();
        }
        break;

      case WAITING_SOCKET:
        SocketManager.connect();
        break;

      case NOT_JOINED:
        setState(ChatState.CHAT_JOINING, true);
        SocketManager.joinChat(chatId);
        break;

      case MESSAGES_NOT_LOADED:
        fetchMessages();
        break;
    }
  }

  private void setState(ChatState state, boolean showRefresh) {
    this.state = state;
    view.setRefreshVisibility(showRefresh);
  }

  private boolean isStateEquals(ChatState targetState) {
    return targetState.equals(state);
  }

  @Override
  public void userInfoChanged(Object object) {
    if (object instanceof String) {
      dictionary.setUserName((String) object);
    } else if (object instanceof MobileNumberItem) {
      dictionary.setUserMobilNumber((MobileNumberItem) object);
    }
  }

  @Override
  public void getCountries(final int position) {
    if (countries == null || countries.isEmpty()) {
      view.showProgress(ResUtils.getString(context, "ch.loading_information"));

      ChannelIO.getApi().getCountries()
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<CountryWrapper>() {
            @Override
            public void onError(RetrofitException e) {
              view.hideProgress();
            }

            @Override
            public void onNext(CountryWrapper countryWrapper) {
              view.hideProgress();

              if (countryWrapper != null) {
                countries = countryWrapper.getCountries();
                showCountryDialog(position);
              }
            }
          });
    } else {
      showCountryDialog(position);
    }
  }

  private void showCountryDialog(final int itemPosition) {
    if (dictionary == null) {
      return;
    }

    final MobileNumberItem mobileNumberItem = dictionary.getUserMobilNumber();
    if (mobileNumberItem != null) {
      new CountryCodeDialog(
          context,
          mobileNumberItem.getCheckedPosition(),
          countries,
          new CountryCodeDialog.OnCountryCodeSelectListener() {
            @Override
            public void onCountryCodeSelected(int position, String countryCode) {
              mobileNumberItem.setCountry(countryCode);
              mobileNumberItem.setCheckedPosition(position);

              dictionary.setUserMobilNumber(mobileNumberItem);
              adapterView.notifyItemChange(itemPosition);
            }
          }).show();
    }
  }

  @Override
  public void sendUserInfo(final UserInfoItem item) {
    RequestUtils requestUtils;
    switch (item.getUserInfoType()) {
      case NAME:
        requestUtils = RequestUtils.form().set("name", dictionary.getUserName());
        break;

      case MOBILE_NUMBER:
        MobileNumberItem mobileNumberItem = dictionary.getUserMobilNumber();
        requestUtils = RequestUtils.form()
            .set(
                "mobileNumber",
                String.format(
                    "+%s%s",
                    mobileNumberItem.getCountry(),
                    mobileNumberItem.getMobileNumber()));
        break;

      default:
        return;
    }

    if (requestUtils != null) {
      ChannelIO.getApi().updateGuest(requestUtils.create())
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<UserVeilWrapper>() {
            @Override
            public void onError(RetrofitException e) {
              L.e(e.getMessage());
              item.setError(e.getMessage());
              adapterModel.addMessageItem(item);
            }

            @Override
            public void onNext(UserVeilWrapper wrapper) {
              ChannelStore.setUserVeil(wrapper.getUser(), wrapper.getVeil());

              adapterModel.removeMessageItem(item);
              checkRequireUserInfo(System.currentTimeMillis(), true);
            }
          });
    }
  }

  private void checkRequireUserInfo(long timestamp, boolean isComplete) {
    final UserInfoItem userInfoItem;
    if (!ChannelStore.hasName()) {
      userInfoItem = new UserInfoItem(UserInfoType.NAME, timestamp, null);
    } else if (ChannelStore.getMobileNumber() == null) {
      userInfoItem = new UserInfoItem(UserInfoType.MOBILE_NUMBER, timestamp, null);
    } else {
      userInfoItem = null;
    }

    if (userInfoItem != null) {
      Observable.timer(512, TimeUnit.MILLISECONDS)
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new Action1<Long>() {
            @Override
            public void call(Long aLong) {
              adapterModel.addMessageItem(userInfoItem);

              view.scrollToBottom(true);
            }
          });
    } else {
      if (isComplete) {
        adapterModel.addMessageItem(new UserInfoItem(UserInfoType.COMPLETE, timestamp, null));
      }
    }
  }

  @Override
  public void onMessageReview(String review, final ResolveMessageItem resolveMessageItem) {
    ChannelIO.getApi().reviewUserChat(chatId, review)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<UserChatWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            L.e(error.getMessage());
          }

          @Override
          public void onNext(UserChatWrapper userChatWrapper) {
            if (userChatWrapper != null) {
              UserChat userChat = userChatWrapper.getUserChat();

              if (userChat != null) {
                adapterModel.addMessageItem(userChat);
              }
            }
          }
        });
  }

  @Override
  public void leaveChat() {
    SocketManager.leaveChat(chatId);
  }

  private ChatManager getChatManager() {
    return ChatManager.get();
  }
}
