package com.zoyi.channel.plugin.android.activity.chat;

import android.content.DialogInterface;
import android.content.Intent;
import android.os.Bundle;
import android.os.Handler;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.AlertDialog;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.view.View;
import android.widget.Button;
import android.widget.ImageButton;

import com.zoyi.channel.plugin.android.ChannelPlugin;
import com.zoyi.channel.plugin.android.ChannelStore;
import com.zoyi.channel.plugin.android.OnChatListener;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.base.BaseActivity;
import com.zoyi.channel.plugin.android.activity.chat.model.SendingMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.type.MessageType;
import com.zoyi.channel.plugin.android.activity.download.DownloadActivity;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumActivity;
import com.zoyi.channel.plugin.android.activity.photo_album.PhotoAlbumStorage;
import com.zoyi.channel.plugin.android.activity.photo_picker.PhotoPickerActivity;
import com.zoyi.channel.plugin.android.activity.settings.SettingsActivity;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.ChannelModel;
import com.zoyi.channel.plugin.android.model.rest.File;
import com.zoyi.channel.plugin.android.model.rest.Manager;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.presenter.chat.ChatContract;
import com.zoyi.channel.plugin.android.presenter.chat.ChatPresenter;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.Executor;
import com.zoyi.channel.plugin.android.util.IntentUtils;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.channel.plugin.android.util.UIUtils;
import com.zoyi.channel.plugin.android.util.io.Keyboard;
import com.zoyi.channel.plugin.android.view.handler.BackgroundToucher;
import com.zoyi.channel.plugin.android.view.handler.InfiniteScrollListener;
import com.zoyi.channel.plugin.android.view.layout.BigBar.MenuPosition;
import com.zoyi.channel.plugin.android.view.layout.MenuButton.ActionType;
import com.zoyi.channel.plugin.android.view.layout.MenuButton.MenuState;
import com.zoyi.channel.plugin.android.view.layout.WatchedEditText;

/**
 * Created by mika on 2016. 12. 7..
 */
public class ChatActivity extends BaseActivity
    implements ChatContract.View, View.OnClickListener {

  private RecyclerView recyclerView;
  private WatchedEditText editChat;
  private Button buttonSend;
  private ImageButton buttonAttach;
  private View toast;
  private View reconnectLayout;
  private View layoutInput;
  private PreviewFragment previewFragment;
  private ChatCompltedFragment compltedFragment;

  private LinearLayoutManager layoutManager;

  private ChatContract.Presenter presenter;

  private ChatAdapter adapter;
  private int recyclerViewBottom, currentPosition;
  private boolean changeChatOption = false, deleteChat = false;

  private TypingManager typingManager;

  @Override
  protected void onCreate(@Nullable Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    init(R.layout.ch_plugin_activity_chat);

    adapter = new ChatAdapter();
    presenter = new ChatPresenter(this);

    adapter.setPresenter(presenter);

    presenter.setView(this);
    presenter.setAdapterModel(adapter);
    presenter.setAdapterView(adapter);

    buttonSend = (Button) findViewById(R.id.button_send);
    buttonSend.setOnClickListener(this);
    buttonAttach = (ImageButton) findViewById(R.id.button_attach);
    buttonAttach.setOnClickListener(this);
    toast = findViewById(R.id.layout_toast);
    toast.setOnClickListener(this);
    findViewById(R.id.button_close_toast).setOnClickListener(this);
    reconnectLayout = findViewById(R.id.layout_reconnect);
    layoutInput = findViewById(R.id.layout_input);
    previewFragment = (PreviewFragment) getSupportFragmentManager().findFragmentById(R.id.f_chat_preview);
    compltedFragment = (ChatCompltedFragment) getSupportFragmentManager().findFragmentById(R.id.f_chat_completed);

    layoutManager = new LinearLayoutManager(this);
    recyclerView = (RecyclerView) findViewById(R.id.recycler_chat);
    setRecyclerViewBottom();
    recyclerView.setLayoutManager(layoutManager);
    recyclerView.setAdapter(adapter);
    recyclerView.setItemAnimator(null);
    recyclerView.getRecycledViewPool().setMaxRecycledViews(MessageType.VERIFY.toInt(), 1);
    recyclerView.addOnScrollListener(new InfiniteScrollListener(layoutManager, InfiniteScrollListener.TOP) {
      @Override
      public void onScrolled(RecyclerView recyclerView, int dx, int dy) {
        if (recyclerViewBottom == recyclerView.getBottom()) {
          currentPosition = layoutManager.findLastCompletelyVisibleItemPosition();
        }
        super.onScrolled(recyclerView, dx, dy);
      }

      @Override
      public void refresh() {
        presenter.fetchBackwardMessages();
      }

      @Override
      public void firstItem(int index) {
        presenter.calculateNewToast(index);
      }

      @Override
      public void scrollAttachedToBottom() {
        previewFragment.hide();
      }
    });
    recyclerView.addOnLayoutChangeListener(new View.OnLayoutChangeListener() {
      @Override
      public void onLayoutChange(View v, int left, int top, int right, int bottom, int oldLeft, int oldTop, int oldRight, int oldBottom) {
        if ((bottom < oldBottom) && (currentPosition == adapter.getItemCount()-1)) {
          recyclerView.postDelayed(new Runnable() {
            @Override
            public void run() {
              layoutManager.scrollToPosition(currentPosition);
            }
          }, 100);
        }
      }
    });

    editChat = (WatchedEditText) findViewById(R.id.edit_chat);
    editChat.setHandledButton(
        buttonSend,
        ContextCompat.getColor(this, R.color.ch_cobalt),
        ContextCompat.getColor(this, R.color.ch_gray));
    editChat.setText(PrefSupervisor.getStoredMessage(this));
    PrefSupervisor.clearStoredMessage(this);

    BackgroundToucher.makeKeyboardEvent(this, recyclerView, editChat);

    String chatId = null;
    int chatCount = 0;
    boolean chatEmpty = false;

    if (getIntent() != null) {
      chatId = getIntent().getStringExtra(Const.EXTRA_CHAT_ID);
      chatCount = getIntent().getIntExtra(Const.EXTRA_CHAT_COUNT, 0);
      chatEmpty = getIntent().getBooleanExtra(Const.EXTRA_CHAT_EMPTY, false);
    }

    getBigBar().withActivity(this)
        .addMenu(
            ActionType.BACK,
            MenuPosition.LEFT,
            chatEmpty ? MenuState.HIDDEN : MenuState.VISIBLE)
        .addMenu(
            ActionType.EXIT,
            MenuPosition.LEFT,
            chatEmpty ? MenuState.VISIBLE : MenuState.HIDDEN)
        .addText(MenuPosition.LEFT)
        .addMenu(ActionType.REFRESH, MenuPosition.RIGHT, MenuState.HIDDEN)
        .addMenu(ActionType.MORE, MenuPosition.RIGHT, chatEmpty ? MenuState.VISIBLE : MenuState.HIDDEN)
        .build();

    getBigBar().setText(chatCount);

    if (SocketManager.isError()) {
      setReconnectVisibility(true);
    }

    setStyle();
    if (chatId != null) {
      presenter.setChatId(chatId);
    }
    presenter.init();

    typingManager = new TypingManager(chatId, UserChat.CLASSNAME, editChat);
  }

  private void setRecyclerViewBottom() {
    new Handler().post(new Runnable() {
      @Override
      public void run() {
        if (recyclerView.getBottom() > 0) {
          recyclerViewBottom = recyclerView.getBottom();
        } else {
          setRecyclerViewBottom();
        }
      }
    });
  }

  @Override
  public void finish() {
    if (presenter != null) {
      if (presenter.getChatId() == null) {
        if (!changeChatOption && !deleteChat) {
          setResult(Const.RESULT_CHAT_NOT_EXISTS);
        }
      } else {
        presenter.leaveChat();
      }
    }
    super.finish();
  }

  private void setStyle() {
    Channel channel = ChannelStore.getChannel();

    Plugin plugin = ChannelStore.getPlugin();

    if (plugin != null) {
      presenter.setColor(plugin.getBackgroundColor(), plugin.getTextColor());

      setStatusBarColor(plugin.getBackgroundColor());
      getBigBar().setTheme(plugin.getBackgroundColor(), plugin.getTextColor());
    }

    if (channel != null) {
      getBigBar().setTitle(channel.getName());
    }
  }

  @Override
  public void onLoadUserChat() {
    getBigBar().setMenuState(ActionType.EXIT, MenuState.HIDDEN);
    getBigBar().setMenuState(ActionType.BACK, MenuState.VISIBLE);

    typingManager.setChatId(presenter.getChatId());
  }

  @Override
  public void setInputLayoutVisibility() {
    if (presenter.isStateCompleted()) {
      Keyboard.close(this, editChat);

      compltedFragment.show(editChat.getText().toString().trim());
      layoutInput.setVisibility(View.GONE);
    } else {
      compltedFragment.hide();
      layoutInput.setVisibility(View.VISIBLE);
    }
  }

  @Override
  public boolean isBottomPosition() {
    return layoutManager.findLastCompletelyVisibleItemPosition() == adapter.getItemCount() - 1;
  }

  @Override
  public void scrollToBottom(boolean force) {
    if (layoutManager.findLastCompletelyVisibleItemPosition() == adapter.getItemCount() - 2 || force) {
      layoutManager.scrollToPosition(adapter.getItemCount() - 1);
    }
  }

  @Override
  public void scrollTo(int index) {
    recyclerView.smoothScrollToPosition(index);
  }

  @Override
  protected void onActivityResult(int requestCode, int resultCode, Intent data) {
    super.onActivityResult(requestCode, resultCode, data);
    switch (requestCode) {
      case Const.PHOTO_REQUEST_CODE:
        if (resultCode == Const.PHOTO_RESULT_CODE) {
          presenter.sendImageFiles(data.getStringArrayListExtra(Const.PHOTO_INTENT_KEY));
          break;
        }

      case Const.REQUEST_PHOTO_ALBUM:
        PhotoAlbumStorage.getInstance().release();
        break;

      case Const.REQUEST_PLUGIIN_SETTINGS:
        switch (resultCode) {
          case Const.RESULT_CHAT_OPTION_CHANGED:
            changeChatOption = true;
            setResult(Const.RESULT_CHAT_OPTION_CHANGED);
            break;

          case Const.RESULT_CHAT_REMOVE:
            deleteChat = true;
            setResult(Const.RESULT_CHAT_REMOVE, data);
            finish();
            break;
        }
        break;
    }
  }

  @Override
  public void receiveCommand(Command command, Object object) {
    presenter.receiveCommand(command, object);
  }

  @Override
  public void receiveData(ChannelModel channelModel, boolean upsert) {
    presenter.receiveData(channelModel, upsert);
  }

  @Override
  public void receiveChatCounter(int count) {
    if (count > 0) {
      onLoadUserChat();
    }

    getBigBar().setText(count);
  }

  @Override
  public void receiveTyping(Typing typing) {
    if (typing != null
        && typing.isOther()
        && typing.isSameChat(presenter.getChatId(), UserChat.CLASSNAME)) {
      typingManager.addTTLTimer(typing);

      presenter.receiveTyping(typing);
    }
  }

  @Override
  public void optionClicked(ActionType actionType) {
    super.optionClicked(actionType);
    switch (actionType) {
      case REFRESH:
        presenter.refresh();
        break;

      case MORE:
        IntentUtils.setNextActivity(this, SettingsActivity.class)
            .startActivityForResult(Const.REQUEST_PLUGIIN_SETTINGS);
        break;
    }
  }

  @Override
  public void setRefreshVisibility(boolean showRefresh) {
    getBigBar().setMenuState(ActionType.REFRESH, showRefresh ? MenuState.VISIBLE : MenuState.HIDDEN);
  }

  @Override
  public void setReconnectVisibility(boolean show) {
    reconnectLayout.setVisibility(UIUtils.getVisible(show, true));
  }

  @Override
  public void setToastVisibility(boolean showToast) {
    toast.setVisibility(UIUtils.getVisible(showToast, true));
  }

  @Override
  public void showBottomPreview(Manager manager, Message message) {
    if (manager == null || message == null || message.getLog() != null || !layoutManager.canScrollVertically()) {
      return;
    }
    if (isBottomPosition()) {
      return;
    }
    previewFragment.show(manager);
  }

  @Override
  public void onRemovedChat() {
    finish();
  }

  @Override
  public void onClick(View v) {
    int id = v.getId();

    if (id == R.id.layout_toast) {
      presenter.newToastClicked();
    }
    if (id == R.id.button_close_toast) {
      presenter.hideNewToast();
    }
    if (id == R.id.button_send) {
      presenter.sendTextMessage(editChat.getString());
      editChat.setText("");
    }
    if (id == R.id.button_attach) {
      hideKeyboard(editChat);
      IntentUtils.setNextActivity(ChatActivity.this, PhotoPickerActivity.class)
          .startActivityForResult(Const.PHOTO_REQUEST_CODE);
    }
  }

  @Override
  public void onFileDownload(Message message) {
    IntentUtils.setNextActivity(this, DownloadActivity.class)
        .putExtra(Const.EXTRA_URL, message.getFile().getUrl())
        .putExtra(Const.EXTRA_FILENAME, message.getFile().getFilename())
        .putExtra(Const.EXTRA_IMAGE, message.getFile().isImage())
        .startActivity();
  }

  @Override
  public void onShowPhotoAlbum(File file) {
    IntentUtils.setNextActivity(this, PhotoAlbumActivity.class)
        .putExtra(Const.EXTRA_FILE_ID, file.getId())
        .startActivityForResult(Const.REQUEST_PHOTO_ALBUM);
  }

  @Override
  public void onUrlClicked(String url) {
    boolean handle = false;
    OnChatListener listener = ChannelPlugin.getOnChatListener();
    if (listener != null) {
      handle = listener.onClickChatLink(url);
    }
    if (!handle) {
      Executor.goWeb(this, url);
    }
  }

  @Override
  public void sendingMessageClicked(final SendingMessageItem item) {
    final AlertDialog dialog = new AlertDialog.Builder(this)
        .setMessage(ResUtils.getString(this, "ch.chat.resend.description"))
        .setPositiveButton(ResUtils.getString(this, "ch.chat.resend"), new DialogInterface.OnClickListener() {
          @Override
          public void onClick(DialogInterface dialog, int which) {
            presenter.resend(item);
          }
        })
        .setNegativeButton(ResUtils.getString(this, "ch.chat.resend.cancel"), null)
        .setNeutralButton(ResUtils.getString(this, "ch.chat.delete"), new DialogInterface.OnClickListener() {
          @Override
          public void onClick(DialogInterface dialog, int which) {
            presenter.removeFailedItem(item);
          }
        })
        .setCancelable(true)
        .create();

    dialog.setOnShowListener(new DialogInterface.OnShowListener() {
      @Override
      public void onShow(DialogInterface args) {
        int dark = ContextCompat.getColor(ChatActivity.this, R.color.ch_dark);
        int cobalt = ContextCompat.getColor(ChatActivity.this, R.color.ch_cobalt);

        dialog.getButton(AlertDialog.BUTTON_POSITIVE).setTextColor(cobalt);
        dialog.getButton(AlertDialog.BUTTON_NEGATIVE).setTextColor(dark);
        dialog.getButton(AlertDialog.BUTTON_NEUTRAL).setTextColor(dark);
      }
    });
    dialog.show();
  }
}