package com.zoyi.channel.plugin.android;

import android.app.Application;
import android.content.Context;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.Size;
import android.text.TextUtils;

import com.splunk.mint.Mint;
import com.zoyi.channel.plugin.android.activity.chat.ChatManager;
import com.zoyi.channel.plugin.android.activity.userchat_list.UserChatListActivity;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.CheckInPrefSupervisor;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.rest.Channel;
import com.zoyi.channel.plugin.android.model.rest.Event;
import com.zoyi.channel.plugin.android.model.wrapper.PackageWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.network.ChannelApi;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.network.ServiceFactory;
import com.zoyi.channel.plugin.android.push.ChannelPushManager;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.IntentUtils;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.RequestUtils;
import com.zoyi.channel.plugin.android.wrapper.Tracker;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Created by mika on 2016. 4. 19..
 */
public class ChannelPlugin {
  private static final String MINT_DEBUG_KEY = "e72f17f4";
  private static final String MINT_PROD_KEY = "22464da6";

  private static boolean isDebugMode = false;
  private static boolean isEnabledTrackDefaultEvent = true;

  private static ChannelPlugin channelPlugin;
  private static ChannelApi channelApi;

  private Application application;
  private String pluginId;
  private Thread.UncaughtExceptionHandler mUncaughtExceptionHandler;

  private AtomicInteger checkInCounter = new AtomicInteger(0);

  public static void initialize(Application application, String pluginId) {
    initialize(application, pluginId, false, true);
  }

  public static void initialize(Application application, String pluginId, boolean debugMode) {
    initialize(application, pluginId, debugMode, true);
  }

  public static void initialize(
      Application application,
      String pluginId,
      boolean debugMode,
      boolean enabledTrackDefaultEvent) {
    if (channelPlugin != null) {
      L.i("Channel plugin already initialized");
      return;
    }
    if (application == null) {
      L.e("Application cannot be null");
      return;
    }
    if (pluginId == null) {
      L.e("Plugin key cannot be null");
      return;
    }
    isDebugMode = debugMode;
    isEnabledTrackDefaultEvent = enabledTrackDefaultEvent;
    channelPlugin = new ChannelPlugin(application, pluginId);
  }

  private ChannelPlugin(Application application, String pluginId) {
    this.application = application;
    this.pluginId = pluginId;

    ChannelStore.create(application, pluginId);
    SocketManager.create(application);

    application.registerActivityLifecycleCallbacks(new ActivityLifecycleManager());

    mUncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
    Thread.setDefaultUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
      @Override
      public void uncaughtException(Thread thread, Throwable ex) {
        if (Looper.getMainLooper().getThread() == thread) {
          mUncaughtExceptionHandler.uncaughtException(thread, ex);
        } else if (ex instanceof RejectedExecutionException) {
          SocketManager.reconnect();
        }
      }
    });
  }

  public static void checkIn() {
    checkIn(null, null);
  }

  public static void checkIn(OnCheckInListener listener) {
    checkIn(null, listener);
  }

  public static void checkIn(CheckIn checkIn) {
    checkIn(checkIn, null);
  }

  public static void checkIn(CheckIn checkIn, OnCheckInListener listener) {
    if (isInitialized()) {
      if (!isDataStored()) {
        channelPlugin.checkVersion(checkIn, listener);
      } else {
        if (listener != null) {
          listener.onFailed(new ChannelException(
              ChannelException.StatusCode.ALREADY_CHECKED_IN,
              "Already checked in. Please check out and try again."));
        }
      }
    } else {
      if (listener != null) {
        listener.onFailed(new ChannelException(
            ChannelException.StatusCode.NOT_INITIALIZED,
            "Please initialize first"));
      }
    }
  }

  private void checkVersion(final CheckIn checkIn, final OnCheckInListener listener) {
    final int counter = checkInCounter.incrementAndGet();

    getApi().getLastestPackage("com.zoyi.channel.plugin.android", BuildConfig.VERSION_NAME)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<PackageWrapper>() {
          @Override
          public void onError(Throwable error) {
            if (counter != checkInCounter.get()) {
              return;
            }
            if (listener != null) {
              listener.onFailed(new ChannelException(
                  ChannelException.StatusCode.NETWORK_TIMEOUT,
                  error.getMessage()));
            }
          }

          @Override
          public void onNext(PackageWrapper wrapper) {
            if (counter != checkInCounter.get()) {
              // show error
              return;
            }

            if (!wrapper.isNeedToUpgrade()) {
              if (!CompareUtils.isSame(wrapper.getVersionString(), BuildConfig.VERSION_NAME)) {
                L.i("Newest version is: " + wrapper.getVersionString());
              }
              checkInProcess(checkIn, listener, counter);
            } else {
              if (listener != null) {
                listener.onFailed(new ChannelException(
                    ChannelException.StatusCode.NOT_AVAILABLE_VERSION,
                    "Need to upgrade, Minimum version is: " + wrapper.getMinCompatibleVersion()
                ));
              }
            }
          }
        });
  }

  private void checkInProcess(
      final CheckIn checkIn,
      final OnCheckInListener listener,
      final int counter) {
    Map<String, Object> form = new HashMap<>();
    if (checkIn != null) {
      form.put("name", checkIn.getName());
      form.put("mobileNumber", checkIn.getMobileNumber());
      form.put("avatarUrl", checkIn.getAvatarUrl());
      form.put("meta", checkIn.getMeta());
    }

    RequestBody body = RequestUtils.form(form).create();
    Map<String, String> headers = new HashMap<>();

    if (checkIn != null && checkIn.getUserId() != null) {
      headers.put("X-User-Id", checkIn.getUserId());
    }
    if (PrefSupervisor.getVeilId(application) != null) {
      headers.put("X-Veil-Id", PrefSupervisor.getVeilId(application));
    }

    getApi().checkIn(headers, pluginId, null, body)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<PluginWrapper>() {
          @Override
          public void onError(Throwable e) {
            if (counter != checkInCounter.get()) {
              return;
            }

            L.e(e.getMessage());
            if (listener != null) {
              ChannelException.StatusCode code = ChannelException.StatusCode.CHECK_IN_ERROR;

              if (e instanceof RetrofitException) {
                switch (((RetrofitException) e).getKind()) {
                  case HTTP:
                    code = ChannelException.StatusCode.CHECK_IN_ERROR;
                    break;

                  case NETWORK:
                    code = ChannelException.StatusCode.NETWORK_TIMEOUT;
                    break;

                  case UNEXPECTED:
                    code = ChannelException.StatusCode.UNKNOWN;
                    break;
                }
              }

              listener.onFailed(new ChannelException(code, e.getMessage()));
            }
          }

          @Override
          public void onNext(PluginWrapper pluginWrapper) {
            if (counter != checkInCounter.get()) {
              return;
            }

            initBugTracking(pluginWrapper.getChannel());

            if (checkIn == null) {
              CheckInPrefSupervisor.setVeil(application, pluginWrapper.getVeil());
            } else {
              CheckInPrefSupervisor.set(application, checkIn);
            }

            ChannelStore.checkIn(pluginWrapper);

            ChannelPushManager.sendTokenToChannelPlugin(application);

            SocketManager.setChannelId(pluginWrapper.getChannel().getId());
            SocketManager.connect();

            RxBus.post(new CommandBus(Command.CHECKED_IN, pluginWrapper.getPlugin()));

            if (isEnabledTrackDefaultEvent) {
              track(application, Tracker.Name.CHECK_IN, null);
            }

            if (listener != null) {
              listener.onSuccessed();
            }
          }
        });
  }

  public static void checkOut() {
    if (isInitialized()) {
      channelPlugin.checkOutProcess();
    }
  }

  private void checkOutProcess() {
    checkInCounter.incrementAndGet();

    ChannelPushManager.deleteToken(application);
    CheckInPrefSupervisor.clear(application);
    ChannelStore.clear();
    RxBus.post(new CommandBus(Command.CHECKED_OUT));

    SocketManager.setChannelId(null);
    SocketManager.disconnect();

    ChatManager.release();

    releaseBugTracking();
  }

  public static boolean launch(Context context) {
    if (!isInitialized() || !isDataStored() || context == null) {
      L.e("Please check in first");
      return false;
    }
    IntentUtils.setNextActivity(context, UserChatListActivity.class).startActivity();
    return true;
  }

  public static ChannelApi getApi() {
    if (channelApi == null) {
      channelApi = ServiceFactory.create();
    }
    return channelApi;
  }

  @Nullable
  public static String getPluginId() {
    if (channelPlugin == null) {
      return null;
    }
    return channelPlugin.pluginId;
  }

  private void initBugTracking(Channel channel) {
    if (application != null) {
      Mint.disableNetworkMonitoring();

      Mint.initAndStartSession(application, MINT_PROD_KEY);
      Mint.addExtraData("plugin_version", BuildConfig.VERSION_NAME);

      if (channel != null) {
        Mint.addExtraData("channel_id", channel.getId());
        Mint.addExtraData("channel_name", channel.getName());
      }
    }
  }

  private void releaseBugTracking() {
    if (application != null) {
      Mint.clearExtraData();
      Mint.closeSession(application);
      Mint.flush();
    }
  }

  public static boolean isInitialized() {
    if (channelPlugin == null) {
      L.e("Please initialize first");
      return false;
    }
    return true;
  }

  public static boolean isDebugMode() {
    return isDebugMode;
  }

  public static boolean isEnabledTrackDefaultEvent() {
    return isEnabledTrackDefaultEvent;
  }

  public static boolean isDataStored() {
    return ChannelStore.isDataStored();
  }

  public static void updateGuest() {
    updateGuest(null);
  }

  public static void updateGuest(OnGuestUpdatedListener listener) {
    ChannelStore.fetchMe(listener);
  }

  public static void addOnChannelPluginChangedListener(OnChannelPluginChangedListener listener) {
    ChannelStore.addOnChannelPluginChangedListener(listener);
  }

  public static void removeOnChannelPluginChangedListener(OnChannelPluginChangedListener listener) {
    ChannelStore.removeOnChannelPluginChangedListener(listener);
  }

  public static void track(
      @NonNull Context context,
      @NonNull @Size(min = 1L,max = 30L) String name,
      @Nullable Map<String, Object> properties) {
    if (TextUtils.isEmpty(name) || name.length() > 30) {
      return;
    }
    track(new Event(context, name, properties));
  }

  private static void track(@NonNull Event event) {
    getApi().trackEvent(event)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<Void>());
  }
}
