package com.zoyi.channel.plugin.android;

import android.app.Application;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.ScriptsWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.channel.plugin.android.util.Utils;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscriber;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Created by mika on 2017. 2. 7..
 */
public class ChannelStore {
  private static ChannelStore channelStore;

  private Application application;
  private String pluginId;

  private Plugin plugin;
  private Channel channel;
  private User user;
  private Veil veil;

  private boolean mainRunning = false;

  private Subscription subscription;
  private Map<String, Script> scriptMap;

  static void create(Application application, String pluginId) {
    if (channelStore == null) {
      channelStore = new ChannelStore(application, pluginId);
    }
  }

  private ChannelStore(Application application, String pluginId) {
    this.application = application;
    this.pluginId = pluginId;
  }

  static void checkIn(PluginWrapper pluginWrapper) {
    if (channelStore != null) {
      PrefSupervisor.setVeilId(channelStore.application, pluginWrapper.getVeilId());

      if (pluginWrapper.getUser() != null) {
        channelStore.user = pluginWrapper.getUser();
      }
      channelStore.plugin = pluginWrapper.getPlugin();
      channelStore.channel = pluginWrapper.getChannel();
      channelStore.veil = pluginWrapper.getVeil();

      channelStore.startFetchPlugin();
      channelStore.fetchScripts();
    }
  }

  static void clear() {
    if (channelStore != null) {
      channelStore.stopFetchPlugin();
      channelStore.plugin = null;
      channelStore.channel = null;
      channelStore.user = null;
      channelStore.veil = null;
    }
  }

  private void startFetchPlugin() {
    stopFetchPlugin();
    subscription = Observable.interval(3, TimeUnit.MINUTES)
        .subscribe(new Subscriber<Long>() {
          @Override
          public void onCompleted() {
          }

          @Override
          public void onError(Throwable e) {
          }

          @Override
          public void onNext(Long aLong) {
            if (channelStore != null) {
              channelStore.fetchPlugin();
            }
          }
        });
  }

  private void stopFetchPlugin() {
    if (subscription != null && !subscription.isUnsubscribed()) {
      subscription.unsubscribe();
      subscription = null;
    }
  }

  private void fetchPlugin() {
    if (pluginId != null) {
      ChannelPlugin.getApi().getPlugin(pluginId)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<PluginWrapper>() {
            @Override
            public void onNext(PluginWrapper pluginWrapper) {
              if (pluginWrapper != null) {
                plugin = pluginWrapper.getPlugin();
                channel = pluginWrapper.getChannel();
              }
            }
          });
    }
  }

  private void fetchScripts() {
    ChannelPlugin.getApi().getScripts()
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<ScriptsWrapper>() {
          @Override
          public void onNext(ScriptsWrapper scriptsWrapper) {
            setScripts(scriptsWrapper.getScripts());
          }
        });
  }

  private void setScripts(List<Script> scripts) {
    if (scriptMap == null) {
      scriptMap = new HashMap<>();
    }

    if (scripts != null) {
      for (Script script : scripts) {
        if (script.getKey() != null && script.getMessage() != null) {
          scriptMap.put(script.getKey(), script);
        }
      }
    }
  }

  public static String getScript(String key, String defaultResourceKey) {
    if (channelStore != null) {
      if (channelStore.scriptMap != null && channelStore.scriptMap.containsKey(key)) {
        String locale = Locale.getDefault().getLanguage();
        if (channelStore.scriptMap.get(key).getMessage(locale) != null) {
          return channelStore.scriptMap.get(key).getMessage(locale);
        }
      }
      if (channelStore.application != null) {
        return ResUtils.getString(channelStore.application, defaultResourceKey);
      }
    }
    return "";
  }

  public static Channel getChannel() {
    return channelStore != null ? channelStore.channel : null;
  }

  public static User getUser() {
    return channelStore != null ? channelStore.user : null;
  }

  public static String getMobileNumber() {
    if (channelStore != null) {
      if (channelStore.user != null) {
        return channelStore.user.getMobileNumber();
      }
      if (channelStore.veil != null) {
        return channelStore.veil.getMobileNumber();
      }
    }
    return null;
  }

  public static String getPersonType() {
    if (channelStore != null && channelStore.user != null) {
      return User.CLASSNAME;
    }
    return Veil.CLASSNAME;
  }

  public static String getPersonId() {
    if (channelStore != null) {
      if (channelStore.user != null) {
        return channelStore.user.getId();
      }
      return PrefSupervisor.getVeilId(channelStore.application);
    }
    return null;
  }

  public static String getVeilId() {
    if (channelStore != null && channelStore.application != null) {
      return PrefSupervisor.getVeilId(channelStore.application);
    }
    return null;
  }

  public static String getUserId() {
    if (channelStore != null && channelStore.user != null) {
      return channelStore.user.getId();
    }
    return null;
  }

  public static boolean isWorking() {
    return channelStore != null && channelStore.channel != null && channelStore.channel.isWorking();
  }

  public static boolean hasName() {
    if (channelStore != null) {
      if (channelStore.user != null) {
        return !channelStore.user.isGhost();
      }
      if (channelStore.veil != null) {
        return !channelStore.veil.isGhost();
      }
    }
    return true;
  }

  public static void setUserVeil(User user, Veil veil) {
    if (channelStore != null) {
      if (user != null) {
        channelStore.user = user;
      }
      if (veil != null) {
        channelStore.veil = veil;
      }
    }
  }

  public static Plugin getPlugin() {
    if (channelStore != null) {
      return channelStore.plugin;
    }
    return null;
  }

  public static String getPluginId() {
    if (channelStore != null) {
      return channelStore.pluginId;
    }
    return null;
  }

  static void setMainRunning(boolean flag) {
    if (channelStore != null) {
      channelStore.mainRunning = flag;
    }
  }

  static boolean isMainRunning() {
    return channelStore != null && channelStore.mainRunning;
  }

  static boolean isCheckedIn() {
    return channelStore != null && channelStore.channel != null && channelStore.plugin != null;
  }
}
