package com.zoyi.channel.plugin.android;

import android.app.Application;
import android.content.Context;
import android.os.Looper;

import com.splunk.mint.Mint;
import com.zoyi.channel.plugin.android.activity.userchat_list.UserChatListActivity;
import com.zoyi.channel.plugin.android.enumerate.Command;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.wrapper.PackageWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.network.ChannelApi;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.ServiceFactory;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.IntentUtils;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.RequestUtils;
import com.zoyi.retrofit2.okhttp3.RequestBody;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.RejectedExecutionException;

/**
 * Created by mika on 2016. 4. 19..
 */
public class ChannelPlugin {
  private static boolean isDebugMode = false;

  private static ChannelPlugin channelPlugin;
  private static ChannelApi channelApi;

  private Application application;
  private String pluginId;
  private Thread.UncaughtExceptionHandler mUncaughtExceptionHandler;

  private boolean doCheckIn = false;

  public static void initialize(Application application, String pluginId) {
    initialize(application, pluginId, false);
  }

  public static void initialize(Application application, String pluginId, boolean debugMode) {
    if (channelPlugin != null) {
      L.i("Channel plugin already initialized");
      return;
    }
    if (application == null) {
      L.e("Application cannot be null");
      return;
    }
    if (pluginId == null) {
      L.e("Plugin key cannot be null");
      return;
    }
    isDebugMode = debugMode;
    channelPlugin = new ChannelPlugin(application, pluginId);
  }

  private ChannelPlugin(Application application, String pluginId) {
    this.application = application;
    this.pluginId = pluginId;

    ChannelStore.create(application, pluginId);
    SocketManager.create(application);

    application.registerActivityLifecycleCallbacks(new ActivityLifecycleManager());

    mUncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
    Thread.setDefaultUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
      @Override
      public void uncaughtException(Thread thread, Throwable ex) {
        if (Looper.getMainLooper().getThread() == thread) {
          mUncaughtExceptionHandler.uncaughtException(thread, ex);
        } else if (ex instanceof RejectedExecutionException) {
          SocketManager.reconnect();
        }
      }
    });
  }

  private void checkVersion(final CheckIn checkIn, final OnCheckInListener listener) {
    if (!doCheckIn) {
      doCheckIn = true;
      checkOut();

      getApi().getLastestPackage("com.zoyi.channel.plugin.android", BuildConfig.VERSION_NAME)
          .subscribeOn(Schedulers.newThread())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(new RestSubscriber<PackageWrapper>() {
            @Override
            public void onError(Throwable error) {
              L.e(error.getMessage());
              doCheckIn = false;
              if (listener != null) {
                listener.onFailed();
              }
            }

            @Override
            public void onNext(PackageWrapper wrapper) {
              if (!wrapper.isNeedToUpgrade()) {
                L.i(wrapper.getVersionString() + " " + BuildConfig.VERSION_NAME);
                if (!CompareUtils.isSame(wrapper.getVersionString(), BuildConfig.VERSION_NAME)) {
                  L.i("Newest version is: " + wrapper.getVersionString());
                }
                checkInProcess(checkIn, listener);
              } else {
                L.e("Need to upgrade, Minimum version is: " + wrapper.getMinCompatibleVersion());
                doCheckIn = false;
                if (listener != null) {
                  listener.onFailed();
                }
              }
            }
          });
    }
  }

  private void checkInProcess(CheckIn checkIn, final OnCheckInListener listener) {
    Map<String, Object> form = new HashMap<>();
    if (checkIn != null) {
      form.put("name", checkIn.getName());
      form.put("mobileNumber", checkIn.getMobileNumber());
      form.put("avatarUrl", checkIn.getAvatarUrl());
      form.put("meta", checkIn.getMeta());
    }

    RequestBody body = RequestUtils.form(form).create();
    Map<String, String> headers = new HashMap<>();

    if (checkIn != null && checkIn.getUserId() != null) {
      headers.put("X-User-Id", checkIn.getUserId());
    }
    if (PrefSupervisor.getVeilId(application) != null) {
      headers.put("X-Veil-Id", PrefSupervisor.getVeilId(application));
    }

    getApi().checkIn(headers, pluginId, "channel.io", body)
        .subscribeOn(Schedulers.newThread())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(new RestSubscriber<PluginWrapper>() {
          @Override
          public void onError(Throwable e) {
            L.e(e.getMessage());
            doCheckIn = false;
            if (listener != null) {
              listener.onFailed();
            }
          }

          @Override
          public void onNext(PluginWrapper pluginWrapper) {
            initBugTracking();

            ChannelStore.checkIn(pluginWrapper);

            SocketManager.setChannelId(pluginWrapper.getChannel().getId());
            SocketManager.connect();

            RxBus.post(new CommandBus(Command.CHECKED_IN, pluginWrapper.getPlugin()));

            doCheckIn = false;

            if (listener != null) {
              listener.onSuccessed();
            }
          }
        });
  }

  private void checkOutProcess() {
    ChannelStore.clear();
    RxBus.post(new CommandBus(Command.CHECKED_OUT));

    SocketManager.setChannelId(null);
    SocketManager.disconnect();

    releaseBugTracking();
  }

  private static boolean isInitialized() {
    if (channelPlugin == null) {
      L.e("Please initialize first");
      return false;
    }
    return true;
  }

  public static void checkIn() {
    checkIn(null, null);
  }

  public static void checkIn(OnCheckInListener listener) {
    checkIn(null, listener);
  }

  public static void checkIn(CheckIn checkIn) {
    checkIn(checkIn, null);
  }

  public static void checkIn(CheckIn checkIn, OnCheckInListener listener) {
    if (isInitialized()) {
      channelPlugin.checkVersion(checkIn, listener);
    } else {
      if (listener != null) {
        listener.onFailed();
      }
    }
  }

  public static void checkOut() {
    if (isInitialized()) {
      channelPlugin.checkOutProcess();
    }
  }

  public static void launch(Context context) {
    if (isInitialized() && ChannelStore.isCheckedIn() && context != null) {
      IntentUtils.setNextActivity(context, UserChatListActivity.class).startActivity();
    } else {
      L.e("Please check in first");
    }
  }

  public static ChannelApi getApi() {
    if (channelApi == null) {
      channelApi = ServiceFactory.create();
    }
    return channelApi;
  }

  public static boolean isDebugMode() {
    return isDebugMode;
  }

  private void initBugTracking() {
    if (application != null) {
      Mint.disableNetworkMonitoring();
      Mint.initAndStartSession(application, "22464da6");
      Mint.addExtraData("plugin_version", BuildConfig.VERSION_NAME);
    }
  }

  private void releaseBugTracking() {
    if (application != null) {
      Mint.clearExtraData();
      Mint.closeSession(application);
      Mint.flush();
    }
  }
}
