package com.zoyi.channel.plugin.android.view.textview;

import android.content.Context;
import android.content.res.TypedArray;

import androidx.annotation.Nullable;
import androidx.appcompat.widget.AppCompatImageView;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.TextView;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.model.rest.File;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.util.io.FilenameUtils;

public class FileRowTextView extends FrameLayout {

  private TextView textFileName;
  private TextView textFileExtension;
  private TextView textFileSize;
  private TextView textFileCount;

  public FileRowTextView(Context context) {
    super(context);
    init(context, null);
  }

  public FileRowTextView(Context context, AttributeSet attrs) {
    super(context, attrs);
    init(context, attrs);
  }

  public FileRowTextView(Context context, AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context, attrs);
  }

  @Initializer
  private void init(Context context, @Nullable AttributeSet attrs) {
    View root = LayoutInflater.from(context).inflate(R.layout.ch_plugin_view_file_row_text, this, true);

    AppCompatImageView imageFileClip = root.findViewById(R.id.ch_imageFileClip);
    textFileName = root.findViewById(R.id.ch_textFileName);
    textFileExtension = root.findViewById(R.id.ch_textFileExtension);
    textFileSize = root.findViewById(R.id.ch_textFileSize);
    textFileCount = root.findViewById(R.id.ch_textFileCount);

    if (attrs != null) {
      TypedArray typedArray = context.getTheme().obtainStyledAttributes(
          attrs,
          R.styleable.FileRowTextView,
          0,
          0
      );

      try {
        int iconSize = typedArray.getDimensionPixelSize(
            R.styleable.FileRowTextView_ch_cfr_iconSize,
            getResources().getDimensionPixelSize(R.dimen.ch_file_row_text_image_default_size)
        );
        int textSizeFileName = typedArray.getDimensionPixelSize(
            R.styleable.FileRowTextView_ch_cfr_fileNameTextSize,
            getResources().getDimensionPixelSize(R.dimen.ch_file_row_text_name_default_size)
        );
        int textSizeFileSize = typedArray.getDimensionPixelSize(
            R.styleable.FileRowTextView_ch_cfr_fileSizeTextSize,
            getResources().getDimensionPixelSize(R.dimen.ch_file_row_text_size_default_size)
        );
        int textSizeFileCount = typedArray.getDimensionPixelSize(
            R.styleable.FileRowTextView_ch_cfr_fileCountTextSize,
            getResources().getDimensionPixelSize(R.dimen.ch_file_row_text_count_default_size)
        );
        int fileColor = typedArray.getColor(
            R.styleable.FileRowTextView_ch_cfr_themeColor,
            ResUtils.getColor(R.color.ch_grey900)
        );

        boolean useOnlyName = typedArray.getBoolean(
            R.styleable.FileRowTextView_ch_cfr_useOnlyName,
            false
        );

        ViewGroup.LayoutParams layoutParams = imageFileClip.getLayoutParams();
        layoutParams.width = iconSize;
        layoutParams.height = iconSize;

        imageFileClip.setLayoutParams(layoutParams);

        textFileName.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeFileName);
        textFileSize.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeFileSize);
        textFileCount.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeFileCount);

        imageFileClip.setColorFilter(fileColor);
        textFileName.setTextColor(fileColor);
        textFileExtension.setTextColor(fileColor);
        textFileCount.setTextColor(fileColor);

        if (useOnlyName) {
          textFileSize.setVisibility(View.GONE);
          textFileCount.setVisibility(View.GONE);
        }
      } finally {
        typedArray.recycle();
      }
    }
  }

  public void setFile(File file, int count) {
    String name = file.getName();
    String ext = FilenameUtils.getExtension(name);
    textFileName.setText(FilenameUtils.getBaseName(name) + ".");
    if (!TextUtils.isEmpty(ext)) {
      textFileExtension.setVisibility(View.VISIBLE);
      textFileExtension.setText(ext);
    } else {
      textFileExtension.setVisibility(View.GONE);
    }
    textFileSize.setText("(" + FormatUtils.formatFileSize(file.getSize()) + ")");

    if (count > 1) {
      textFileCount.setVisibility(View.VISIBLE);
      textFileCount.setText("+" + (count - 1));
    } else {
      textFileCount.setVisibility(View.GONE);
    }
  }
}
