package com.zoyi.channel.plugin.android;

import android.app.Activity;
import android.content.Context;
import android.graphics.Color;
import android.os.Build;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.FrameLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.zoyi.channel.plugin.android.open.option.ChannelButtonOption;
import com.zoyi.channel.plugin.android.selector.GlobalSelector;
import com.zoyi.channel.plugin.android.selector.SettingsSelector;
import com.zoyi.channel.plugin.android.bind.Binder;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.channel.plugin.android.view.external.fab.FloatingActionButton;

import static android.widget.RelativeLayout.ALIGN_PARENT_BOTTOM;
import static android.widget.RelativeLayout.ALIGN_PARENT_END;
import static android.widget.RelativeLayout.ALIGN_PARENT_LEFT;
import static android.widget.RelativeLayout.ALIGN_PARENT_RIGHT;
import static android.widget.RelativeLayout.ALIGN_PARENT_START;
import static android.widget.RelativeLayout.ALIGN_PARENT_TOP;

/**
 * Created by jerry on 2018. 11. 16..
 */

public class ChannelLauncherView extends FrameLayout {

  public static final int BOTTOM_LEFT = 0;
  public static final int BOTTOM_RIGHT = 1;

  private Context context;

  private FloatingActionButton buttonFloatingAction;
  private View layoutLauncherBadge;
  private TextView textBadgeCount;

  private int defaultMargin;

  private Animation launcherAnimation;
  private Animation badgeAnimation;

  @Nullable
  private Binder launcherConfigBinder;

  @Nullable
  private Binder styleBinder;

  public ChannelLauncherView(@NonNull Context context) {
    super(context);
    initView(context);
  }

  public ChannelLauncherView(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    initView(context);
  }

  public ChannelLauncherView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    initView(context);
  }

  private void initView(Context context) {
    this.context = context;

    View view = LayoutInflater.from(context).inflate(R.layout.ch_plugin_view_launcher, this, true);

    buttonFloatingAction = view.findViewById(R.id.ch_fabLauncher);
    buttonFloatingAction.setColorRipple(Color.WHITE);
    buttonFloatingAction.setOnClickListener(v -> {
      Activity activity = ContextUtils.getActivity(context);
      if (activity != null) {
        ChannelIO.showMessenger(activity);
      }
    });

    layoutLauncherBadge = view.findViewById(R.id.ch_layoutLauncherBadge);
    textBadgeCount = view.findViewById(R.id.ch_textLauncherBadge);

    defaultMargin = context.getResources().getDimensionPixelSize(R.dimen.ch_default_channel_button_margin);

    launcherAnimation = AnimationUtils.loadAnimation(context, R.anim.ch_plugin_anim_launcher);
    badgeAnimation = AnimationUtils.loadAnimation(context, R.anim.ch_plugin_anim_launcher_badge);

    // initialize state is gone to show animation
    setVisibility(View.GONE);
    textBadgeCount.setVisibility(View.GONE);
  }

  @Override
  protected void onAttachedToWindow() {
    super.onAttachedToWindow();

    launcherConfigBinder = SettingsSelector.bindLauncherConfig(this::setLauncherLayout);

    styleBinder = GlobalSelector.bindLauncherStyle((plugin, alert, hasMarketingChat) -> {
      setLauncherBackgroundColor(plugin.getBackgroundColor(), plugin.getGradientColor());
      setBadgeCount(alert);
      setLauncherIcon(UIUtils.getLauncherIcon(alert, plugin.getTextColor(), hasMarketingChat));
    });
  }

  @Override
  protected void onDetachedFromWindow() {
    super.onDetachedFromWindow();

    hide();

    if (launcherConfigBinder != null) {
      launcherConfigBinder.unbind();
    }

    if (styleBinder != null) {
      styleBinder.unbind();
    }
  }

  public void show() {
    if (getVisibility() == View.GONE) {
      setVisibility(View.VISIBLE);

      buttonFloatingAction.startAnimation(launcherAnimation);
      layoutLauncherBadge.startAnimation(badgeAnimation);
    }
  }

  public void hide() {
    buttonFloatingAction.clearAnimation();
    layoutLauncherBadge.clearAnimation();

    setVisibility(View.GONE);
  }

  // private getter for layout

  private void setLauncherLayout(@Nullable ChannelButtonOption channelButtonOption) {
    RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);

    params.addRule(ALIGN_PARENT_BOTTOM);
    params.addRule(ALIGN_PARENT_TOP, 0);

    switch (getGravity(channelButtonOption)) {
      case BOTTOM_LEFT:
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
          params.addRule(ALIGN_PARENT_START);
          params.addRule(ALIGN_PARENT_END, 0);
        }
        params.addRule(ALIGN_PARENT_LEFT);
        params.addRule(ALIGN_PARENT_RIGHT, 0);
        params.setMargins(getXMargin(channelButtonOption), 0, 0, getYMargin(channelButtonOption));
        break;

      case BOTTOM_RIGHT:
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
          params.addRule(RelativeLayout.ALIGN_PARENT_START, 0);
          params.addRule(RelativeLayout.ALIGN_PARENT_END);
        }
        params.addRule(ALIGN_PARENT_LEFT, 0);
        params.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
        params.setMargins(0, 0, getXMargin(channelButtonOption), getYMargin(channelButtonOption));
        break;
    }

    setLayoutParams(params);
    requestLayout();
    textBadgeCount.bringToFront();
  }

  private int getXMargin(@Nullable ChannelButtonOption option) {
    if (option != null) {
      return (int) Utils.dpToPx(context, option.getXMargin());
    } else {
      return defaultMargin;
    }
  }

  private int getYMargin(@Nullable ChannelButtonOption option) {
    if (option != null) {
      return (int) Utils.dpToPx(context, option.getYMargin());
    } else {
      return defaultMargin;
    }
  }

  private int getGravity(@Nullable ChannelButtonOption option) {
    if (option != null) {
      return option.getLauncherGravity();
    } else {
      return BOTTOM_RIGHT;
    }
  }

  private void setLauncherBackgroundColor(int backgroundColor, int gradientColor) {
    buttonFloatingAction.setColorNormalWithGradient(backgroundColor, gradientColor);
    buttonFloatingAction.setColorPressed(Utils.addBlackMask(backgroundColor));
  }

  private void setBadgeCount(Integer chatCount) {
    if (chatCount == null || chatCount <= 0) {
      textBadgeCount.setVisibility(GONE);
    } else {
      textBadgeCount.setVisibility(VISIBLE);
      textBadgeCount.setText(Utils.getCount(chatCount, true));
    }
  }

  private void setLauncherIcon(int launcherIcon) {
    buttonFloatingAction.setImageResource(launcherIcon);
  }
}
