package com.zoyi.channel.plugin.android.view.coordinator;

import android.content.Context;
import android.graphics.*;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.design.widget.AppBarLayout;
import android.support.v4.view.ViewCompat;
import android.support.v4.view.WindowInsetsCompat;
import android.support.v4.widget.NestedScrollView;
import android.util.AttributeSet;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.selector.PluginSelector;
import com.zoyi.channel.plugin.android.store.binder.Binder;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.util.Views;

import java.util.ArrayList;
import java.util.List;

public class GradientHeaderCoordinatorLayout extends FrameLayout {

  static final String TAG = "GradientHeader";

  private Context context;

  @Nullable
  private AppBarLayout appBarLayout;
  @Nullable
  private NestedScrollView nestedScrollView;

  @Nullable
  private WindowInsetsCompat mLastInsets;

  @Nullable
  private android.support.v4.view.OnApplyWindowInsetsListener mApplyWindowInsetsListener;

  @Nullable
  OnNestedScrollChangeListener onNestedScrollChangeListener;

  private int themeColor = Color.BLACK;
  private int gradientColor = Color.WHITE;

  private List<View> paddingEffectiveViews = new ArrayList<>();

  @Nullable
  private Paint backgroundPaint;

  private int headerHeight = 0;

  @Nullable
  private Binder binder;

  public GradientHeaderCoordinatorLayout(@NonNull Context context) {
    super(context);
    init(context);
  }

  public GradientHeaderCoordinatorLayout(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    init(context);
  }

  public GradientHeaderCoordinatorLayout(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context);
  }

  @Initializer
  private void init(Context context) {
    this.context = context;

    LayoutInflater.from(context).inflate(R.layout.ch_plugin_layout_gradient_header_coordinator, this, true);

    if (Views.isFullscreenSupporting()) {
      setFitsSystemWindows(true);
      setupForInsets();
    }
  }

  /* inset */

  private void setupForInsets() {
    if (ViewCompat.getFitsSystemWindows(this)) {
      if (mApplyWindowInsetsListener == null) {
        mApplyWindowInsetsListener = (v, insets) -> {
          applyInsets(insets);
          return insets;
        };
      }
      // First apply the insets listener
      ViewCompat.setOnApplyWindowInsetsListener(this, mApplyWindowInsetsListener);

      // Now set the sys ui flags to enable us to lay out in the window insets
      setSystemUiVisibility(View.SYSTEM_UI_FLAG_LAYOUT_STABLE | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN);
    } else {
      ViewCompat.setOnApplyWindowInsetsListener(this, null);
    }
  }

  private void applyInsets(WindowInsetsCompat insets) {
    mLastInsets = insets;
    setPadding(0, 0, 0, 0);

    invalidate();
  }

  @Override
  public void setPadding(int left, int top, int right, int bottom) {
    if (mLastInsets != null) {
      super.setPadding(left, top + mLastInsets.getSystemWindowInsetTop(), right, bottom);
    } else {
      super.setPadding(left, top, right, bottom);
    }
  }

  /* attach, detach */

  @Override
  protected void onAttachedToWindow() {
    super.onAttachedToWindow();

    binder = PluginSelector.bindPlugin(plugin -> {
      if (plugin != null) {
        refreshColor(plugin.getBackgroundColor(), plugin.getGradientColor());
        invalidate();
      }
    });
  }

  @Override
  protected void onDetachedFromWindow() {
    super.onDetachedFromWindow();

    if (binder != null) {
      binder.unbind();
      binder = null;
    }

    paddingEffectiveViews.clear();
  }

  /* handle background */

  private void refreshColor(int themeColor, int gradientColor) {
    this.themeColor = themeColor;
    this.gradientColor = gradientColor;

    applyPaint();

    invalidate();
  }

  private void refreshHeaderHeight(int headerHeight) {
    this.headerHeight = headerHeight;

    applyPaint();
    applyContentPadding();

    invalidate();
  }

  protected int getBackgroundHeight() {
    return mLastInsets != null ? mLastInsets.getSystemWindowInsetTop() + headerHeight : headerHeight;
  }

  private void applyPaint() {
    applyPaint(getBackgroundHeight());
  }

  protected void applyPaint(int backgroundHeight) {
    if (backgroundHeight > 0) {
      LinearGradient backgroundGradient = new LinearGradient(
          0,
          0,
          this.getMeasuredWidth(),
          backgroundHeight,
          themeColor,
          gradientColor,
          Shader.TileMode.CLAMP
      );

      if (backgroundPaint == null) {
        backgroundPaint = new Paint();
      }
      backgroundPaint.clearShadowLayer();
      backgroundPaint.setShader(backgroundGradient);
    }
  }

  @Override
  protected void onDraw(Canvas canvas) {
    super.onDraw(canvas);
    drawBackground(canvas);
  }

  protected void drawBackground(Canvas canvas) {
    if (backgroundPaint != null) {
      canvas.drawRect(0, 0, getWidth(), getBackgroundHeight(), backgroundPaint);
    }
  }

  /* handle header */

  @Override
  public void addView(View child, int index, ViewGroup.LayoutParams params) {
    if (child == null) {
      return;
    }

    if (child.getId() == R.id.ch_rootGradientHeaderCoordinator) {
      super.addView(child, index, params);
      return;
    }

    if (child instanceof AppBarLayout) {
      if (appBarLayout == null) {
        appBarLayout = (AppBarLayout) child;
        appBarLayout.setBackgroundColor(Color.TRANSPARENT);
        appBarLayout.setFitsSystemWindows(false);
        appBarLayout.getViewTreeObserver().addOnGlobalLayoutListener(() -> {
          if (appBarLayout != null && appBarLayout.getHeight() != headerHeight) {
            refreshHeaderHeight(appBarLayout.getHeight());

            if (nestedScrollView != null && onNestedScrollChangeListener != null) {
              onNestedScrollChangeListener.onScrollChange(
                  nestedScrollView.getScrollY(),
                  appBarLayout.getHeight()
              );
            }
          }
        });
        super.addView(child, index, params);

        invalidate();
      } else {
        Log.e(TAG, "AppBarLayout can handle only one");
      }
    } else if (child instanceof NestedScrollView) {
      if (nestedScrollView == null) {
        nestedScrollView = (NestedScrollView) child;
        nestedScrollView.setOnScrollChangeListener((NestedScrollView.OnScrollChangeListener) (
            v,
            scrollX,
            scrollY,
            oldScrollX,
            oldScrollY
        ) -> {
          if (onNestedScrollChangeListener != null) {
            onNestedScrollChangeListener.onScrollChange(scrollY, headerHeight);
          }
        });

        super.addView(child, index, params);

        appendPaddingEffectiveViews(child);
      } else {
        Log.e(TAG, "NestedScrollView can handle only one");
      }
    } else if (child instanceof OverlayView) {
      super.addView(child, index, params);
    } else {
      FrameLayout frameLayout = new FrameLayout(context);
      frameLayout.addView(child, index, params);

      super.addView(frameLayout, index, new LayoutParams(-1, -1));

      appendPaddingEffectiveViews(frameLayout);
    }
  }

  private void appendPaddingEffectiveViews(View view) {
    paddingEffectiveViews.add(view);

    applyContentPadding();
  }

  private void applyContentPadding() {
    for (View view : paddingEffectiveViews) {
      view.setPadding(
          view.getPaddingLeft(),
          headerHeight,
          view.getPaddingRight(),
          view.getPaddingBottom()
      );
    }
  }

  public void setOnNestedScrollChangeListener(@Nullable OnNestedScrollChangeListener onNestedScrollChangeListener) {
    this.onNestedScrollChangeListener = onNestedScrollChangeListener;
  }
}
