package com.zoyi.channel.plugin.android.view.youtube.player.views;

import android.arch.lifecycle.Lifecycle;
import android.arch.lifecycle.LifecycleObserver;
import android.arch.lifecycle.OnLifecycleEvent;
import android.content.Context;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.support.annotation.LayoutRes;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.view.youtube.player.PlayerConstants;
import com.zoyi.channel.plugin.android.view.youtube.player.YouTubePlayer;
import com.zoyi.channel.plugin.android.view.youtube.player.listener.AbstractYouTubePlayerListener;
import com.zoyi.channel.plugin.android.view.youtube.player.listener.YouTubePlayerCallback;
import com.zoyi.channel.plugin.android.view.youtube.player.listener.YouTubePlayerFullScreenListener;
import com.zoyi.channel.plugin.android.view.youtube.player.listener.YouTubePlayerListener;
import com.zoyi.channel.plugin.android.view.youtube.player.option.IFramePlayerOptions;
import com.zoyi.channel.plugin.android.view.youtube.player.util.FullScreenHelper;
import com.zoyi.channel.plugin.android.view.youtube.player.util.NetworkListener;
import com.zoyi.channel.plugin.android.view.youtube.player.util.PlaybackResumer;
import com.zoyi.channel.plugin.android.view.youtube.ui.DefaultPlayerUiController;
import com.zoyi.channel.plugin.android.view.youtube.ui.PlayerUiController;
import com.zoyi.rx.functions.Action0;

import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

public class LegacyYouTubePlayerView extends SixteenByNineFrameLayout implements LifecycleObserver {

  public LegacyYouTubePlayerView(Context context) {
    super(context);
    init(context);
  }

  public LegacyYouTubePlayerView(Context context, @Nullable AttributeSet attrs) {
    super(context, null);
    init(context);
  }

  private WebViewYouTubePlayer youTubePlayer;
  private DefaultPlayerUiController defaultPlayerUiController;

  private NetworkListener networkListener = new NetworkListener();
  private PlaybackResumer playbackResumer = new PlaybackResumer();
  private FullScreenHelper fullScreenHelper = new FullScreenHelper(this);

  private boolean isYouTubePlayerReady = false;

  @Nullable
  private Action0 initialize;

  private Set<YouTubePlayerCallback> youTubePlayerCallbacks;

  private boolean canPlay = true;
  private boolean isUsingCustomUi = false;

  public boolean isCanPlay() {
    return canPlay;
  }

  @Initializer
  private void init(Context context) {
    youTubePlayer = new WebViewYouTubePlayer(context);

    networkListener = new NetworkListener();
    playbackResumer = new PlaybackResumer();
    fullScreenHelper = new FullScreenHelper(this);

    youTubePlayerCallbacks = new HashSet<>();

    addView(youTubePlayer, new FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT));
    defaultPlayerUiController = new DefaultPlayerUiController(this, youTubePlayer);

    fullScreenHelper.addFullScreenListener(defaultPlayerUiController);

    youTubePlayer.addListener(defaultPlayerUiController);
    youTubePlayer.addListener(playbackResumer);

    // stop playing if the user loads a video but then leaves the app before the video starts playing.
    youTubePlayer.addListener(new AbstractYouTubePlayerListener() {
      @Override
      public void onStateChange(YouTubePlayer youTubePlayer, PlayerConstants.PlayerState state) {
        if (state == PlayerConstants.PlayerState.PLAYING && !isEligibleForPlayback()) {
          youTubePlayer.pause();
        }
      }
    });

    youTubePlayer.addListener(new AbstractYouTubePlayerListener() {
      @Override
      public void onReady(YouTubePlayer youTubePlayer) {
        isYouTubePlayerReady = true;

        for (YouTubePlayerCallback callback : youTubePlayerCallbacks) {
          callback.onYouTubePlayer(youTubePlayer);

        }
        youTubePlayerCallbacks.clear();

        youTubePlayer.removeListener(this);
      }
    });

    networkListener.setOnNetworkAvailable(() -> {
      if (!isYouTubePlayerReady && initialize != null) {
        initialize.call();
      } else {
        playbackResumer.resume(youTubePlayer);
      }
    });
  }

  public WebViewYouTubePlayer getYouTubePlayer() {
    return youTubePlayer;
  }

  /**
   * Initialize the player. You must call this method before using the player.
   *
   * @param youTubePlayerListener listener for player events
   * @param handleNetworkEvents   if set to true a broadcast receiver will be registered and network events will be handled automatically.
   *                              If set to false, you should handle network events with your own broadcast receiver.
   * @param playerOptions         customizable options for the embedded video player, can be null.
   */
  public void initialize(Context context, YouTubePlayerListener youTubePlayerListener, boolean handleNetworkEvents,
      @Nullable IFramePlayerOptions playerOptions) {
    if (isYouTubePlayerReady) {
      throw new IllegalStateException("This YouTubePlayerView has already been initialized.");
    }

    if (handleNetworkEvents) {
      context.registerReceiver(networkListener, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));
    }

    initialize = () -> {
      try {
        youTubePlayer.initialize(youTubePlayer -> youTubePlayer.addListener(youTubePlayerListener), playerOptions);
      } catch (IOException e) {
        e.printStackTrace();
      }
    };

    if (!handleNetworkEvents) {
      initialize.call();
    }
  }

  /**
   * Initialize the player.
   *
   * @param handleNetworkEvents if set to true a broadcast receiver will be registered and network events will be handled automatically. If
   *                            set to false, you should handle network events with your own broadcast receiver.
   * @see LegacyYouTubePlayerView.initialize
   */
  public void initialize(Context context, YouTubePlayerListener youTubePlayerListener, boolean handleNetworkEvents) {
    initialize(context, youTubePlayerListener, handleNetworkEvents, null);
  }

  /**
   * Initialize the player. Network events are automatically handled by the player.
   *
   * @param youTubePlayerListener listener for player events
   * @see LegacyYouTubePlayerView.initialize
   */
  public void initialize(Context context, YouTubePlayerListener youTubePlayerListener) {
    initialize(context, youTubePlayerListener, true);
  }

  /**
   * Initialize a player using the web-base Ui instead pf the native Ui. The default PlayerUiController will be removed and
   * [LegacyYouTubePlayerView.getPlayerUiController] will throw exception.
   *
   * @see LegacyYouTubePlayerView.initialize
   */
  public void initializeWithWebUi(Context context, YouTubePlayerListener youTubePlayerListener, boolean handleNetworkEvents) {
    IFramePlayerOptions iFramePlayerOptions = new IFramePlayerOptions.Builder().controls(1).build();
    inflateCustomPlayerUi(context, R.layout.ch_plugin_ayp_empty_layout);
    initialize(context, youTubePlayerListener, handleNetworkEvents, iFramePlayerOptions);
  }

  /**
   * @param youTubePlayerCallback A callback that will be called when the YouTubePlayer is ready. If the player is ready when the function
   *                              is called, the callback is called immediately. This function is called only once.
   */
  public void getYouTubePlayerWhenReady(YouTubePlayerCallback youTubePlayerCallback) {
    if (isYouTubePlayerReady) {
      youTubePlayerCallback.onYouTubePlayer(youTubePlayer);
    } else {
      youTubePlayerCallbacks.add(youTubePlayerCallback);
    }
  }

  /**
   * Use this method to replace the default Ui of the player with a custom Ui.
   * <p>
   * You will be responsible to manage the custom Ui from your application, the default controller obtained through
   * [LegacyYouTubePlayerView.getPlayerUiController] won't be available anymore.
   *
   * @param layoutId the ID of the layout defining the custom Ui.
   * @return The inflated View
   */
  public View inflateCustomPlayerUi(Context context, @LayoutRes int layoutId) {
    removeViews(1, getChildCount() - 1);

    if (!isUsingCustomUi) {
      youTubePlayer.removeListener(defaultPlayerUiController);
      fullScreenHelper.removeFullScreenListener(defaultPlayerUiController);
    }

    isUsingCustomUi = true;

    return View.inflate(context, layoutId, this);
  }

  /**
   * Call this method before destroying the host Fragment/Activity, or register this View as an observer of its host lifecycle
   */
  @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
  public void release() {
    removeView(youTubePlayer);
    youTubePlayer.removeAllViews();
    youTubePlayer.destroy();
    try {
      getContext().unregisterReceiver(networkListener);
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  @OnLifecycleEvent(Lifecycle.Event.ON_RESUME)
  public void onResume() {
    playbackResumer.onLifecycleResume();
    canPlay = true;
  }

  @OnLifecycleEvent(Lifecycle.Event.ON_STOP)
  public void onStop() {
    youTubePlayer.pause();
    playbackResumer.onLifecycleStop();
    canPlay = false;
  }

  /**
   * Checks whether the player is in an eligible state for playback in respect of the {@link WebViewYouTubePlayer#isBackgroundPlaybackEnabled}
   * property.
   */
  public boolean isEligibleForPlayback() {
    return canPlay || youTubePlayer.isBackgroundPlaybackEnabled();
  }

  /**
   * Don't use this method if you want to publish your app on the PlayStore. Background playback is against YouTube terms of service.
   */
  public void enableBackgroundPlayback(boolean enable) {
    youTubePlayer.setBackgroundPlaybackEnabled(enable);
  }

  public PlayerUiController getPlayerUiController() {
    if (isUsingCustomUi) {
      throw new RuntimeException("You have inflated a custom player Ui. You must manage it with your own controller.");
    }

    return defaultPlayerUiController;
  }

  public void enterFullScreen() {
    fullScreenHelper.enterFullScreen();
  }

  public void exitFullScreen() {
    fullScreenHelper.exitFullScreen();
  }

  public void toggleFullScreen() {
    fullScreenHelper.toggleFullScreen();
  }

  public boolean isFullScreen() {
    return fullScreenHelper.isFullScreen();
  }

  public boolean addFullScreenListener(YouTubePlayerFullScreenListener fullScreenListener) {
    return fullScreenHelper.addFullScreenListener(fullScreenListener);
  }


  public boolean removeFullScreenListener(YouTubePlayerFullScreenListener fullScreenListener) {
    return fullScreenHelper.removeFullScreenListener(fullScreenListener);
  }
}
