package com.zoyi.channel.plugin.android.util;

import android.content.Context;
import android.support.annotation.Nullable;
import android.telephony.TelephonyManager;
import android.util.Pair;

import com.zoyi.channel.plugin.android.deserializer.*;
import com.zoyi.channel.plugin.android.model.rest.Message;
import com.zoyi.channel.plugin.android.model.rest.NestedMessage;
import com.zoyi.channel.plugin.android.model.wrapper.TranslationRepo;
import com.zoyi.com.annimon.stream.*;
import com.zoyi.com.google.gson.*;
import com.zoyi.com.google.i18n.phonenumbers.PhoneNumberUtil;

import java.util.Map;

/**
 * Created by jerry on 2018. 3. 27..
 */

public class ParseUtils {

  @Nullable
  private static Gson gson;

  public static Gson getCustomGson() {
    if (gson == null) {
      GsonBuilder builder = new GsonBuilder();
      builder.registerTypeAdapter(Map.class, new AnyMapDeserializer());
      builder.registerTypeAdapter(Message.class, new MessageDeserializer());
      builder.registerTypeAdapter(NestedMessage.class, new NestedMessageDeserializer());
      builder.registerTypeAdapter(TranslationRepo.class, new TranslationRepoDeserializer());

      gson = builder.create();
    }

    return gson;
  }

  @Nullable
  public static <T> T parseJson(String entity, Class<T> target) {
    try {
      return getCustomGson().fromJson(entity, target);
    } catch (Exception ex) {
      L.e(ex.getMessage());
    }
    return null;
  }

  @Nullable
  public static String parsePhoneNumber(Context context, @Nullable String phoneNumber) {
    try {
      PhoneNumberUtil phoneNumberUtil = PhoneNumberUtil.getInstance();

      Integer simCountryCode = Optional.ofNullable(context.getSystemService(Context.TELEPHONY_SERVICE))
          .map(service -> (TelephonyManager) service)
          .map(TelephonyManager::getSimCountryIso)
          .map(String::toUpperCase)
          .map(phoneNumberUtil::getCountryCodeForRegion)
          .filter(it -> it != 0)
          .orElse(null);

      return Optional.ofNullable(phoneNumberUtil.parse(phoneNumber, "ZZ"))
          .filter(number -> number.hasCountryCode() && simCountryCode != null && number.getCountryCode() == simCountryCode)
          .map(number -> phoneNumberUtil.format(number, PhoneNumberUtil.PhoneNumberFormat.NATIONAL))
          .orElse(phoneNumber);

    } catch (Exception ex) {
      return phoneNumber;
    }
  }

  @Nullable
  public static Object parseJsonElement(JsonElement json) {
    try {
      if (json.isJsonArray()) {
        return Stream.ofNullable(json.getAsJsonArray()).map(ParseUtils::parseJsonElement).collect(Collectors.toList());
      } else if (json.isJsonObject()) {
        return Stream.ofNullable(json.getAsJsonObject().entrySet())
            .map(entry -> new Pair<>(entry.getKey(), ParseUtils.parseJsonElement(entry.getValue())))
            .collect(Collectors.toMap(pair -> pair.first, pair -> pair.second));
      } else if (json.isJsonPrimitive()) {
        JsonPrimitive jsonPrimitive = json.getAsJsonPrimitive();

        if (jsonPrimitive.isBoolean()) {
          return jsonPrimitive.getAsBoolean();
        } else if (jsonPrimitive.isString()) {
          return jsonPrimitive.getAsString();
        } else if (jsonPrimitive.isNumber()) {
          Long tryLong = ParseUtils.parseLong(jsonPrimitive.getAsString());

          if (tryLong != null) {
            return tryLong;
          }
          return jsonPrimitive.getAsDouble();
        } else {
          return null;
        }
      } else {
        return null;
      }
    } catch (Exception ex) {
      return null;
    }
  }

  @Nullable
  public static Long parseLong(@Nullable Object object) {
    if (object == null) {
      return null;
    }
    if (object instanceof Long) {
      return (Long) object;
    }

    try {
      String[] split = object.toString().split("\\.");

      if (split.length == 1) {
        return Long.valueOf(split[0]);
      } else if (split.length == 2) {
        for (int i = 0; i < split[1].length(); i++) {
          if (split[1].charAt(i) != '0') {
            return null;
          }
        }
        return Long.valueOf(split[0]);
      }
      return null;
    } catch (Exception ex) {
      return null;
    }
  }

  @Nullable
  public static Object parseNumber(@Nullable String string) {
    if (string != null) {
      try {
        return Integer.parseInt(string);
      } catch (Exception ex) {
      }

      try {
        return Double.parseDouble(string);
      } catch (Exception ex) {
      }
    }
    return null;
  }
}
