package com.zoyi.channel.plugin.android.activity.chat;


import android.app.Activity;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.action.*;
import com.zoyi.channel.plugin.android.activity.chat.contract.ChatAdapterContract;
import com.zoyi.channel.plugin.android.activity.chat.contract.ChatContract;
import com.zoyi.channel.plugin.android.activity.chat.listener.MessageSendListener;
import com.zoyi.channel.plugin.android.activity.chat.manager.ChatManager;
import com.zoyi.channel.plugin.android.activity.chat.manager.ChatManagerInterface;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.activity.common.chat.ChatContentType;
import com.zoyi.channel.plugin.android.base.AbstractAdapterPresenter;
import com.zoyi.channel.plugin.android.enumerate.*;
import com.zoyi.channel.plugin.android.global.*;
import com.zoyi.channel.plugin.android.model.entity.ProfileEntity;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.source.photopicker.PhotoItem;
import com.zoyi.channel.plugin.android.model.wrapper.MessagesWrapper;
import com.zoyi.channel.plugin.android.model.wrapper.UserChatWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.selector.*;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store.*;
import com.zoyi.channel.plugin.android.store.binder.Binder;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.TimeUtils;
import com.zoyi.com.annimon.stream.Optional;
import com.zoyi.com.annimon.stream.Stream;
import com.zoyi.com.annimon.stream.function.IndexedFunction;
import com.zoyi.rx.Subscription;

import java.util.*;

import static com.zoyi.channel.plugin.android.global.Const.ACTION_TYPE_CLOSE;
import static com.zoyi.channel.plugin.android.global.Const.ACTION_TYPE_SOLVE;

public class ChatPresenter
    extends AbstractAdapterPresenter<ChatContract.View, ChatAdapterContract.View, ChatAdapterContract.Model>
    implements ChatContract.Presenter, MessageSendListener {

  private ChatContentType contentType;
  @Nullable
  private String contentId;

  @Nullable
  private Binder welcomeBinder;

  @Nullable
  private Binder welcomeAcceptInputBinder;

  @Nullable
  private Subscription joinSubscription;

  @Nullable
  private Subscription messageSubscription;

  @Nullable
  private Session lastReadSession;

  @Nullable
  private List<SendItem> tempItems;

  // queued message when message initializing
  private List<Message> queueItems = new ArrayList<>();

  @Nullable
  private String backwardId;

  private boolean init = false;

  @Nullable
  private Long lastUserChatUpdatedAt = null;

  public ChatPresenter(
      ChatContract.View view,
      ChatAdapterContract.View adapterView,
      ChatAdapterContract.Model adapterModel,
      ChatContentType contentType,
      @Nullable String contentId
  ) {
    super(view, adapterView, adapterModel);

    TimeUtils.refreshOffset();

    this.contentType = contentType;
    this.contentId = contentId;
  }

  private ChatManagerInterface getChatManager() {
    return ChatManager.get(contentType == ChatContentType.USER_CHAT ? contentId : null);
  }

  @Override
  public void init() {
    switch (contentType) {
      case USER_CHAT:
        view.onFetchStateChange(FetchState.LOADING);

        if (contentId != null) {
          bindUserChatEventReceiver();
        }

        break;

      case NONE:
        this.tempItems = new ArrayList<>();

        view.onFetchStateChange(FetchState.LOADING);

        Plugin plugin = PluginStore.get().pluginState.get();
        if (plugin != null && plugin.getId() != null) {
          bindWelcomeMessage();
        }
        break;
    }
  }

  @Override
  protected void handleAction(ActionType actionType) {
    switch (actionType) {
      case SOCKET_DISCONNECTED:
      case SHUTDOWN:
        this.init = false;
        this.backwardId = null;
        this.queueItems.clear();
        break;
    }
  }

  // welcome message, logic

  private void bindWelcomeMessage() {
    welcomeBinder = ChatSelector.bindWelcomeMessage((fetchState, supportBotEntry, welcomeMessage) -> {
      switch (fetchState) {
        case FAILED:
          view.onFetchStateChange(FetchState.FAILED);
          break;

        case COMPLETE:
          if (supportBotEntry != null) {
            this.contentType = ChatContentType.SUPPORT_BOT_CHAT;
            this.contentId = supportBotEntry.getId();

            adapterModel.setInitMessage(new SupportBotMessageItem(supportBotEntry));

            view.onWelcomeStateChange(ChatContentType.SUPPORT_BOT_CHAT, false);
            view.onFetchStateChange(FetchState.COMPLETE);

          } else if (welcomeMessage != null) {
            adapterModel.setInitMessage(new WelcomeMessageItem(welcomeMessage));

            bindWelcomeAcceptInput();
            view.onFetchStateChange(FetchState.COMPLETE);
          } else {
            view.onFetchStateChange(FetchState.FAILED);
          }

          if (welcomeBinder != null) {
            welcomeBinder.unbind();
          }
          break;
      }
    });
  }

  private void bindWelcomeAcceptInput() {
    if (welcomeAcceptInputBinder == null) {
      welcomeAcceptInputBinder = ChannelSelector.bindAcceptInput(acceptInput ->
          view.onWelcomeStateChange(ChatContentType.NONE, acceptInput)
      );
    }
  }

  // chat init logic for normal chat

  private void createUserChat() {
    if (contentType == ChatContentType.NONE && !Api.isRunning(ActionType.CREATE_USER_CHAT)) {
      Plugin plugin = PluginStore.get().pluginState.get();
      String pluginId = plugin != null ? plugin.getId() : null;

      Activity topActivity = GlobalStore.get().topActivity.get();
      String url = Optional.ofNullable(topActivity).map(Activity::getClass).map(Class::getSimpleName).orElse("");

      if (pluginId != null) {
        ChatAction.createUserChat(pluginId, url, new RestSubscriber<UserChatWrapper>() {
          @Override
          public void onError(RetrofitException error) {
            setTempSendingItemsFail();
          }

          @Override
          public void onSuccess(@NonNull UserChatWrapper repo) {
            handleUserChatInit(repo.getUserChat(), repo.getMessage());
          }
        });
      }
    }
  }

  // chat init logic for support bot

  private void createSupportBotUserChat(SendActionItem item) {
    if (contentType == ChatContentType.SUPPORT_BOT_CHAT && !Api.isRunning(ActionType.CREATE_USER_CHAT)) {
      String supportBotId = contentId;

      String url = Optional.ofNullable(GlobalStore.get().topActivity.get())
          .map(Activity::getClass)
          .map(Class::getSimpleName)
          .orElse("");

      addTempSendItem(item);

      SupportBotAction.createSupportBotUserChat(supportBotId, url, new RestSubscriber<UserChatWrapper>() {
        @Override
        public void onError(RetrofitException error) {
          removeTempSendingItem(item);
        }

        @Override
        public void onSuccess(@NonNull UserChatWrapper repo) {
          item.updateMessageOnActionInput(repo.getMessage());

          handleUserChatInit(repo.getUserChat(), repo.getMessage());
        }
      });
    }
  }

  // Add temporary sending items before chat created

  private void addTempSendItem(SendItem item) {
    if (item != null) {
      addTempSendItems(Collections.singleton(item));
    }
  }

  private void addTempSendItems(Collection<SendItem> items) {
    if (tempItems != null && items != null && items.size() > 0) {
      adapterModel.addMessageItems(items);
      tempItems.addAll(items);
    }
  }

  private boolean removeTempSendingItem(SendItem item) {
    if (tempItems != null && item != null) {
      adapterModel.removeMessageItem(item);
      return tempItems.remove(item);
    }
    return false;
  }

  private void setTempSendingItemsChatId(String chatId) {
    if (tempItems != null) {
      for (SendItem item : tempItems) {
        item.setChatId(chatId);
      }
      adapterModel.addMessageItems(tempItems);
    }
  }

  private void setTempSendingItemsFail() {
    if (tempItems != null) {
      for (SendItem item : tempItems) {
        item.doFail();
      }
      adapterModel.addMessageItems(tempItems);
    }
  }

  // Call when user chat create by user manually

  private void handleUserChatInit(UserChat userChat, Message message) {
    if (welcomeBinder != null) {
      welcomeBinder.unbind();
      welcomeBinder = null;
    }
    if (welcomeAcceptInputBinder != null) {
      welcomeAcceptInputBinder.unbind();
      welcomeAcceptInputBinder = null;
    }
    if (userChat != null && userChat.getId() != null) {
      this.contentType = ChatContentType.USER_CHAT;
      this.contentId = userChat.getId();

      adapterModel.replaceInitMessageItem(message);

      // set chat id to pre-sending items
      setTempSendingItemsChatId(userChat.getId());

      bindUserChatEventReceiver();
    }
  }

  // user chat logic

  private void bindUserChatEventReceiver() {
    // subscribe only once

    if (joinSubscription == null && contentType == ChatContentType.USER_CHAT && contentId != null) {
      getChatManager().attachListener(this);

      bind(SocketSelector.bindSocket(socketStatus -> {
        if (socketStatus == SocketStatus.READY) {
          fetchUserChat();
        }
      }));

      bind(ChatSelector.bindChat(contentId, (userChat, recentMessage, acceptInput, temporaryInputOpened, isWorking) -> {
        // when after select 'keep going chatting' and chat state changed, we recognize who send message after that situation.

        if (temporaryInputOpened && !Const.USER_CHAT_STATE_SOLVED.equals(userChat.getState())) {
          ChatStore.get().temporaryInputOpenedState.set(false);
        }

        if (lastUserChatUpdatedAt == null || (userChat.getUpdatedAt() != null && lastUserChatUpdatedAt < userChat.getUpdatedAt())) {
          lastUserChatUpdatedAt = userChat.getUpdatedAt();
          ChatAction.read(contentId);
        }

        view.onChatStateChange(userChat, recentMessage, acceptInput, temporaryInputOpened, isWorking);
      }));

      bind(ProfileBotSelector.bindProfileBotActivation(isActivated -> {
        if (isActivated) {
          view.setInputDim(true);
        }
      }));

      bind(ChatSelector.bindTyping(isTyping -> SocketManager.typing(
          Typing.create(
              isTyping ? Const.TYPING_START : Const.TYPING_STOP,
              contentId
          )
      )));

      bind(ChatSelector.bindHostTyping(contentId, typings -> adapterModel.setTypings(typings)));

      this.joinSubscription = SocketManager.observable().subscribe(joinedChatId -> {
        if (contentId != null && contentId.equals(joinedChatId)) {
          fetchInitMessages();
        }
      });

      this.messageSubscription = RxBus.subscribe(o -> {
        if (o instanceof Message) {
          Message message = (Message) o;

          if (contentId != null && CompareUtils.isSame(message.getChatId(), contentId)) {
            onReceiveMessage(message);
          }
        }
      });
    }
  }

  // data fetch

  private void fetchUserChat() {
    ChatAction.fetchUserChat(contentId, new RestSubscriber<UserChatWrapper>() {
      @Override
      public void onError(RetrofitException error) {
        view.onFetchStateChange(FetchState.FAILED);
      }

      @Override
      public void onSuccess(@NonNull UserChatWrapper repo) {
        lastReadSession = repo.getSession();
        queueItems.clear();

        SocketManager.joinChat(repo.getUserChat().getId());
      }
    });
  }

  private void fetchInitMessages() {
    ChatAction.fetchMessages(contentId, new RestSubscriber<MessagesWrapper>() {
      @Override
      public void onError(RetrofitException error) {
        SocketManager.leaveChat(contentId);
        view.onFetchStateChange(FetchState.FAILED);
      }

      @Override
      public void onSuccess(@NonNull MessagesWrapper repo) {
        ArrayList<Message> messages = new ArrayList<>(repo.getMessages());

        init = true;
        backwardId = repo.getNext();

        // if not sent message exists before create user chat, send it
        if (tempItems != null && tempItems.size() > 0) {
          getChatManager().sendMessages(tempItems);

          tempItems.clear();
          tempItems = null;
        }

        if (queueItems.size() > 0) {
          messages.addAll(queueItems);

          queueItems.clear();
        }

        adapterModel.setMessages(messages, getChatManager().getUnsentItems(), lastReadSession, repo.getNext());

        view.scrollToBottom();
        view.onFetchStateChange(FetchState.COMPLETE);

        ChatAction.read(contentId);
      }
    });
  }

  @Override
  public void fetchBackwardMessages() {
    if (!Api.isRunning(ActionType.FETCH_BACKWARD_MESSAGES) && backwardId != null) {
      ChatAction.fetchBackwardMessages(contentId, backwardId, new RestSubscriber<MessagesWrapper>() {
        @Override
        public void onSuccess(@NonNull MessagesWrapper repo) {
          adapterModel.addMessages(repo.getMessages(), lastReadSession, repo.getNext());
          backwardId = repo.getNext();
        }
      });
    }
  }

  // handle action

  @Override
  public void sendText(String message) {
    if (tempItems != null) {
      addTempSendItem(new SendTextItem(null, message));
      createUserChat();
    } else if (contentType == ChatContentType.USER_CHAT && contentId != null) {
      getChatManager().sendMessage(new SendTextItem(contentId, message));
    }

    view.scrollToBottom();
  }

  public void uploadFiles(List<PhotoItem> photoItems) {
    if (photoItems != null) {
      List<SendItem> items = Stream.of(photoItems)
          .mapIndexed((IndexedFunction<PhotoItem, SendItem>) (index, item) ->
              new SendFileItem(contentId, index, item.getUri(), item.getName(), item.getSize())
          )
          .toList();

      if (tempItems != null) {
        addTempSendItems(items);
        createUserChat();
      } else if (contentType == ChatContentType.USER_CHAT && contentId != null) {
        getChatManager().sendMessages(items);
      }

      view.scrollToBottom();
    }
  }

  @Override
  public void onActionClick(@NonNull String actionType, @NonNull ActionButton actionButton) {
    if (contentType == ChatContentType.SUPPORT_BOT_CHAT && Const.ACTION_TYPE_SUPPORT_BOT.equals(actionType)) {
      createSupportBotUserChat(new SendActionItem(null, actionType, actionButton));
      return;
    }

    if (CompareUtils.exists(actionType, ACTION_TYPE_SOLVE, ACTION_TYPE_CLOSE) &&
        CompareUtils.isSame(actionButton.getKey(), Const.ACTION_KEY_REOPEN)
    ) {
      reopenChat(actionButton);
      return;
    }

    if (contentType == ChatContentType.USER_CHAT && contentId != null) {
      getChatManager().sendMessage(new SendActionItem(contentId, actionType, actionButton));
    }

    view.scrollToBottom();
  }

  @Override
  public void onCreateSupportBotClick() {
    SupportBotAction.createMarketingSupportBotUserChat(contentId);
  }

  @Override
  public void resend(SendItem item) {
    if (tempItems != null && removeTempSendingItem(item)) {
      item.renew();
      tempItems.add(item);

      createUserChat();
    } else if (contentType == ChatContentType.USER_CHAT) {
      getChatManager().resend(item);
    }

    view.scrollToBottom();
  }

  @Override
  public void removeFailedItem(SendItem item) {
    if (tempItems == null || !removeTempSendingItem(item)) {
      getChatManager().remove(item);
    }
  }

  @Override
  public void cancelSendingFile(SendFileItem item) {
    getChatManager().cancelRecentSendingFile();
  }

  // profile bot action

  @Override
  public void updateProfile(ProfileBotMessageItem item, String key, @Nullable Object value) {
    ProfileBotStore.get().inputKey.set(key);
    ProfileBotStore.get().inputValue.set(value);
    ProfileBotStore.get().inputError.set(null);
    ProfileBotStore.get().requestState.set(FetchState.LOADING);

    adapterModel.addMessageItem(item);

    UserAction.updateProfileBot(
        item.getUserChatId(),
        item.getMessageId(),
        key,
        value,
        repo -> {
          ProfileBotStore.get().requestFocus.set(true);
          ProfileBotStore.get().requestState.set(FetchState.COMPLETE);
          ProfileBotStore.get().inputKey.set(null);
          ProfileBotStore.get().inputValue.set(null);
          ProfileBotStore.get().inputError.set(null);

          if (init) {
            adapterModel.addMessage(repo.getMessage());
          }
        }, (errorMessage) -> {
          ProfileBotStore.get().inputError.set(errorMessage);
          ProfileBotStore.get().requestFocus.set(true);
          ProfileBotStore.get().requestState.set(FetchState.FAILED);

          adapterModel.addMessageItem(item);
        }
    );
  }

  // private functions

  private void reopenChat(@NonNull ActionButton actionButton) {
    Message message = actionButton.getMessage();
    message.clearAction();
    adapterModel.addMessage(message);

    ChatStore.get().temporaryInputOpenedState.set(true);

    view.scrollToBottom();
  }

  // release

  @Override
  public void release() {
    super.release();

    getChatManager().clearListener();

    if (welcomeBinder != null) {
      welcomeBinder.unbind();
      welcomeBinder = null;
    }

    if (welcomeAcceptInputBinder != null) {
      welcomeAcceptInputBinder.unbind();
      welcomeAcceptInputBinder = null;
    }

    if (joinSubscription != null && !joinSubscription.isUnsubscribed()) {
      joinSubscription.unsubscribe();
      joinSubscription = null;
    }

    if (messageSubscription != null && !messageSubscription.isUnsubscribed()) {
      messageSubscription.unsubscribe();
      messageSubscription = null;
    }

    if (contentType == ChatContentType.USER_CHAT && contentId != null) {
      SocketManager.leaveChat(contentId);
    }

    TypingStore.get().myTypingState.upsert(Typing.dummy());

    if (contentType == ChatContentType.USER_CHAT && contentId != null) {
      SocketManager.typing(Typing.create(Const.TYPING_STOP, contentId));
    }
  }

  //region ChatManager callbacks

  @Override
  public void onMessageItemUpsert(MessageItem item) {
    adapterModel.addMessageItem(item);
  }

  @Override
  public void onMessageItemRemove(MessageItem item) {
    adapterModel.removeMessageItem(item);
  }

  @Override
  public void onMessageSendSuccess(Message message) {
    onReceiveMessage(message);
  }

  //endregion

  // message receive

  private void onReceiveMessage(Message message) {
    if (init) {
      boolean isScrollOnBottom = view.isScrollOnBottom();

      adapterModel.addMessage(message);

      if (isScrollOnBottom) {
        view.scrollToBottom();
      } else if (!CompareUtils.isSame(message.getPersonType(), Const.USER)) {
        // Show new message alert in bottom

        ProfileEntity profileEntity = ProfileSelector.getProfile(message.getPersonType(), message.getPersonId());

        if (profileEntity != null) {
          view.showNewMessageAlert(profileEntity);
        }
      }
    } else if (Api.isRunning(ActionType.FETCH_MESSAGES)) {
      queueItems.add(message);
    }
  }
}
