package com.zoyi.channel.plugin.android;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.support.annotation.*;
import android.text.TextUtils;

import com.zoyi.channel.plugin.android.action.*;
import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.global.Action;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.rest.Event;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.open.callback.TagUpdateCallback;
import com.zoyi.channel.plugin.android.open.callback.UserUpdateCallback;
import com.zoyi.channel.plugin.android.open.exception.ChannelException;
import com.zoyi.channel.plugin.android.open.model.UserData;
import com.zoyi.channel.plugin.android.push.ChannelPushClient;
import com.zoyi.channel.plugin.android.push.ChannelPushManager;
import com.zoyi.channel.plugin.android.selector.GlobalSelector;
import com.zoyi.channel.plugin.android.store.PluginStore;
import com.zoyi.channel.plugin.android.store.SettingsStore;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.ListUtils;

import java.util.List;
import java.util.Map;

public class ChannelIO {

  private static boolean isDebugMode = !BuildConfig.IS_PRODUCTION_DEPLOYMENT;

  @Nullable
  private static RealChannelIO realChannelIO;

  // Main Method

  public static void initialize(@Nullable Application application) {
    if (application == null) {
      L.e("Fail to 'initialize', Application can't be NULL");
    } else if (realChannelIO != null) {
      L.e("Fail to 'initialize', Channel plugin already initialized");
    } else {
      realChannelIO = new RealChannelIO(application);
    }
  }

  public static void setDebugMode(boolean enable) {
    isDebugMode = enable;
  }

  // boot functions

  public static void boot(@Nullable ChannelPluginSettings pluginSettings) {
    boot(pluginSettings, null, null);
  }

  public static void boot(@Nullable ChannelPluginSettings pluginSettings, @Nullable OnBootListener completion) {
    boot(pluginSettings, null, completion);
  }

  public static void boot(@Nullable ChannelPluginSettings pluginSettings, @Nullable Profile profile) {
    boot(pluginSettings, profile, null);
  }

  public static void boot(@Nullable ChannelPluginSettings pluginSettings, @Nullable Profile profile, @Nullable OnBootListener completion) {
    if (realChannelIO == null) {
      L.e("Fail to 'boot', Initialize first");

      if (completion != null) {
        completion.onCompletion(ChannelPluginCompletionStatus.NOT_INITIALIZED, null);
      }
    } else if (pluginSettings == null || pluginSettings.getPluginKey() == null) {
      L.e("Fail to 'boot', Check plugin information");

      if (completion != null) {
        completion.onCompletion(ChannelPluginCompletionStatus.NOT_INITIALIZED, null);
      }
    } else {
      realChannelIO.boot(pluginSettings, profile, completion);
    }
  }

  // shutdown function

  public static void shutdown() {
    if (realChannelIO != null) {
      realChannelIO.shutdown();
    }
  }

  // open

  public static boolean open(@Nullable Context context) {
    return open(context, true);
  }

  @Deprecated
  public static boolean open(@Nullable Context context, boolean animate) {
    return openChat(context, null, animate);
  }

  // close

  public static void close() {
    Action.invoke(ActionType.EXIT);
  }

  @Deprecated
  public static void close(boolean animated) {
    close();
  }

  // open chat

  public static boolean openChat(@Nullable Context context) {
    return openChat(context, null);
  }

  public static boolean openChat(@Nullable Context context, @Nullable String chatId) {
    return openChat(context, chatId, true);
  }

  @Deprecated
  public static boolean openChat(@Nullable Context context, @Nullable String chatId, boolean animate) {
    if (realChannelIO != null) {
      return realChannelIO.startMessenger(context, chatId, animate);
    }
    return false;
  }

  // show

  public static void show() {
    if (!SettingsStore.get().showLauncher.get()) {
      SettingsStore.get().showLauncher.set(true);

      if (isBooted() && SettingsStore.get().enabledTrackDefaultEvent.get()) {
        EventAction.trackPageView();
      }
    }
  }

  // hide

  public static void hide() {
    SettingsStore.get().showLauncher.set(false);
  }

  // canShowLauncher

  public static boolean canShowLauncher() {
    return GlobalSelector.getLauncherVisibility();
  }

  // setChannelPluginListener

  public static void setChannelPluginListener(ChannelPluginListener channelPluginListener) {
    if (realChannelIO != null) {
      realChannelIO.setListener(channelPluginListener);
    }
  }

  // clearChannelPluginListener

  public static void clearChannelPluginListener() {
    if (realChannelIO != null) {
      realChannelIO.clearListener();
    }
  }

  @Nullable
  public static ChannelPluginListener getChannelPluginListener() {
    if (realChannelIO != null) {
      return realChannelIO.getListener();
    }
    return null;
  }

  // track

  public static void track(@NonNull @Size(min = 1L, max = 30L) String eventName) {
    track(eventName, null);
  }

  public static void track(@NonNull @Size(min = 1L, max = 30L) String eventName, @Nullable Map<String, Object> eventProperty) {
    if (TextUtils.isEmpty(eventName)) {
      L.e("Fail to track event. Event name can't be blank or null.");
      return;
    }

    if (eventName.length() > 30) {
      L.e("Fail to track event. Event name must be 30 characters or less.");
      return;
    }

    Plugin plugin = PluginStore.get().pluginState.get();

    if (plugin == null || plugin.getId() == null) {
      L.e("Fail to track event. Unauthorized access.");
      return;
    }

    EventAction.track(new Event(eventName, eventProperty));
  }

  // initPushToken

  public static void initPushToken(String token) {
    Context context = getAppContext();
    if (context != null) {
      PrefSupervisor.setDeviceToken(context, token);
    }
  }

  // isChannelPushNotification

  public static boolean isChannelPushNotification(Map<String, String> message) {
    return ChannelPushManager.isChannelPushNotification(message);
  }

  // handlePushNotification

  public static void handlePushNotification(Activity activity) {
    if (activity != null) {
      ChannelPushClient.handlePushNotification(activity);
    }
  }

  // showPushNotification

  public static void showPushNotification(Context context, Map<String, String> message) {
    if (context != null) {
      ChannelPushManager.showPushNotification(context, message);
    }
  }

  // updateProfile

  @Deprecated
  public static void updateUser(Map<String, Object> profileMap) {
    updateUser(new UserData.Builder().setProfileMap(profileMap).build(), null);
  }

  public static void updateUser(@Nullable UserData userData, @Nullable UserUpdateCallback callback) {
    if (isBooted()) {
      UserAction.updateUser(userData, callback);
    } else if (callback != null) {
      callback.callback(ChannelException.newInstance("Please boot first"), null);
    }
  }

  // Add tags

  public static void addTags(String... tags) {
    if (tags != null) {
      addTags(ListUtils.newArrayList(tags), null);
    }
  }

  public static void addTags(List<String> tags, @Nullable TagUpdateCallback callback) {
    if (isBooted()) {
      TagAction.addTags(tags, callback);
    } else if (callback != null) {
      callback.callback(ChannelException.newInstance("Please boot first"), null);
    }
  }

  // Remove tags

  public static void removeTags(String... tags) {
    if (tags != null) {
      removeTags(ListUtils.newArrayList(tags), null);
    }
  }

  public static void removeTags(List<String> tags, @Nullable TagUpdateCallback callback) {
    if (isBooted()) {
      TagAction.removeTags(tags, callback);
    } else if (callback != null) {
      callback.callback(ChannelException.newInstance("Please boot first"), null);
    }
  }

  // library inner functions

  public static boolean isInitializedChannelIO() {
    return realChannelIO != null;
  }

  @Nullable
  public static Context getAppContext() {
    if (realChannelIO != null) {
      return realChannelIO.getApplication();
    }
    return null;
  }

  public static boolean isDebugMode() {
    return isDebugMode;
  }

  public static boolean isBooted() {
    return PluginStore.get().pluginState.get() != null;
  }
}
