package com.zoyi.channel.plugin.android.global;

import android.arch.lifecycle.*;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.ChannelIO;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store.TimerStore;
import com.zoyi.channel.plugin.android.util.L;
import com.zoyi.channel.plugin.android.util.TimeUtils;
import com.zoyi.rx.Observable;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.concurrent.TimeUnit;

public class LifecycleController implements LifecycleObserver {

  private boolean isForeground = false;

  @Nullable
  private Subscription workingTimer;

  @Nullable
  private Subscription activeActionDelayHolder;

  @OnLifecycleEvent(Lifecycle.Event.ON_START)
  public void onStart() {
    L.d("Application on foreground");
    this.isForeground = true;

    if (ChannelIO.isBooted()) {
      doOnActivated();
    }
  }

  @OnLifecycleEvent(Lifecycle.Event.ON_STOP)
  public void onStop() {
    L.d("Application on background");
    this.isForeground = false;

    doOnDeactivated();
  }

  public boolean isForeground() {
    return isForeground;
  }

  public void doOnActivated() {
    if (activeActionDelayHolder == null || activeActionDelayHolder.isUnsubscribed()) {
      activeActionDelayHolder = Observable.timer(1000, TimeUnit.MILLISECONDS)
          .onBackpressureBuffer()
          .subscribeOn(Schedulers.io())
          .observeOn(AndroidSchedulers.mainThread())
          .subscribe(l -> {
            SocketManager.connect();
            attachTimestampTimer();
          });
    }
  }

  public void doOnDeactivated() {
    if (activeActionDelayHolder != null && !activeActionDelayHolder.isUnsubscribed()) {
      activeActionDelayHolder.unsubscribe();
      activeActionDelayHolder = null;
    }

    SocketManager.disconnect();
    detachTimestampTimer();
  }

  public void attachTimestampTimer() {
    detachTimestampTimer();

    workingTimer = Observable.interval(60, TimeUnit.SECONDS)
        .onBackpressureBuffer()
        .subscribeOn(Schedulers.io())
        .observeOn(AndroidSchedulers.mainThread())
        .subscribe(l -> TimerStore.get().now.set(TimeUtils.getCurrentTime()));
  }

  public void detachTimestampTimer() {
    if (workingTimer != null) {
      if (!workingTimer.isUnsubscribed()) {
        workingTimer.unsubscribe();
      }
      workingTimer = null;
    }
  }
}
