package com.zoyi.channel.plugin.android.activity.chat.viewholder;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.InputMethodManager;
import android.widget.ImageButton;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.zoyi.channel.plugin.android.CHLocale;
import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.activity.chat.model.MobileNumber;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnProfileUpdateRequestListener;
import com.zoyi.channel.plugin.android.activity.chat.model.ProfileBotMessageItem;
import com.zoyi.channel.plugin.android.activity.chat.view.viewholder.ProfileBotCompleteView;
import com.zoyi.channel.plugin.android.base.adapter.BaseViewHolder;
import com.zoyi.channel.plugin.android.enumerate.FetchState;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.rest.Plugin;
import com.zoyi.channel.plugin.android.model.rest.ProfileBot;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.store.PluginStore;
import com.zoyi.channel.plugin.android.store.ProfileBotStore;
import com.zoyi.channel.plugin.android.store.SettingsStore;
import com.zoyi.channel.plugin.android.store.UserStore;
import com.zoyi.channel.plugin.android.util.IntentUtils;
import com.zoyi.channel.plugin.android.util.NumberUtils;
import com.zoyi.channel.plugin.android.util.ProfileUtils;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.channel.plugin.android.util.Utils;
import com.zoyi.channel.plugin.android.view.layout.WatchedEditText;

import java.util.Locale;

import static android.text.InputType.TYPE_CLASS_NUMBER;
import static android.text.InputType.TYPE_CLASS_TEXT;
import static android.text.InputType.TYPE_NUMBER_FLAG_DECIMAL;

public class ProfileBotMessageHolder extends BaseViewHolder {

  private LinearLayout layoutCompleteItems;
  private View viewProfileBotInput;

  private TextView inputLabel;
  private TextView textPage;
  private TextView textTotalPages;

  private View layoutInput;
  private View buttonSelectCountry;
  private TextView textCountry;
  private WatchedEditText editValue;
  private View inputProgress;
  private ImageButton buttonSubmit;

  private TextView textTerms;

  @Nullable
  private ProfileBotMessageItem item;

  @Nullable
  private ProfileBot activeProfileBot;

  @Nullable
  private OnProfileUpdateRequestListener listener;

  public ProfileBotMessageHolder(@NonNull View itemView, @Nullable OnProfileUpdateRequestListener listener) {
    super(itemView);

    this.listener = listener;

    layoutCompleteItems = itemView.findViewById(R.id.layoutProfileBotItemCompleteItems);
    viewProfileBotInput = itemView.findViewById(R.id.viewProfileBotItemInput);

    inputLabel = itemView.findViewById(R.id.textProfileBotItemLabel);
    textPage = itemView.findViewById(R.id.textProfileBotItemPage);
    textTotalPages = itemView.findViewById(R.id.textProfileBotItemTotalPages);

    layoutInput = itemView.findViewById(R.id.layoutProfileBotInput);

    buttonSelectCountry = itemView.findViewById(R.id.buttonProfileBotItemSelectCountry);
    buttonSelectCountry.setOnClickListener(v -> {
      if (listener != null && item != null) {
        listener.onCountrySelectorClick(item);
      }
    });

    textCountry = itemView.findViewById(R.id.textProfileBotItemSelectCountry);

    editValue = itemView.findViewById(R.id.editProfileBotItem);
    editValue.setOnFocusChangeListener((v, hasFocus) -> {
      if (ProfileBotStore.get().requestState.get() == FetchState.COMPLETE) {
        if (hasFocus) {
          layoutInput.setBackgroundResource(R.drawable.ch_plugin_profile_message_input);
        } else {
          layoutInput.setBackgroundResource(R.drawable.ch_plugin_profile_message_input_out_focus);
        }
      }

      if (ProfileBotStore.get().requestState.get() != FetchState.LOADING) {
        ProfileBotStore.get().requestFocus.set(hasFocus);
      }
    });
    editValue.setWatchedTextChangedListener(s -> {
      if (activeProfileBot != null) {
        if (Const.PROFILE_MOBILE_NUMBER_KEY.equals(activeProfileBot.getKey())) {
          Object restoreData = ProfileBotStore.get().inputValue.get();

          if (restoreData instanceof MobileNumber) {
            ((MobileNumber) restoreData).setPhoneNumber(s);
            ProfileBotStore.get().inputValue.set(restoreData);
          }
        } else {
          ProfileBotStore.get().inputValue.set(s);
        }

        ProfileBotStore.get().requestState.set(FetchState.COMPLETE);
        setLayoutStyle(activeProfileBot, FetchState.COMPLETE);
      }
    });

    inputProgress = itemView.findViewById(R.id.viewProfileBotItemProgress);
    buttonSubmit = itemView.findViewById(R.id.buttonProfileBotItemSubmit);
    buttonSubmit.setOnClickListener(v -> {
      if (listener != null && item != null) {
        listener.onProfileRequest(item);
      }
    });

    textTerms = itemView.findViewById(R.id.textProfileBotItemTerms);
    textTerms.setText(Utils.getHtmlFormattedText(ResUtils.getString("ch.agreement")));
    textTerms.setOnClickListener(v -> {
      Plugin plugin = PluginStore.get().pluginState.get();
      CHLocale locale = SettingsStore.get().locale.get();

      if (plugin != null && plugin.getKey() != null) {
        IntentUtils.setUrl(
            itemView.getContext(),
            String.format("https://channel.io/%s/terms_user?plugin_key=%s", locale.toString(), plugin.getKey())
        ).startActivity();
      } else {
        IntentUtils.setUrl(
            itemView.getContext(),
            String.format("https://channel.io/%s/terms_user", locale.toString())
        ).startActivity();
      }
    });
  }

  private void setLayoutStyle(ProfileBot profileBot, FetchState fetchState) {
    if (profileBot == null) {
      return;
    }

    switch (fetchState) {
      case LOADING:
        inputLabel.setText(profileBot.getName());
        inputLabel.setTextColor(ResUtils.getColor(R.color.ch_light));

        editValue.setEnabled(false);

        inputProgress.setVisibility(View.VISIBLE);
        buttonSubmit.setVisibility(View.GONE);

        layoutInput.setBackgroundResource(R.drawable.ch_plugin_profile_message_input_out_focus);
        break;

      case FAILED:
        inputLabel.setText(ResUtils.getString("ch.profile_form.error"));
        inputLabel.setTextColor(ResUtils.getColor(R.color.ch_orange));

        editValue.setEnabled(true);

        inputProgress.setVisibility(View.GONE);
        buttonSubmit.setVisibility(View.VISIBLE);
        buttonSubmit.setImageResource(R.drawable.ch_send_error);

        layoutInput.setBackgroundResource(R.drawable.ch_plugin_profile_message_input_wrong);
        break;

      case COMPLETE:
        inputLabel.setText(profileBot.getName());
        inputLabel.setTextColor(ResUtils.getColor(R.color.ch_light));

        editValue.setEnabled(true);

        inputProgress.setVisibility(View.GONE);
        buttonSubmit.setVisibility(View.VISIBLE);
        buttonSubmit.setImageResource(R.drawable.ch_send);

        if (editValue.isFocused()) {
          layoutInput.setBackgroundResource(R.drawable.ch_plugin_profile_message_input);
        } else {
          layoutInput.setBackgroundResource(R.drawable.ch_plugin_profile_message_input_out_focus);
        }
        break;
    }
  }

  public void bind(ProfileBotMessageItem item) {
    this.item = item;

    // ignore null case (not necessary)
    if (item == null) {
      return;
    }

    int selectedIndex = -1;

    for (int i = 0; i < item.getProfileBots().size(); i++) {
      ProfileBot profileBot = item.getProfileBots().get(i);

      if (profileBot == null || profileBot.getKey() == null) {
        continue;
      }

      // complete scheme
      if (profileBot.getValue() != null) {
        if (i < layoutCompleteItems.getChildCount()) {
          View view = layoutCompleteItems.getChildAt(i);

          if (view instanceof ProfileBotCompleteView) {
            view.setVisibility(View.VISIBLE);
            ((ProfileBotCompleteView) view).bind(profileBot);
          }
        } else {
          ProfileBotCompleteView view = new ProfileBotCompleteView(itemView.getContext());
          view.bind(profileBot);

          layoutCompleteItems.addView(view);
        }
      } else {
        selectedIndex = i;

        // set visibility when gone

        if (viewProfileBotInput.getVisibility() == View.GONE) {
          viewProfileBotInput.setVisibility(View.VISIBLE);
        }

        activeProfileBot = profileBot;

        textPage.setText(String.valueOf(i + 1));
        textTotalPages.setText(String.format(Locale.US, "/%d", item.getProfileBots().size()));

        String prevKey = ProfileBotStore.get().inputKey.get();

        // if re-render holder
        if (prevKey == null || profileBot.getKey().equals(prevKey)) {
          setLayoutStyle(profileBot, ProfileBotStore.get().requestState.get());
        }

        // save profile key to handle in presenter
        ProfileBotStore.get().inputKey.set(profileBot.getKey());

        if (Const.PROFILE_MOBILE_NUMBER_KEY.equals(profileBot.getKey())) {
          setMobileNumberInput();
        } else if (Const.PROFILE_TYPE_NUMBER.equals(profileBot.getType())) {
          setNumericInput(profileBot.getKey());
        } else {
          setStringInput(profileBot.getKey());
        }

        // hide overflow complete view (maybe not used)

        for (int index = i; i < layoutCompleteItems.getChildCount(); i++) {
          layoutCompleteItems.getChildAt(index).setVisibility(View.GONE);
        }

        break;
      }
    }

    // terms only show in first profile bot input
    textTerms.setVisibility(selectedIndex == 0 ? View.VISIBLE : View.GONE);

    // if all profile bot is set, hide input view
    if (selectedIndex == -1) {
      activeProfileBot = null;
      viewProfileBotInput.setVisibility(View.GONE);

      if (ProfileBotStore.get().requestFocus.get()) {
        ProfileBotStore.get().requestFocus.set(false);

        if (listener != null) {
          listener.onProfileUpdateComplete();
        }
      }
    } else if (ProfileBotStore.get().requestFocus.get() && ProfileBotStore.get().requestState.get() != FetchState.LOADING) {
      editValue.requestFocus();

      InputMethodManager imm = (InputMethodManager) itemView.getContext().getSystemService(Context.INPUT_METHOD_SERVICE);
      imm.toggleSoftInput(InputMethodManager.SHOW_FORCED, 0);
    }
  }

  @NonNull
  private String restoreStringData(String key) {
    User user = UserStore.get().user.get();

    if (user != null) {
      return ProfileUtils.getFormattedString(user.getProfileValue(key));
    }
    return "";
  }

  @NonNull
  private MobileNumber restoreMobileNumberData() {
    User user = UserStore.get().user.get();

    if (user != null) {
      Object profileValue = user.getProfileValue(Const.PROFILE_MOBILE_NUMBER_KEY);

      if (profileValue instanceof String) {
        return new MobileNumber(String.valueOf(profileValue));
      }
    }
    return new MobileNumber();
  }

  private void setNumericInput(String key) {
    editValue.setInputType(TYPE_CLASS_NUMBER | TYPE_NUMBER_FLAG_DECIMAL);

    Object savedData = ProfileBotStore.get().inputValue.get();

    if (savedData instanceof String) {
      setTextKeepSelection((String) savedData);
    } else {
      String restoreData = restoreStringData(key);

      ProfileBotStore.get().inputValue.set(restoreData);

      setTextWithBlock(restoreData);
    }

    buttonSelectCountry.setVisibility(View.GONE);
  }

  private void setStringInput(String key) {
    editValue.setInputType(TYPE_CLASS_TEXT);

    Object savedData = ProfileBotStore.get().inputValue.get();

    if (savedData instanceof String) {
      setTextKeepSelection((String) savedData);
    } else {
      String restoreData = restoreStringData(key);

      ProfileBotStore.get().inputValue.set(restoreData);

      setTextWithBlock(restoreData);
    }

    buttonSelectCountry.setVisibility(View.GONE);
  }

  private void setMobileNumberInput() {
    editValue.setInputType(TYPE_CLASS_NUMBER | TYPE_NUMBER_FLAG_DECIMAL);

    Object savedData = ProfileBotStore.get().inputValue.get();

    if (savedData instanceof MobileNumber) {
      textCountry.setText(String.format("+%s", ((MobileNumber) savedData).getCountryCode()));

      setTextKeepSelection(((MobileNumber) savedData).getPhoneNumber());
    } else {
      MobileNumber restoreData = restoreMobileNumberData();

      ProfileBotStore.get().inputValue.set(restoreData);

      textCountry.setText(String.format("+%s", restoreData.getCountryCode()));

      setTextWithBlock(restoreData.getPhoneNumber());
    }

    buttonSelectCountry.setVisibility(View.VISIBLE);
  }

  private void setTextKeepSelection(String value) {
    int selectionStart = NumberUtils.bewteen(editValue.getSelectionStart(), 0, value.length());
    int selectionEnd = NumberUtils.bewteen(editValue.getSelectionEnd(), 0, value.length());

    editValue.setTextWithBlock(value);
    editValue.setSelection(selectionStart, selectionEnd);
    editValue.setBlock(false);
  }

  private void setTextWithBlock(String value) {
    editValue.setTextWithBlock(value);
    editValue.selectAll();
    editValue.setBlock(false);
  }

  public static ProfileBotMessageHolder newInstance(ViewGroup parent, @Nullable OnProfileUpdateRequestListener listener) {
    LayoutInflater inflater = LayoutInflater.from(parent.getContext());

    return new ProfileBotMessageHolder(
        inflater.inflate(R.layout.ch_plugin_item_message_profile_bot, parent, false),
        listener
    );
  }
}
