package com.zoyi.channel.plugin.android;

import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.os.Looper;
import android.support.annotation.Nullable;

import com.zoyi.channel.plugin.android.action.*;
import com.zoyi.channel.plugin.android.activity.chat.manager.ChatManager;
import com.zoyi.channel.plugin.android.activity.common.chat.ChatContentType;
import com.zoyi.channel.plugin.android.activity.lounge.LoungeActivity;
import com.zoyi.channel.plugin.android.enumerate.ActionType;
import com.zoyi.channel.plugin.android.enumerate.Transition;
import com.zoyi.channel.plugin.android.global.*;
import com.zoyi.channel.plugin.android.model.rest.User;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.wrapper.PluginWrapper;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.push.ChannelPushManager;
import com.zoyi.channel.plugin.android.selector.GlobalSelector;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store.*;
import com.zoyi.channel.plugin.android.store.base.Store;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.com.annimon.stream.Optional;

import java.util.concurrent.RejectedExecutionException;

import static com.zoyi.channel.plugin.android.ChannelIO.isInitializedChannelIO;

class RealChannelIO {

  private Application application;

  private Thread.UncaughtExceptionHandler uncaughtExceptionHandler;

  private ChannelActionHandler handler;

  private ActivityLifecycleManager activityLifecycleManager;

  RealChannelIO(Application application) {
    this.application = application;
    this.uncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
    this.handler = new ChannelActionHandler();

    activityLifecycleManager = new ActivityLifecycleManager();

    SocketManager.create(application);
    application.registerActivityLifecycleCallbacks(activityLifecycleManager);

    Thread.setDefaultUncaughtExceptionHandler((thread, ex) -> {
      if (uncaughtExceptionHandler != null) {
        uncaughtExceptionHandler.uncaughtException(thread, ex);
      }

      if (!SocketManager.isReady() && Looper.getMainLooper().getThread() != thread && ex instanceof RejectedExecutionException) {
        SocketManager.reconnect();
      }
    });
  }

  Application getApplication() {
    return application;
  }

  void boot(@Nullable ChannelPluginSettings pluginSettings, @Nullable Profile profile, @Nullable OnBootListener listener) {
    shutdown();

    BootAction.boot(pluginSettings, profile, listener, new RestSubscriber<PluginWrapper>() {
      @Override
      public void onError(RetrofitException e) {
        BootManager.sendNetworkError(listener, e);
      }

      @Override
      public void onNext(PluginWrapper repo) {
        repo.set();

        onBoot(pluginSettings, repo.getChannel(), repo.getPlugin(), repo.getUser(), listener);

        Optional.ofNullable(repo)
            .map(PluginWrapper::getUser)
            .map(User::getPopUpChatId)
            .ifPresent(chatId -> ChatAction.fetchPopUpChat(repo.getUser().getPopUpChatId()));
      }
    });
  }

  private void onBoot(
      ChannelPluginSettings pluginSettings,
      Channel channel,
      Plugin plugin,
      User user,
      @Nullable OnBootListener listener
  ) {
    // Set latest boot data
    PrefSupervisor.setLatestBootData(application, pluginSettings.getPluginKey(), channel.getId(), user.getId());

    // set prefs
    SettingsStore.get().locale.set(user.getUserLocale());
    SettingsStore.get().launcherConfig.set(pluginSettings.getLauncherConfig());
    SettingsStore.get().hideDefaultInAppPush.set(pluginSettings.isHideDefaultInAppPush());
    SettingsStore.get().enabledTrackDefaultEvent.set(pluginSettings.isEnabledTrackDefaultEvent());

    // Set debug mode to console
    ChannelIO.setDebugMode(pluginSettings.isDebugMode());

    // Set settings
    SettingsStore.get().showTranslation.set(PrefSupervisor.canTranslateMessage(application));
    SettingsStore.get().showClosedChat.set(PrefSupervisor.isShownClosedChat(application));
    SettingsStore.get().raiseSoundVibrate.set(PrefSupervisor.isEnabledPushAlarm(application));

    // Enable chat
    ChatManager.initialize();

    // when jwt is not initialized, fetch from preference
    if (GlobalStore.get().jwt.get() == null) {
      GlobalStore.get().jwt.set(PrefSupervisor.getJwt(application));
    }

    ChannelPushManager.sendTokenToChannelIO(application);

    SocketManager.setChannelId(channel.getId());

    if (GlobalStore.get().topActivity.get() != null) {
      SocketManager.connect();
    }

    // if already call ChannelIO.show(), send page view event
    if (SettingsStore.get().showLauncher.get() && SettingsStore.get().enabledTrackDefaultEvent.get()) {
      EventAction.trackPageView();
    }

    // attach handler
    handler.handle();

    // attach now updater
    activityLifecycleManager.attachTimestampTimer();

    // update now force
    TimerStore.get().now.set(TimeUtils.getCurrentTime());

    // raise event to initialize
    GlobalStore.get().bootState.set(true);

    // send to listener

    if (listener != null) {
      listener.onCompletion(
          ChannelPluginCompletionStatus.SUCCESS,
          com.zoyi.channel.plugin.android.User.newInstance(user)
      );
    }

    // clear push data. when need to handle push after boot, do it in listener.onCompletion
    PrefSupervisor.clearLatestPushData(application);
  }

  void shutdown() {
    GlobalStore.get().bootState.set(false);

    handler.unHandle();

    activityLifecycleManager.detachTimestampTimer();

    SocketManager.setChannelId(null);
    SocketManager.disconnect();

    Action.invoke(ActionType.SHUTDOWN);
    Action.release();

    ChatManager.release();

    String jwt = GlobalStore.get().jwt.get();
    ChannelPushManager.deleteToken(application, jwt);

    Store.destroy();

    PrefSupervisor.clearLatestBootData(application);
    PrefSupervisor.clearLatestPushData(application);

    PrefSupervisor.setJwt(application, null);
  }

  // start messenger

  public boolean startMessenger(@Nullable Context context, @Nullable String chatId, boolean isAnimated) {
    if (!isInitializedChannelIO()) {
      L.e("Fail to start messenger, please initialize ChannelIO first");
      return false;
    }

    if (!GlobalStore.get().bootState.get()) {
      L.e("Fail to start messenger, please 'Boot' first");
      return false;
    }

    if (context == null) {
      L.e("Fail to start messenger, context can't be NULL");
      return false;
    }

    IntentUtils.setNextActivity(context, LoungeActivity.class)
        .putExtra(Const.EXTRA_CHAT_CONTENT_TYPE, ChatContentType.USER_CHAT.toString())
        .putExtra(Const.EXTRA_CHAT_CONTENT_ID, chatId)
        .putExtra(Const.EXTRA_REDIRECT_ANIMATED, isAnimated)
        .putExtra(Const.EXTRA_TOP_ACTIVITY_NAME, GlobalSelector.getTopActivityName())
        .setFlag(Intent.FLAG_ACTIVITY_CLEAR_TOP)
        .setTransition(isAnimated ? Transition.SLIDE_FROM_BOTTOM : Transition.NONE)
        .startActivity();

    return true;
  }

  // handle listener

  void setListener(@Nullable ChannelPluginListener listener) {
    handler.setListener(listener);
  }

  void clearListener() {
    handler.clearListener();
  }

  @Nullable
  ChannelPluginListener getListener() {
    return handler.getListener();
  }
}
