package com.zoyi.channel.plugin.android.view.popup;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.zoyi.channel.plugin.android.R;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.entity.ProfileEntity;
import com.zoyi.channel.plugin.android.model.etc.PushItem;
import com.zoyi.channel.plugin.android.model.rest.PreviewThumb;
import com.zoyi.channel.plugin.android.push_bot.OnPushClickListener;
import com.zoyi.channel.plugin.android.store2.binder.Binder;
import com.zoyi.channel.plugin.android.util.AlarmCenter;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.util.ResUtils;
import com.zoyi.channel.plugin.android.util.TimeUtils;
import com.zoyi.channel.plugin.android.view.layout.AvatarLayout;
import com.zoyi.com.annimon.stream.Optional;
import com.zoyi.com.bumptech.glide.Glide;
import com.zoyi.com.bumptech.glide.load.engine.DiskCacheStrategy;

/**
 * Created by jerry on 2018. 11. 20..
 */

public abstract class BasePushView extends LinearLayout {

  protected Context context;

  protected View root;

  protected View itemRoot;

  @Nullable
  protected View buttonPopupClose;

  @Nullable
  protected AvatarLayout avatarPopup;

  @Nullable
  protected TextView textPopupName;

  @Nullable
  protected TextView textPopupTime;

  @Nullable
  protected TextView textPopupMessage;

  protected ViewGroup layoutPushBotPopupImage;
  protected ImageView imagePushBotPopupImage;
  protected View buttonPushBotPopupButton;

  @Nullable
  protected OnPushClickListener onPushClickListener;

  @Nullable
  protected Binder binder;

  @Nullable
  protected PushItem item;

  public BasePushView(@NonNull Context context) {
    super(context);
    init(context);
  }

  public BasePushView(@NonNull Context context, @Nullable AttributeSet attrs) {
    super(context, attrs);
    init(context);
  }

  public BasePushView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context);
  }

  protected abstract int getLayoutId();

  @Initializer
  protected void init(Context context) {
    this.context = context;

    root = LayoutInflater.from(context).inflate(getLayoutId(), this, true);
    root.setVisibility(GONE);

    itemRoot = root.findViewById(R.id.ch_root_popup);
    buttonPopupClose = root.findViewById(R.id.ch_button_popup_close);
    avatarPopup = root.findViewById(R.id.ch_avatar_popup);
    textPopupName = root.findViewById(R.id.ch_text_popup_name);
    textPopupTime = root.findViewById(R.id.ch_text_popup_time);

    textPopupMessage = root.findViewById(R.id.ch_text_popup_message);

    layoutPushBotPopupImage = root.findViewById(R.id.ch_layout_popup_image);
    imagePushBotPopupImage = root.findViewById(R.id.ch_image_popup_image);
    buttonPushBotPopupButton = root.findViewById(R.id.ch_layout_popup_button);

    if (buttonPopupClose != null) {
      buttonPopupClose.setOnClickListener(v -> hideView());
    }

    itemRoot.setOnClickListener(v -> {
      if (item != null && onPushClickListener != null) {
        onPushClickListener.onPushClick(item.getMobileExposureType(), item.getId());
      }
      hideView();
    });

    layoutPushBotPopupImage.setOnClickListener(v -> {
      if (item != null && onPushClickListener != null) {
        onPushClickListener.onPushContentsClick(item.getId(), item.getContentUrl());
      }
      hideView();
    });

    buttonPushBotPopupButton.setOnClickListener(v -> {
      if (item != null && onPushClickListener != null) {
        onPushClickListener.onPushContentsClick(item.getId(), item.getContentUrl());
      }
      hideView();
    });
  }

  @Override
  protected void onDetachedFromWindow() {
    super.onDetachedFromWindow();

    if (binder != null) {
      binder.unbind();
    }
  }

  protected void bindPopupData(@NonNull PushItem pushItem) {

    // Reset popup content visibility (button, image.. etc)
    resetPopupContent();

    // Set avatar, bot name, time0
    bindProfileData(pushItem);

    // Set message
    bindContent(pushItem);
    bindTitle(pushItem);

    // Set alarm
    if (PrefSupervisor.isEnabledPushAlarm(context)) {
      AlarmCenter.alarm(context);
    }

    // Set button or image
    if (pushItem.getAttachment() != null) {
      switch (pushItem.getAttachment()) {
        case Const.PUSH_BOT_ATTACHMENT_BUTTON:
          bindButton();
          break;

        case Const.PUSH_BOT_ATTACHMENT_IMAGE:
          bindImage();
          break;
      }
    }
  }

  protected void bindProfileData(PushItem pushItem) {
    ProfileEntity person = pushItem.getProfile();

    if (avatarPopup != null) {
      avatarPopup.set(person);
    }

    if (textPopupName != null) {
      textPopupName.setText(person != null ? person.getName() : ResUtils.getUnknown());
    }

    if (textPopupTime != null) {
      textPopupTime.setText(TimeUtils.getTime(pushItem.getCreatedAt()));
    }
  }

  public void setPushBotClickListener(OnPushClickListener onPushClickListener) {
    this.onPushClickListener = onPushClickListener;
  }

  public void show(PushItem pushItem) {
    if(root.getVisibility() == View.VISIBLE) {
      hideView();
    }

    this.item = pushItem;
    bindPopupData(item);

    root.setVisibility(VISIBLE);
  }

  public void hideView() {
    resetPopupContent();

    if (item != null) {
      item.hidePopup();
      item = null;
    }
  }

  protected void resetPopupContent() {
    root.setVisibility(GONE);
    buttonPushBotPopupButton.setVisibility(View.GONE);

    Glide.clear(imagePushBotPopupImage);
    layoutPushBotPopupImage.setVisibility(View.GONE);

    if (textPopupMessage != null) {
      textPopupMessage.setText("");
    }
  }


  protected void bindImage() {
    String url = Optional.ofNullable(item)
        .map(PushItem::getImageThumb)
        .map(PreviewThumb::getUrl)
        .orElse(null);

    if (url != null) {
      Glide.with(context)
          .load(url)
          .centerCrop()
          .diskCacheStrategy(DiskCacheStrategy.SOURCE)
          .into(imagePushBotPopupImage);

      layoutPushBotPopupImage.setVisibility(VISIBLE);

      if (textPopupMessage != null) {
        textPopupMessage.setMaxLines(getAttachmentTextMaxLine());
      }
    } else {
      Glide.clear(imagePushBotPopupImage);

      layoutPushBotPopupImage.setVisibility(GONE);

      if (textPopupMessage != null) {
        textPopupMessage.setMaxLines(getDefaultTextMaxLine());
      }
    }
  }

  abstract protected int getDefaultTextMaxLine();

  abstract protected int getAttachmentTextMaxLine();

  abstract protected void bindTitle(PushItem pushItem);

  abstract protected void bindContent(PushItem pushItem);

  abstract protected void bindButton();

}
