package com.zoyi.channel.plugin.android.activity.chat;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import com.zoyi.channel.plugin.android.ChannelApiManager;
import com.zoyi.channel.plugin.android.ChannelIO;
import com.zoyi.channel.plugin.android.activity.chat.listener.OnMessageSendListener;
import com.zoyi.channel.plugin.android.activity.chat.model.*;
import com.zoyi.channel.plugin.android.enumerate.*;
import com.zoyi.channel.plugin.android.event.ChannelModelBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.global.ApiTag;
import com.zoyi.channel.plugin.android.global.Const;
import com.zoyi.channel.plugin.android.model.ActionInput;
import com.zoyi.channel.plugin.android.model.PushBotItem;
import com.zoyi.channel.plugin.android.model.TranslationInfo;
import com.zoyi.channel.plugin.android.model.entity.Entity;
import com.zoyi.channel.plugin.android.model.entity.Guest;
import com.zoyi.channel.plugin.android.model.etc.Typing;
import com.zoyi.channel.plugin.android.model.rest.*;
import com.zoyi.channel.plugin.android.model.wrapper.*;
import com.zoyi.channel.plugin.android.network.RestSubscriber;
import com.zoyi.channel.plugin.android.network.RetrofitException;
import com.zoyi.channel.plugin.android.selector.*;
import com.zoyi.channel.plugin.android.socket.SocketManager;
import com.zoyi.channel.plugin.android.store.CountryStore;
import com.zoyi.channel.plugin.android.store.ImageFileStore;
import com.zoyi.channel.plugin.android.store.PushBotItemStore;
import com.zoyi.channel.plugin.android.store.Store;
import com.zoyi.channel.plugin.android.store.TranslationStore;
import com.zoyi.channel.plugin.android.util.*;
import com.zoyi.okhttp3.RequestBody;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.schedulers.Schedulers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Created by mika on 2018. 11. 6..
 */

public class ChatPresenter2 implements ChatContract2.Presenter, OnMessageSendListener {

  @NonNull
  private ChatContract2.View view;
  @NonNull
  private ChatAdapterContract2.View adapterView;
  @NonNull
  private ChatAdapterContract2.Model adapterModel;

  @NonNull
  private TypingManager typingManager;

  @Nullable
  private String chatId;
  @Nullable
  private String pushBotId;
  @Nullable
  private UserChat userChat;
  @Nullable
  private Session session;

  @Nullable
  private String backwardId;

  // INIT_MESSAGES_LOADING 부터 READY 까지 소켓으로 온 메시지를 저장하는 리스트
  @NonNull
  private List<Message> receiveMessagesWaitingQueue;

  // 유저챗이 없을 때, 유저챗을 만들고 바로 보낼 메시지들을 저장하는 리스트
  @NonNull
  private List<SendingMessageItem> sendingMessagesWaitingQueue;

  private ChatState chatState = ChatState.IDLE;

  ChatPresenter2(@Nullable String chatId, @Nullable String pushBotId, @NonNull TypingManager typingManager) {
    TimeUtils.refreshOffset();

    this.chatId = chatId;
    this.pushBotId = pushBotId;
    this.typingManager = typingManager;
    this.receiveMessagesWaitingQueue = new ArrayList<>();
    this.sendingMessagesWaitingQueue = new ArrayList<>();

    ChatManager.get().addListener(this);
  }

  @Initializer
  @Override
  public void setView(@NonNull ChatContract2.View view) {
    this.view = view;
  }

  @Initializer
  @Override
  public void setAdapterView(@NonNull ChatAdapterContract2.View adapterView) {
    this.adapterView = adapterView;
  }

  @Initializer
  @Override
  public void setAdapterModel(@NonNull ChatAdapterContract2.Model adapterModel) {
    this.adapterModel = adapterModel;
  }

  @Override
  public void release() {
    ChatManager.get().removeListener(this);

    if (chatId != null) {
      SocketManager.leaveChat(chatId);
    }
    cancelApis();

    Store.getInstance(ImageFileStore.class).clear();
    Store.getInstance(TranslationStore.class).clearData();
  }

  // Initializing

  /**
   * Chat id 유무를 통해 랜더링할 메시지를 정하는 함수
   */
  @Override
  public void init() {
    if (chatId != null) {
      chatState = ChatState.WAITING_SOCKET;
      typingManager.setChatId(chatId);
      checkSocketState();
    } else {
      view.showInitBanner();
      addInitMessage();
    }
  }

  // Logic for init message

  /**
   * 가지고 있는 데이터에 따라 보여지는 로컬메세지 종류가 달라짐
   * 각 상황이 겹칠수 있는 케이스는 없음
   */
  private void addInitMessage() {
    PushBotItem pushBotItem = PushBotSelector.get(pushBotId);

    if (pushBotItem != null) {
      // 읽음 처리
      pushBotItem.read();
      Store.getInstance(PushBotItemStore.class).add(pushBotItem);

      chatState = ChatState.LOCAL_PUSH_BOT;
      adapterModel.upsertInitMessage(new PushBotMessageItem(pushBotItem));
      view.setInputFrameVisibility(false);
      view.setPushBotSaveButtonVisibility(true);
    } else if (SupportBotSelector.getSupportBotEntry() != null) {
      chatState = ChatState.LOCAL_SUPPORTING;
      adapterModel.upsertInitMessage(new SupportBotMessageItem(SupportBotSelector.getSupportBotEntry()));
      view.setInputFrameVisibility(false);
    } else {
      chatState = ChatState.LOCAL_WELCOME;
      adapterModel.upsertInitMessage(new WelcomeMessageItem(PluginSelector.getWelcomeMessage(ChannelIO.getAppContext())));
      view.setInputFrameVisibility(true);
    }
  }

  /**
   * 환영 메시지 갱신, 설정에서 언어 바꿨을 때 불림
   */
  @Override
  public void updateInitMessage() {
    if (chatId == null) {
      addInitMessage();
    }
  }

  // Logic for fetch messages

  /**
   * Refresh 버튼 눌리거나, 앱이 백그라운드에서 돌아오면 불림
   */
  @Override
  public void refresh() {
    view.onTryRefresh();

    switch (chatState) {
      case WAITING_SOCKET:
        checkSocketState();
        break;

      case USER_CHAT_NOT_LOADED:
        fetchUserChat();
        break;

      case INIT_MESSAGES_NOT_LOADED:
        fetchInitMessages();
        break;
    }
  }

  /**
   * 초기화, 소켓 재연결 시에 불림
   */
  private void checkSocketState() {
    if (chatState == ChatState.WAITING_SOCKET) {
      if (chatId != null) {
        if (SocketManager.isReady()) {
          chatState = ChatState.CHAT_JOINING;
          SocketManager.joinChat(chatId);
        } else {
          SocketManager.reconnect();
        }
      }
    }
  }

  /**
   * 채팅 조인에 성공시에 유저챗 정보 불러오는 함수
   */
  private void fetchUserChat() {
    if (chatId != null && CompareUtils.exists(chatState, ChatState.CHAT_JOINED, ChatState.USER_CHAT_NOT_LOADED)) {
      chatState = ChatState.USER_CHAT_LOADING;

      ChannelApiManager.callOnce(
          ApiTag.FETCH_USER_CHAT,
          ChannelApiManager.get().getUserChat(chatId),
          new RestSubscriber<UserChatWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              chatState = ChatState.USER_CHAT_NOT_LOADED;
              view.onError();

              L.e(error.getMessage());
            }

            @Override
            public void onSuccess(@NonNull UserChatWrapper userChatRepo) {
              if (userChatRepo.getUserChat() != null) {
                userChatRepo.update();

                view.onUserChatChange(userChat, userChatRepo.getUserChat());
                onUserChatChange(userChat, userChatRepo.getUserChat());

                userChat = userChatRepo.getUserChat();
                session = userChatRepo.getSession();

                chatState = ChatState.USER_CHAT_LOADED;

                fetchInitMessages();
              }
            }
          }
      );
    }
  }

  /**
   * 유저챗 로드 성공 시에 메시지 불러오는 함수
   */
  private void fetchInitMessages() {
    if ((chatState == ChatState.USER_CHAT_LOADED || chatState == ChatState.INIT_MESSAGES_NOT_LOADED) && chatId != null) {
      receiveMessagesWaitingQueue.clear();

      chatState = ChatState.INIT_MESSAGES_LOADING;

      ChannelApiManager.callOnce(
          ApiTag.FETCH_INIT_MESSAGES,
          ChannelApiManager.get().getMessages(chatId, Const.MESSAGE_ID_MAX, Const.MESSAGE_FETCH_LIMIT, Const.DESC),
          new RestSubscriber<MessagesWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              chatState = ChatState.INIT_MESSAGES_NOT_LOADED;
              view.onError();

              receiveMessagesWaitingQueue.clear();

              L.e(error.getMessage());
            }

            @Override
            public void onSuccess(@NonNull MessagesWrapper messagesRepo) {
              chatState = ChatState.CHAT_READY;

              messagesRepo.update();

              backwardId = messagesRepo.getNext();

              List<Message> messages = ListUtils.union(messagesRepo.getMessages(), receiveMessagesWaitingQueue);
              receiveMessagesWaitingQueue.clear();

              adapterModel.setMessages(messages);
              adapterModel.addMessageItems(ChatManager.get().getSendingItems(chatId));
              adapterModel.addMessageItems(ChatManager.get().getSendingFailedItems(chatId));
              updateNewMessageItem(messages);

              // Check messages that create user chat
              if (chatId != null && sendingMessagesWaitingQueue.size() > 0) {
                List<SendingMessageItem> sendingQueue = new ArrayList<>();
                List<SendingMessageItem> failedQueue = new ArrayList<>();

                for (SendingMessageItem item : sendingMessagesWaitingQueue) {
                  item.setChatId(chatId);

                  if (item.isSending()) {
                    sendingQueue.add(item);
                  } else {
                    failedQueue.add(item);
                  }
                }

                ChatManager.get().send(sendingQueue);
                ChatManager.get().updateFailedMessages(failedQueue);

                adapterModel.addMessageItems(sendingMessagesWaitingQueue);
                sendingMessagesWaitingQueue.clear();
              }

              ChatManager.get().read(chatId);

              view.scrollToBottom();
            }
          }
      );
    }
  }

  /**
   * 상단으로 스크롤을 했을 때 추가 메시지가 있을 경우 불러오는 함수
   */
  @Override
  public void fetchBackwardMessages() {
    if (chatId != null &&
        backwardId != null &&
        chatState == ChatState.CHAT_READY &&
        !ChannelApiManager.isRunning(ApiTag.FETCH_BACKWARD_MESSAGES)) {
      ChannelApiManager.callOnce(
          ApiTag.FETCH_BACKWARD_MESSAGES,
          ChannelApiManager.get().getMessages(
              chatId,
              backwardId,
              Const.MESSAGE_FETCH_LIMIT,
              Const.DESC
          ),
          new RestSubscriber<MessagesWrapper>() {
            @Override
            public void onNext(MessagesWrapper messagesRepo) {
              messagesRepo.update();

              backwardId = messagesRepo.getNext();

              adapterModel.addMessages(messagesRepo.getMessages());
              updateNewMessageItem(messagesRepo.getMessages());
            }
          }
      );
    }
  }

  /**
   * '새로운 메시지입니다' 라는 메시지를 보여줘야 할 경우 보여주는 함수
   * @param messages 현재 가져온 메시지들. 이 메시지들 사이에 껴서 보여줄 수 있을 때만 보여주게 됨
   */
  private void updateNewMessageItem(@Nullable List<Message> messages) {
    if (session == null || session.getReadAt() == null) {
      return;
    }

    long lastReadAt = session.getReadAt();
    Message minMessage = ListUtils.takeFirstPrioirty(messages, Message.getMinComparator());
    Message maxMessage = ListUtils.takeFirstPrioirty(messages, Message.getMaxComparator());

    if (minMessage != null && minMessage.getCreatedAt() <= lastReadAt && maxMessage != null && maxMessage.getCreatedAt() > lastReadAt) {
      adapterModel.upsertNewMessageItem(lastReadAt);
    }
  }

  // event receiver

  @Override
  public void receiveCommand(Command command, @Nullable Object object) {
    switch (command) {
      case APP_STARTED:
        if (chatState == ChatState.WAITING_SOCKET) {
          refresh();
        }
        break;

      case SOCKET_ERROR:
        if (chatState.isInitialized()) {
          chatState = ChatState.WAITING_SOCKET;
          view.onError();
        }
        break;

      case SOCKET_DISCONNECTED:
        if (chatState.isInitialized()) {
          chatState = ChatState.WAITING_SOCKET;

          if (object != null && object instanceof Boolean && (Boolean) object) {
            view.onError();
          }
        }
        break;

      case READY:
        view.onTryRefresh();
        checkSocketState();
        break;

      case JOINED:
        if (object != null &&
            object instanceof String &&
            CompareUtils.isSame(chatId, (String) object) &&
            chatState == ChatState.CHAT_JOINING) {

          chatState = ChatState.CHAT_JOINED;

          view.onTryRefresh();
          fetchUserChat();
        }
        break;
    }
  }

  @Override
  public void receiveStoreEvent(StoreType storeType, UpdateType updateType, @Nullable Entity entity) {
    switch (storeType) {
      case USER:
      case VEIL:
        if (entity != null && entity instanceof Guest) {
          Guest guest = GuestSelector.get();
          Guest newGuest = (Guest) entity;

          if (guest != null && newGuest.isOwnedBy(guest.getPersonType(), guest.getPersonId())) {
            view.refreshChatCount(false);
          }
        }
        break;

      case CHANNEL:
        if (entity != null && entity instanceof Channel && updateType == UpdateType.UPDATE) {
          view.bindChannel((Channel) entity);
        }
        break;

      case USER_CHAT:
        if (entity != null && entity instanceof UserChat) {
          UserChat userChat = (UserChat) entity;

          if (chatId != null && CompareUtils.isSame(chatId, userChat.getId())) {
            view.onUserChatChange(this.userChat, userChat);
            onUserChatChange(this.userChat, userChat);

            this.userChat = userChat;

            ChatManager.get().read(chatId);
          }
        }
        break;

      case MESSAGE:
        if (entity != null && entity instanceof Message && updateType == UpdateType.UPDATE) {
          Message message = (Message) entity;
          if (chatId != null && chatId.equals(message.getChatId())) {
            if (chatState == ChatState.INIT_MESSAGES_LOADING) {
              receiveMessagesWaitingQueue.add(message);
            } else if (chatState == ChatState.CHAT_READY) {
              adapterModel.addMessage(message);

              if (!CompareUtils.exists(message.getPersonType(), Const.VEIL, Const.USER)) {
                view.onHostMessageArrive(message);
              }
            }
          }
        }
        break;

      case PUSH_BOT_ITEM:
        view.refreshMenu(isLocalChat());
        view.refreshChatCount(true);
        break;
    }
  }

  @Override
  public void receiveTyping(Typing typing) {
    if (typingManager.isReady() && typing != null && typing.isOther() && typing.isSameChat(chatId, UserChat.CLASSNAME)) {
      typingManager.refreshTypingExpiresAt(typing);

      switch (typing.getAction()) {
        case "start":
          adapterModel.upsertTyping(typing);
          break;

        case "stop":
          adapterModel.removeTyping(typing);
          break;
      }
    }
  }

  // Send functions

  @Override
  public void sendText(String text) {
    SendingMessageItem item = new SendingMessageItem(chatId).setText(text);
    adapterModel.addMessageItem(item);
    sendMessages(Collections.singletonList(item));
  }

  @Override
  public void sendFiles(@Nullable ArrayList<String> paths) {
    List<SendingMessageItem> items = new ArrayList<>();
    if (paths != null) {
      for (String filePath : paths) {
        SendingMessageItem item = new SendingMessageItem(chatId).setFilePath(filePath);
        adapterModel.addMessageItem(item);
        items.add(item);
      }
    }
    sendMessages(items);
  }

  private void sendMessages(List<SendingMessageItem> items) {
    view.scrollToBottom();

    if (chatId != null) {
      ChatManager.get().send(items);
    } else {
      sendingMessagesWaitingQueue.addAll(items);

      switch (chatState) {
        case LOCAL_WELCOME:
          createUserChat();
          break;
      }
    }
  }

  @Override
  public void sendForm(@Nullable String actionType, @Nullable ActionInput actionInput) {
    // Reopen chat
    if (CompareUtils.exists(actionType, Const.FORM_ACTION_TYPE_SOLVE, Const.FORM_ACTION_CLOSE) &&
        actionInput != null &&
        Const.FORM_ACTION_REOPEN.equals(actionInput.getKey())) {
      reopenChat(actionInput);
    } else {
      SendingMessageItem item = new SendingMessageItem(chatId).setActionInput(actionType, actionInput);
      adapterModel.addMessageItem(item);

      view.scrollToBottom();

      // Support bot initialize
      if (Const.FORM_ACTION_TYPE_SUPPORT_BOT.equals(actionType) && chatId == null) {
        sendingMessagesWaitingQueue.add(item);
        createSupportBotUserChat(item);
      }
      // Other
      else if (chatId != null) {
        ChatManager.get().send(item);
      }
    }
  }

  @Override
  public void resend(@Nullable SendingMessageItem item) {
    if (item != null) {
      cancelSend(item);

      // Renew created at for send to bottom
      SendingMessageItem newItem = SendingMessageItem.copyContentFrom(item);
      adapterModel.addMessageItem(newItem);

      // Remove old item
      adapterModel.removeMessageItem(item);
      sendingMessagesWaitingQueue.remove(item);

      if (chatId != null) {
        ChatManager.get().send(newItem);
      } else {
        sendingMessagesWaitingQueue.add(newItem);

        switch (chatState) {
          case LOCAL_WELCOME:
            createUserChat();
            break;

          case LOCAL_SUPPORTING:
            createSupportBotUserChat(item);
            break;
        }
      }
    }
  }

  @Override
  public void cancelSend(@Nullable SendingMessageItem item) {
    if (item != null) {
      ChatManager.get().removeSendingFailedItem(item.getPrimaryKey());
      adapterModel.removeMessageItem(item);
    }
  }

  // Message sending callbacks

  @Override
  public void onSendSuccess(Message message) {
    adapterModel.addMessage(message);
  }

  @Override
  public void onSendFail(SendingMessageItem item) {
    adapterModel.addMessageItem(item);
  }

  // User chat create function

  private void createUserChat() {
    Plugin plugin = PluginSelector.getPlugin();

    if (plugin != null && plugin.getId() != null) {
      view.showProgress(ResUtils.getString("ch.loading_information"));

      ChannelApiManager.call(
          ChannelApiManager.get().createUserChat(plugin.getId()),
          new RestSubscriber<UserChatWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              view.hideProgress();
              for (SendingMessageItem item : sendingMessagesWaitingQueue) {
                item.setSending(false);
              }
              adapterModel.addMessageItems(sendingMessagesWaitingQueue);

              L.e(error.getMessage());
            }

            @Override
            public void onSuccess(@NonNull UserChatWrapper userChatRepo) {
              view.hideProgress();
              userChatRepo.update();

              if (userChatRepo.getMessage() != null) {
                adapterModel.replaceInitMessage(userChatRepo.getMessage());
              }

              if (userChatRepo.getUserChat() != null) {
                UserChat oldUserChat = userChat;

                chatId = userChatRepo.getUserChat().getId();
                userChat = userChatRepo.getUserChat();
                session = userChatRepo.getSession();

                view.onUserChatChange(oldUserChat, userChatRepo.getUserChat());
                onUserChatChange(oldUserChat, userChatRepo.getUserChat());

                chatState = ChatState.WAITING_SOCKET;
                checkSocketState();
              }
            }
          }
      );
    }
  }

  private void createSupportBotUserChat(final SendingMessageItem item) {
    SupportBotEntry supportBotEntry = SupportBotSelector.getSupportBotEntry();
    SupportBot supportBot = supportBotEntry != null ? supportBotEntry.getSupportBot() : null;

    if (supportBot != null && supportBot.getId() != null) {
      view.showProgress(ResUtils.getString("ch.loading_information"));

      ChannelApiManager.call(
          ChannelApiManager.get().createSupportBotUserChat(supportBot.getId()),
          new RestSubscriber<UserChatWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              view.hideProgress();
              for (SendingMessageItem item : sendingMessagesWaitingQueue) {
                item.setSending(false);
              }
              adapterModel.addMessageItems(sendingMessagesWaitingQueue);

              L.e(error.getMessage());
            }

            @Override
            public void onSuccess(@NonNull UserChatWrapper userChatRepo) {
              view.hideProgress();
              userChatRepo.update();

              if (userChatRepo.getMessage() != null) {
                adapterModel.replaceInitMessage(userChatRepo.getMessage());
                item.updateMessageOnActionInput(userChatRepo.getMessage());
              }

              if (userChatRepo.getUserChat() != null) {
                UserChat oldUserChat = userChat;

                RxBus.post(new ChannelModelBus(userChatRepo.getUserChat(), true));
                RxBus.post(new ChannelModelBus(userChatRepo.getSession(), true));

                chatId = userChatRepo.getUserChat().getId();
                userChat = userChatRepo.getUserChat();
                session = userChatRepo.getSession();

                view.onUserChatChange(oldUserChat, userChatRepo.getUserChat());
                onUserChatChange(oldUserChat, userChatRepo.getUserChat());

                chatState = ChatState.WAITING_SOCKET;
                checkSocketState();
              }
            }
          }
      );
    }
  }

  @Override
  public void createPushBotUserChat() {
    if (pushBotId != null) {
      view.showProgress(ResUtils.getString("ch.loading_information"));

      ChannelApiManager.call(
          ChannelApiManager.get().createPushBotUserChat(pushBotId),
          new RestSubscriber<UserChatWrapper>() {
            @Override
            public void onError(RetrofitException error) {
              view.hideProgress();
              view.setPushBotSaveButtonVisibility(true);
              for (SendingMessageItem item : sendingMessagesWaitingQueue) {
                item.setSending(false);
              }
              adapterModel.addMessageItems(sendingMessagesWaitingQueue);

              L.e(error.getMessage());
            }

            @Override
            public void onSuccess(@NonNull UserChatWrapper userChatRepo) {
              view.hideProgress();
              userChatRepo.update();

              // 채팅이 만들어졌으니 리스트에서 삭제 처리 (실제로 지워지지는 않음)
              if (pushBotId != null) {
                PushBotItem pushBotItem = PushBotSelector.get(pushBotId);
                if (pushBotItem != null) {
                  pushBotItem.remove();
                  Store.getInstance(PushBotItemStore.class).add(pushBotItem);
                }
              }

              if (userChatRepo.getMessage() != null) {
                adapterModel.replaceInitMessage(userChatRepo.getMessage());
              }

              if (userChatRepo.getUserChat() != null) {
                UserChat oldUserChat = userChat;

                chatId = userChatRepo.getUserChat().getId();
                userChat = userChatRepo.getUserChat();
                session = userChatRepo.getSession();

                view.onUserChatChange(oldUserChat, userChatRepo.getUserChat());
                onUserChatChange(oldUserChat, userChatRepo.getUserChat());

                chatState = ChatState.WAITING_SOCKET;
                checkSocketState();

                // keep 성공 시 '👍' 메세지가 response 로 전달됨.
                keepPushBot();
              }
            }
          }
      );
    }
  }

  // Profile bot

  @Override
  public void fetchCountries(final ProfileBotMessageItem item, final String key, final MobileNumber mobileNumber) {
    if (CountrySelector.isEmpty()) {
      view.showProgress(ResUtils.getString("ch.loading_information"));

      ChannelApiManager.call(
          ChannelApiManager.get().getCountries(),
          new RestSubscriber<List<Country>>() {
            @Override
            public void onError(RetrofitException error) {
              view.hideProgress();
            }

            @Override
            public void onSuccess(@NonNull List<Country> countries) {
              Store.getInstance(CountryStore.class).set(countries);

              view.hideProgress();
              view.onCountriesFetch(item, key, mobileNumber);
            }
          }
      );
    } else {
      view.onCountriesFetch(item, key, mobileNumber);
    }
  }

  @Override
  public void selectCountry(ProfileBotMessageItem item, String key, MobileNumber mobileNumber, String countryCode) {
    mobileNumber.setCountryCode(Integer.valueOf(countryCode));
    item.setInputState(key, mobileNumber, false, null);
    adapterModel.addMessageItem(item);
  }

  @Override
  public void updateProfile(final ProfileBotMessageItem item, final String key, Object value) {
    item.setInputState(key, item.getInputState(), true, null);
    adapterModel.addMessageItem(item);

    RequestBody requestBody = RequestUtils.form().set(key, value).create();
    ChannelApiManager.call(
        ChannelApiManager.get().updateProfileBot(item.getMessage().getId(), requestBody),
        new RestSubscriber<MessageWrapper>() {

          @Override
          public void onError(RetrofitException error) {
            item.setInputState(key, item.getInputState(), false, ResUtils.getString("ch.profile_form.error"));
            adapterModel.addMessageItem(item);
            view.tryScrollToBottom();
          }

          @Override
          public void onSuccess(@NonNull MessageWrapper messageRepo) {
            messageRepo.update();

            item.setMessage(messageRepo.getMessage());
            item.resetInputState();
            adapterModel.addMessageItem(item);
            view.tryScrollToBottom();
          }
        }
    );
  }

  // Translation

  @Override
  public void translateMessage(@NonNull final Message message, String targetLanguage) {
    if (chatId != null) {
      final String uniqueId = TranslationInfo.createId(message.getId(), targetLanguage);
      final TranslationInfo translationInfo = TranslationSelector.get(uniqueId);

      if (translationInfo == null) {
        //For showing Progress view
        Store.getInstance(TranslationStore.class).add(TranslationInfo.createProgressStateInfo(uniqueId));
        adapterModel.addMessage(message);

        ChannelApiManager.call(
            ChannelApiManager.get().getTranslatedMessage(message.getId(), targetLanguage),
            new RestSubscriber<TranslationRepo>() {

              @Override
              public void onError(RetrofitException error) {
                Store.getInstance(TranslationStore.class).add(TranslationInfo.createOriginStateInfo(uniqueId));
                adapterModel.addMessage(message);
              }

              @Override
              public void onSuccess(@NonNull TranslationRepo repo) {
                if (repo.getTranslatedMessage() != null) {
                  TranslationInfo info = new TranslationInfo(uniqueId, repo.getTranslatedMessage(), TranslationState.TRANSLATED);
                  Store.getInstance(TranslationStore.class).add(info);
                } else {
                  Store.getInstance(TranslationStore.class).add(TranslationInfo.createOriginStateInfo(uniqueId));
                }
                adapterModel.addMessage(message);
              }
            }
        );
      } else {
        translationInfo.setState(TranslationState.ORIGIN.equals(translationInfo.getState())
            ? TranslationState.TRANSLATED
            : TranslationState.ORIGIN);

        Store.getInstance(TranslationStore.class).add(translationInfo);
        adapterModel.addMessage(message);
      }
    }
  }

  // public getter functions

  @Override
  public boolean hasBackwardMessages() {
    return backwardId != null;
  }

  @Override
  public boolean isLocalChat() {
    return chatId == null;
  }

  // private bind functions

  /**
   * 유저챗 정보가 생성 또는 갱신되었을 때 불리는 함수
   *
   * @param oldUserChat 이전의 유저챗 정보, 처음 불렸을 경우 null일 수 있고, 이전 상태를 알 수 없을 경우 <code>newUserChat</code>과 동일할 수 있음
   * @param newUserChat 새롭게 갱신되는 유저챗 정보
   */
  private void onUserChatChange(@Nullable UserChat oldUserChat, @NonNull UserChat newUserChat) {
    if (!typingManager.isReady()) {
      typingManager.setChatId(newUserChat.getId());
    }
    if (oldUserChat != null &&
        CompareUtils.exists(oldUserChat.getState(), Const.USER_CHAT_STATE_READY, Const.USER_CHAT_STATE_SUPPORTING) &&
        Const.USER_CHAT_STATE_OPEN.equals(newUserChat.getState())) {
      requestProfileBot();
    }
  }

  // private functions

  private void requestProfileBot() {
    Plugin plugin = PluginSelector.getPlugin();

    if (plugin != null && !TextUtils.isEmpty(chatId)) {
      ChannelApiManager.call(
          ChannelApiManager.get().requestProfileBot(chatId, plugin.getId()),
          new RestSubscriber<Void>());
    }
  }

  private void reopenChat(@Nullable ActionInput actionInput) {
    if (userChat != null && actionInput != null) {
      Message message = actionInput.getMessage();
      message.clearForm();

      adapterModel.addMessage(message);

      userChat.setState(Const.USER_CHAT_STATE_CONTINUE_CHAT);
      view.onUserChatChange(userChat, userChat);
    }
  }

  private void keepPushBot() {
    if (chatId != null) {
      ChannelApiManager.call(
          ChannelApiManager.get().keepPushBot(chatId),
          new RestSubscriber<MessageWrapper>() {
            @Override
            public void onSuccess(@NonNull MessageWrapper messageWrapper) {
              adapterModel.addMessage(messageWrapper.getMessage());
              view.scrollToBottom();
            }
          });
    }
  }

  private void cancelApis() {
    ChannelApiManager.cancel(ApiTag.FETCH_USER_CHAT);
    ChannelApiManager.cancel(ApiTag.FETCH_INIT_MESSAGES);
    ChannelApiManager.cancel(ApiTag.FETCH_BACKWARD_MESSAGES);
  }
}
