package com.zoyi.channel.plugin.android;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.os.Build;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.RelativeLayout;
import com.zoyi.channel.plugin.android.enumerate.StoreType;
import com.zoyi.channel.plugin.android.event.CommandBus;
import com.zoyi.channel.plugin.android.event.PushBus;
import com.zoyi.channel.plugin.android.event.RxBus;
import com.zoyi.channel.plugin.android.event.StoreBus;
import com.zoyi.channel.plugin.android.global.PrefSupervisor;
import com.zoyi.channel.plugin.android.model.entity.Guest;
import com.zoyi.channel.plugin.android.model.entity.ProfileEntity;
import com.zoyi.channel.plugin.android.model.rest.UserChat;
import com.zoyi.channel.plugin.android.selector.ChannelIOSelector;
import com.zoyi.channel.plugin.android.selector.ColorSelector;
import com.zoyi.channel.plugin.android.selector.CounterSelector;
import com.zoyi.channel.plugin.android.util.AlarmCenter;
import com.zoyi.channel.plugin.android.util.CompareUtils;
import com.zoyi.channel.plugin.android.util.Initializer;
import com.zoyi.channel.plugin.android.util.Utils;
import com.zoyi.channel.plugin.android.util.message_format.MessageParser;
import com.zoyi.channel.plugin.android.view.external.fab.FloatingActionButton;
import com.zoyi.channel.plugin.android.view.handler.SimpleAnimationListener;
import com.zoyi.channel.plugin.android.view.layout.AvatarLayout;
import com.zoyi.channel.plugin.android.view.layout.CHTextView;
import com.zoyi.rx.Subscription;
import com.zoyi.rx.android.schedulers.AndroidSchedulers;
import com.zoyi.rx.functions.Action1;

public class ChannelView extends RelativeLayout {
  public static final int TOP_LEFT = 0;
  public static final int TOP_RIGHT = 1;
  public static final int BOTTOM_LEFT = 2;
  public static final int BOTTOM_RIGHT = 3;

  private Context context;

  private View pushLayout;
  private AvatarLayout avatarLayout;
  private View reviewAvatar;
  private CHTextView pushName, pushMessage;
  private View fabLayout;
  private CHTextView badge;
  private FloatingActionButton fab;

  private Animation pushAnimation;
  @Nullable
  private Subscription subscription;
  @Nullable
  private UserChat userChat;

  private int gravity;
  private int topMargin, bottomMargin, leftMargin, rightMargin;

  public ChannelView(Context context) {
    super(context);
    init(context, null);
  }

  public ChannelView(Context context, AttributeSet attrs) {
    super(context, attrs);
    init(context, attrs);
  }

  public ChannelView(Context context, AttributeSet attrs, int defStyleAttr) {
    super(context, attrs, defStyleAttr);
    init(context, attrs);
  }

  @Initializer
  private void init(final Context context, @Nullable AttributeSet attrs) {
    this.context = context;

    View view = LayoutInflater.from(context).inflate(R.layout.ch_plugin_layout_channel_view, this);
    fabLayout = view.findViewById(R.id.fab_layout);
    badge = view.findViewById(R.id.text_badge);
    fab = (FloatingActionButton) view.findViewById(R.id.fab);
    fab.setOnClickListener(new OnClickListener() {
      @Override
      public void onClick(View v) {
        ChannelIO.open(context);
      }
    });

    int defaultMargin = context.getResources().getDimensionPixelSize(R.dimen.ch_default_channel_button_margin);

    if (attrs != null) {
      final TypedArray typedArray = getContext().obtainStyledAttributes(attrs, R.styleable.ChannelView, 0, 0);
      try {
        gravity = typedArray.getInteger(R.styleable.ChannelView_channel_button_gravity, BOTTOM_RIGHT);

        topMargin = typedArray.getDimensionPixelSize(R.styleable.ChannelView_channel_button_top_margin, defaultMargin);
        bottomMargin = typedArray.getDimensionPixelSize(R.styleable.ChannelView_channel_button_bottom_margin, defaultMargin);
        leftMargin = typedArray.getDimensionPixelSize(R.styleable.ChannelView_channel_button_left_margin, defaultMargin);
        rightMargin = typedArray.getDimensionPixelSize(R.styleable.ChannelView_channel_button_right_margin, defaultMargin);

        setTopMargin(topMargin);
        setBottomMargin(bottomMargin);
        setLeftMargin(leftMargin);
        setRightMargin(rightMargin);

        setButtonGravity(gravity);

      } finally {
        if (typedArray != null) {
          typedArray.recycle();
        }
      }
    }

    pushLayout = view.findViewById(R.id.push_layout);
    avatarLayout = (AvatarLayout) view.findViewById(R.id.push_avatar);
    reviewAvatar = view.findViewById(R.id.iv_review_avatar);
    pushName = view.findViewById(R.id.push_name);
    pushMessage = view.findViewById(R.id.push_message);

    pushAnimation = AnimationUtils.loadAnimation(getContext(), R.anim.ch_plugin_push);
    pushAnimation.setAnimationListener(new SimpleAnimationListener() {
      @Override
      public void onAnimationStart(Animation animation) {
        pushLayout.setVisibility(VISIBLE);
      }

      @Override
      public void onAnimationEnd(Animation animation) {
        hidePush();
      }
    });

    view.findViewById(R.id.iv_push_close).setOnClickListener(new OnClickListener() {
      @Override
      public void onClick(View v) {
        hidePush();
      }
    });

    pushLayout.setOnClickListener(new OnClickListener() {
      @Override
      public void onClick(View v) {
        String chatId = userChat != null ? userChat.getId() : null;
        ChannelIO.openChat(context, chatId);
      }
    });
  }

  public void setButtonGravity(int gravity) {
    if (!CompareUtils.exists(gravity, TOP_LEFT, TOP_RIGHT, BOTTOM_LEFT, BOTTOM_RIGHT)) {
      return;
    }

    this.gravity = gravity;
    LayoutParams params = (LayoutParams) fabLayout.getLayoutParams();

    switch (gravity) {
      case TOP_LEFT:
      case TOP_RIGHT:
        params.addRule(RelativeLayout.ALIGN_PARENT_TOP);
        params.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM, 0);
        break;

      case BOTTOM_LEFT:
      case BOTTOM_RIGHT:
        params.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        params.addRule(RelativeLayout.ALIGN_PARENT_TOP, 0);
        break;
    }

    switch (gravity) {
      case TOP_LEFT:
      case BOTTOM_LEFT:
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
          params.addRule(RelativeLayout.ALIGN_PARENT_START);
          params.addRule(RelativeLayout.ALIGN_PARENT_END, 0);
        }
        params.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
        params.addRule(RelativeLayout.ALIGN_PARENT_RIGHT, 0);
        break;

      case TOP_RIGHT:
      case BOTTOM_RIGHT:
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
          params.addRule(RelativeLayout.ALIGN_PARENT_START, 0);
          params.addRule(RelativeLayout.ALIGN_PARENT_END);
        }
        params.addRule(RelativeLayout.ALIGN_PARENT_LEFT, 0);
        params.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
        break;
    }

    fabLayout.setLayoutParams(params);
    fabLayout.requestLayout();
    badge.bringToFront();
  }

  public void setTopMargin(int topMargin) {
    this.topMargin = topMargin;
    refreshSize();
  }

  public void setBottomMargin(int bottomMargin) {
    this.bottomMargin = bottomMargin;
    refreshSize();
  }

  public void setLeftMargin(int leftMargin) {
    this.leftMargin = leftMargin;
    refreshSize();
  }

  public void setRightMargin(int rightMargin) {
    this.rightMargin = rightMargin;
    refreshSize();
  }

  private void refreshSize() {
    LayoutParams params = (LayoutParams) fabLayout.getLayoutParams();

    switch (gravity) {
      case TOP_LEFT:
        params.setMargins(leftMargin, topMargin, 0, 0);
        break;

      case TOP_RIGHT:
        params.setMargins(0, topMargin, rightMargin, 0);
        break;

      case BOTTOM_LEFT:
        params.setMargins(leftMargin, 0, 0, bottomMargin);
        break;

      case BOTTOM_RIGHT:
        params.setMargins(0, 0, rightMargin, bottomMargin);
        break;
    }

    fabLayout.setLayoutParams(params);
    fabLayout.requestLayout();
    badge.bringToFront();
  }

  @Override
  protected void onAttachedToWindow() {
    super.onAttachedToWindow();

    initView();

    subscription = RxBus.observable()
        .subscribe(new Action1<Object>() {
          @Override
          public void call(Object o) {
            if (o instanceof CommandBus) {
              CommandBus commandBus = (CommandBus) o;
              switch (commandBus.getCommand()) {
                case SHOW_CHANNEL_VIEW:
                  showLauncherButton();
                  break;

                case HIDE_CHANNEL_VIEW:
                  hideLauncherButton();
                  break;

                case BOOT:
                  initView();
                  break;

                case SHUTDOWN:
                  shutdown();
                  break;
              }
            }

            if (o instanceof PushBus) {
              PushBus pushBus = (PushBus) o;

              if (!ChannelIOManager.isMainRunning() && ChannelIOSelector.shouldShowInAppPush()) {
                showPush(pushBus.getProfileEntity(), pushBus.getMessage(), pushBus.getUserChat());
              }
            }

            if (o instanceof StoreBus) {
              StoreBus storeBus = (StoreBus) o;

              if (CompareUtils.exists(storeBus.getStoreType(), StoreType.USER, StoreType.VEIL) &&
                  storeBus.getEntity() != null &&
                  storeBus.getEntity() instanceof Guest) {
                Guest guest = (Guest) storeBus.getEntity();
                setBadge(guest.getAlert());
              }
            }
          }
        });
  }

  private void initView() {
    if (ChannelIOSelector.isBooted()) {
      setVisibility(VISIBLE);
      setLauncherStyle();

      if (ChannelIOSelector.shouldShowDefaultLauncherOnInit()) {
        showLauncherButton();
      }
    }
  }

  @Override
  protected void onDetachedFromWindow() {
    super.onDetachedFromWindow();
    if (subscription != null && !subscription.isUnsubscribed()) {
      subscription.unsubscribe();
    }
  }

  private void setLauncherStyle() {
    fab.setColorNormal(ColorSelector.getBackgroundColor());
    fab.setColorPressed(Utils.addBlackMask(ColorSelector.getBackgroundColor()));
    fab.setColorRipple(Color.WHITE);
    fab.setImageResource(
        ColorSelector.getTextColor() == Color.BLACK
            ? R.drawable.ch_plugin_balloon_black
            : R.drawable.ch_plugin_balloon_white);
  }

  private void showLauncherButton() {
    if (ChannelIOSelector.isBooted()) {
      refreshCount();
      fabLayout.setVisibility(VISIBLE);
    }
  }

  private void hideLauncherButton() {
    fabLayout.setVisibility(GONE);
  }

  public void refreshCount() {
    setBadge(CounterSelector.getTotalCounter());
  }

  private void setBadge(int count) {
    if (count <= 0) {
      badge.setVisibility(GONE);
    } else {
      badge.setVisibility(VISIBLE);
      badge.setText(Utils.getCount(count, true));
    }
  }

  private void shutdown() {
    setVisibility(GONE);
    hideLauncherButton();
  }

  private void showPush(ProfileEntity person, String message, @Nullable UserChat userChat) {
    this.userChat = userChat;

    if (userChat != null) {
      if (userChat.isStateCompleted()) {
        avatarLayout.setVisibility(View.GONE);
        reviewAvatar.setVisibility(View.VISIBLE);

        pushName.setTextByKey("ch.review.title.preview");
        pushMessage.setTextByKey("ch.review.require.preview");
      } else {
        reviewAvatar.setVisibility(View.GONE);
        avatarLayout.setVisibility(View.VISIBLE);
        avatarLayout.set(person);
        avatarLayout.setTextSize(14);

        if (person != null && person.getName() != null) {
          pushName.setText(person.getName());
        } else {
          pushName.setTextByKey("ch.unknown");
        }

        if (MessageParser.get() != null) {
          pushMessage.setText(MessageParser.get().parse(message));
        } else {
          pushMessage.setText(message);
        }
      }

      pushLayout.setVisibility(VISIBLE);
      pushLayout.startAnimation(pushAnimation);

      if (PrefSupervisor.isEnabledPushAlarm(context)) {
        AlarmCenter.alarm(context);
      }
    }
  }

  private void hidePush() {
    this.userChat = null;

    pushLayout.clearAnimation();
    pushLayout.setVisibility(GONE);
  }
}
