package com.zhuge.analysis.stat;

import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;
import android.util.Base64;

import com.zhuge.analysis.util.ActivityServicesUtils;
import com.zhuge.analysis.util.HttpServices;
import com.zhuge.analysis.util.ZGJSONObject;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.Deflater;

/**
 * 会话事件管理
 * Created by jiaokang on 15/11/9.
 */
/*package*/ class EventStore {

    private static final Map<Context, EventStore> sInstances = new HashMap<Context, EventStore>();


    /**
     * 事件管理
     */
    /*package*/ static final int MESSAGE_START_SESSION = 0;
    /*package*/ static final int MESSAGE_COMPLETE_SESSION = 1;
    /*package*/ static final int MESSAGE_FLUSH = 2;
    /*package*/ static final int MESSAGE_INIT_INFO = 3;
    /*package*/ static final int MESSAGE_INIT_ACCOUNTS = 4;
    /*package*/ static final int MESSAGE_INIT_APPLIST = 5;
    /*package*/ static final int MESSAGE_CUSTOM_EVENT = 6;
    /*package*/ static final int MESSAGE_IDENTIFY_EVENT = 7;
    /*package*/ static final int MESSAGE_PUSH_MSG = 8;
    /*package*/ static final int MESSAGE_CHECK_ACTIVE = 9;
    /*package*/ static final int MESSAGE_WILL_EXIT = 10;


    private EventHandler eventHandler;
    private ZhugeConfig config;

    /*
     事件存储key
     */
    private static final String sessionEventCount = "session_event_count";

    private final Context mContext;

    private EventStore(Context context, ZhugeConfig zhugeConfig) {
        final HandlerThread thread = new HandlerThread(EventStore.class.getCanonicalName());
        thread.setPriority(android.os.Process.THREAD_PRIORITY_BACKGROUND);
        thread.start();
        mContext = context;
        config = zhugeConfig;
        eventHandler = new EventHandler(thread.getLooper());
    }

    /*package*/
    static EventStore getInstance(Context context, ZhugeConfig zhugeConfig) {

        synchronized (sInstances) {
            final Context appContext = context.getApplicationContext();
            EventStore ret;
            if (!sInstances.containsKey(appContext)) {
                ret = new EventStore(appContext, zhugeConfig);
                sInstances.put(appContext, ret);
            } else {
                ret = sInstances.get(appContext);
            }
            return ret;
        }
    }


    /*package*/ void addEventToHandler(ZGJSONObject info, int message_state) {
        Message message = eventHandler.obtainMessage(message_state);
        message.obj = info;
        message.sendToTarget();
    }

    /*package*/int getEventCount() {
        return config.getGlobalSP().getInt(sessionEventCount, 0);
    }


    private class EventHandler extends Handler {


        private final int mFlushInterval;
        private final String myPackageName;
        private boolean inCheck = false;

        //今日发送事件数
        private int todayTotal = 0;

        //本地存储事件数
        private int localSize = 0;

        //当次会话事件数
        private int sessionCount = 0;


        private ZhugeDbAdapter mDb;
        private ActivityServicesUtils asUtils;
        private HttpServices httpServices;

        public EventHandler(Looper looper) {
            super(looper);
            mFlushInterval = ZhugeConfig.getPeriod_time() * 1000;
            asUtils = new ActivityServicesUtils(mContext);
            myPackageName = mContext.getPackageName();
            String todayInfo = config.getGlobalSP().getString(config.getTotaldayKey(), "");
            if (!"".equals(todayInfo)) {
                String[] toInfo = todayInfo.split("\\|");
                if (System.currentTimeMillis() / 1000 / 86400 - Integer.parseInt(toInfo[0]) != 0) {
                    todayTotal = 0;
                } else {
                    todayTotal = Integer.parseInt(toInfo[1]);
                }
            }
            mDb = null;
        }


        @Override
        public void handleMessage(Message msg) {
            if (inCheck && hasMessages(MESSAGE_WILL_EXIT)) {
                removeMessages(MESSAGE_WILL_EXIT);
                inCheck = false;
                config.debug("重新活跃，移除退出检测");
            }
            if (null == mDb) {
                mDb = new ZhugeDbAdapter(mContext);
            }
            switch (msg.what) {
                case MESSAGE_COMPLETE_SESSION:
                    ZGJSONObject info = (ZGJSONObject) msg.obj;
                    addEvent(info);
                    break;
                case MESSAGE_START_SESSION:
                    ZGJSONObject seStart = (ZGJSONObject) msg.obj;
                    addEvent(seStart);
                    updateSessionTime(false);
                    break;
                case MESSAGE_FLUSH:
                    ZGJSONObject flush = (ZGJSONObject) msg.obj;
                    updateSessionTime(true);
                    addEvent(flush);
                    flushEvent(true);
                    if (hasMessages(MESSAGE_CHECK_ACTIVE)) {
                        removeMessages(MESSAGE_CHECK_ACTIVE);
                    }
                    config.debug("flush，退出会话");
                    return;
                case MESSAGE_INIT_INFO:
                    ZGJSONObject initInfo = (ZGJSONObject) msg.obj;
                    addEvent(initInfo);
                    updateInitInfo();
                    break;
                case MESSAGE_INIT_ACCOUNTS:
                    ZGJSONObject init_acc = (ZGJSONObject) msg.obj;
                    addEvent(init_acc);
                    updateInitAccount();
                    break;
                case MESSAGE_INIT_APPLIST:
                    ZGJSONObject init_app = (ZGJSONObject) msg.obj;
                    addEvent(init_app);
                    updateInitPkg();
                    break;
                case MESSAGE_CUSTOM_EVENT:
                    ZGJSONObject customEvent = (ZGJSONObject) msg.obj;
                    addEvent(customEvent);
                    updateSessionTime(false);
                    break;
                case MESSAGE_IDENTIFY_EVENT:
                    ZGJSONObject identify = (ZGJSONObject) msg.obj;
                    addEvent(identify);
                    flushEvent(true);
                    updateCid(identify);
                    break;
                case MESSAGE_PUSH_MSG:
                    ZGJSONObject channel = (ZGJSONObject) msg.obj;
                    addEvent(channel);
                    flushEvent(true);
                    break;
                case MESSAGE_CHECK_ACTIVE:
                    updateSessionTime(false);
                    if (!checkActive()) {
                        inCheck = true;
                        config.debug("不在活跃状态，发送退出检测，30秒后进行退出检测");
                        sendEmptyMessageDelayed(MESSAGE_WILL_EXIT, 30 * 1000);
                        return;
                    } else {
                        flushEvent(false);
                        break;
                    }
                case MESSAGE_WILL_EXIT:
                    inCheck = false;
                    if (checkActive()) {
                        //还在活跃之中，继续轮询检查
                        config.debug("退出检测不成立");
                        break;
                    } else {
                        //用户退出，会话结束，清除数据
                        if (hasMessages(MESSAGE_CHECK_ACTIVE)) {
                            removeMessages(MESSAGE_CHECK_ACTIVE);
                        }
                        ZGJSONObject json = config.completeLastSession(0, getEventCount());
                        updateSessionTime(true);
                        addEvent(json);
                        flushEvent(true);
                        config.debug("退出检测成立，退出会话");
                        return;
                    }
                default:
                    break;
            }
            if (!inCheck && !hasMessages(MESSAGE_CHECK_ACTIVE)) {
                sendEmptyMessageDelayed(MESSAGE_CHECK_ACTIVE, mFlushInterval);
            }
        }


        private boolean checkActive() {
            String presentPackage = asUtils.getForegroundPackage();
            return presentPackage.contains(myPackageName);

        }

        private void updateInitInfo() {
            config.getGlobalSP().edit().putLong(config.getConfig_info(), System.currentTimeMillis())
                    .apply();
        }

        private void updateInitAccount() {
            config.getGlobalSP().edit().putLong(config.getConfig_account(), System.currentTimeMillis())
                    .apply();
        }

        private void updateInitPkg() {
            config.getGlobalSP().edit().putLong(config.getConfig_pkg(), System.currentTimeMillis())
                    .apply();
        }

        private void updateCid(ZGJSONObject identify) {
            try {
                String uid = identify.getString("cuid");
                config.getGlobalSP().edit().putString(config.getConfig_Cuid(), uid).apply();
            } catch (JSONException e) {
                config.debug("更新用户ID出错：" + identify.toString() + "\n" + e.getMessage());
            }
        }


        private void updateSessionTime(boolean exit) {
            if (exit) {
                config.preSessionEnd = -1;
                config.getGlobalSP().edit().putString(config.getLastSession(), "").apply();
                return;
            }
            long now = System.currentTimeMillis();
            config.preSessionEnd = now;
            String updateSession = config.getSessionTime() + "|" + now;
            config.getGlobalSP().edit().putString(config.getLastSession(), updateSession).apply();
        }


        private void addEvent(ZGJSONObject info) {
            if (localSize >= 400) {
                return;
            }
            config.getGlobalSP().edit().putInt(sessionEventCount, ++sessionCount).apply();
            localSize = mDb.addEvent(info);
        }


        private void flushEvent(boolean flush) {

            if (!ZhugeConfig.debug) {
                if (!flush && localSize < 5) {
                    return;
                }
                if (!HttpServices.isOnline(mContext)) {
                    config.debug("网络链接不可用，退出发送。");
                    return;
                }
                if (todayTotal > config.getUpload_per_day()) {
                    config.debug("数据超过每日上传限制。");
                }
            }
            if (httpServices == null) {
                httpServices = new HttpServices();
            }
            String[] mDbData = mDb.getData();
            if (null == mDbData) {
                return;
            }
            Map<String, Object> postMap = new HashMap<String, Object>();
            JSONObject postData = config.wrapData(mDbData[1]);
            String data = Base64.encodeToString(compress(postData.toString().getBytes()),
                    Base64.DEFAULT).replace("\r", "").replace("\n", "");
            postMap.put("method", "event_statis_srv.upload");
            postMap.put("compress", "1");
            postMap.put("event", data);
            byte[] returnByte = httpServices.requestApi(ZhugeConfig.API_PATH, postMap, true);
            JSONObject responseDict = null;
            try {
                responseDict = new JSONObject(new String(returnByte, "UTF-8"));
            } catch (Exception e) {
                config.debug("返回数据无法正确处理.");
            }
            if (null != responseDict) {
                if (responseDict.optInt("return_code") == 0) {
                    config.debug("发送成功:" + responseDict.toString());
                    todayTotal += Integer.parseInt(mDbData[2]);
                    if (!ZhugeConfig.debug) {
                        updateEventCount();
                    }
                    localSize -= Integer.parseInt(mDbData[2]);
                    mDb.removeEvent(mDbData[0]);
                } else {
                    config.debug("发送失败，返回码：" + responseDict.optInt("return_code"));
                }
            }
        }

        private void updateEventCount() {
            int day = (int) (System.currentTimeMillis() / 1000 / 86400);
            config.getGlobalSP().edit().putString(config.getTotaldayKey(), day + "|" + todayTotal)
                    .commit();
        }

        private byte[] compress(byte[] data) {
            byte[] output = new byte[0];

            Deflater deflater = new Deflater();
            deflater.reset();
            deflater.setInput(data);
            deflater.finish();
            ByteArrayOutputStream bos = new ByteArrayOutputStream(data.length);
            try {
                int dataLen = data.length;
                byte[] buf = new byte[dataLen + 1];
                while (!deflater.finished()) {
                    int i = deflater.deflate(buf);
                    bos.write(buf, 0, i);
                }
                output = bos.toByteArray();
            } catch (Exception e) {
                output = data;
                e.printStackTrace();
            } finally {
                try {
                    bos.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            deflater.end();
            return output;
        }

    }
}
