package com.qipeng.captcha;

import android.app.Dialog;
import android.content.Context;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.qipeng.captcha.ui.QPCaptchaDialog;
import com.qipeng.captcha.utils.HttpUtils;
import com.qipeng.captcha.utils.LogUtils;
import com.qipeng.captcha.utils.QPUtils;
import com.qipeng.captcha.utils.SpUtils;

import java.util.HashMap;
import java.util.UUID;

/**
 * OnePiece
 * Created by xukq on 4/28/19.
 */
public class QPCaptcha {

    private final String VERSION = "v1.0.7";
    public static String SDK_H5_VERSION = "0.0.1";
    public static String SDK_H5_MINOR = "0.2";
    public static String SDK_JS_VERSION = "0.2.2";
    public static String SDK_JS_MINOR = "0.2.3";
    public static final String H5_NAME = "yunpian.html";
    public static final String JS_NAME = "riddler-sdk.js";
    public static final String JS_HYBRID_NAME = "hybird.js";
    private static QPCaptcha mInstance;

    private static boolean isIniting; // 保证只有一个线程在初始化

    private QPCaptcha() {
    }

    public void release() {
        mInstance = null;
        HttpUtils.sHeader = null;
    }

    public static QPCaptcha getInstance() {
        if (mInstance == null) {
            synchronized (QPCaptcha.class) {
                if (mInstance == null) {
                    mInstance = new QPCaptcha();
                }
            }
        }
        return mInstance;
    }

    public void init(final Context context, String appId) {
        if (context == null || TextUtils.isEmpty(appId)) {
            throw new RuntimeException("context or appId is null");
        }
        boolean isHybirdFileExist = QPUtils.getSDKHybridJSFile(context).exists();
        String preVersion = SpUtils.getInstance(context).getString("qp_version", "");
        boolean isNeedUpdateFile = !TextUtils.equals(preVersion, getSDKVersion());
        if (!isHybirdFileExist || isNeedUpdateFile) {
            QPUtils.copyFilesAssets(context, H5_NAME, QPUtils.getSDKH5File(context).getAbsolutePath());
            QPUtils.copyFilesAssets(context, JS_NAME, QPUtils.getSDKJSFile(context).getAbsolutePath());
            QPUtils.copyFilesAssets(context, JS_HYBRID_NAME, QPUtils.getSDKHybridJSFile(context).getAbsolutePath());
            SpUtils.getInstance(context).putString("qp_version", getSDKVersion());
        }
        final SpUtils spUtils = SpUtils.getInstance(context);
        // 生成随机 32 位 id
        String clientId = spUtils.getString(SpUtils.KEY_STRING_CLIENT_ID, "");
        if (TextUtils.isEmpty(clientId)) {
            clientId = UUID.randomUUID().toString().replace("-", "");
            spUtils.putString(SpUtils.KEY_STRING_CLIENT_ID, clientId);
        }
        spUtils.putString(SpUtils.KEY_STRING_APP_ID, appId);
        // 更新 header
        if (HttpUtils.sHeader == null) {
            HttpUtils.sHeader = new HashMap<>();
            String userAgentString = QPUtils.getUA() + " YunPianCaptchaSDK/" + VERSION;
            HttpUtils.sHeader.put("User-Agent", userAgentString);
            HttpUtils.sHeader.put("x-client-id", clientId);
            HttpUtils.sHeader.put("x-internal-versions", "h5=" + QPCaptcha.SDK_H5_VERSION + "(" + QPCaptcha.SDK_H5_MINOR + ");" + "js-sdk=" + QPCaptcha.SDK_JS_VERSION + "(" + QPCaptcha.SDK_JS_MINOR + ");");
            HttpUtils.sHeader.put("x-app-info", QPUtils.getHeaderDeviceInfos(context));
            HttpUtils.sHeader.put("x-captcha-id", appId);
        }
        // 判断是否需要初始化
        long lastInitTime = spUtils.getLong(SpUtils.KEY_LONG_LAST_INIT_TIME, 0);
        long intervalTime = 1000 * 60 * 60 * 24; // 一天
        final boolean isNeedInit = (System.currentTimeMillis() - lastInitTime) > intervalTime;
        if (isNeedInit && !isIniting) {
            isIniting = true;
            new Thread(new Runnable() {
                @Override
                public void run() {
                    HttpUtils.Result<String> result = HttpUtils.init(context);
                    if (result.getCode() == 200) {
                        spUtils.putLong(SpUtils.KEY_LONG_LAST_INIT_TIME, System.currentTimeMillis());
                    } else {
                        // 写入错误信息到本地
                        LogUtils.writeLogToSD(context, "init", "code = " + result.getCode() + " msg = " + result.getMsg());
                    }
                    // 检测更新
                    HttpUtils.checkVersion(context);
                    isIniting = false;
                }
            }).start();
        }
    }

    public void verify(@NonNull QPCaptchaConfig config) {
        Dialog captchaDialog = new QPCaptchaDialog(config, R.style.QPDialog);
        captchaDialog.show();
    }

    public String getSDKVersion() {
        return VERSION;
    }

}
