package com.sensorsdata.analytics.android.minisdk;

import org.json.JSONObject;

import java.util.List;
import java.util.Map;
import java.util.Set;
/**
 * @author :huangguofeng
 * date :2020-03-24
 * package :com.sensorsdata.analytics.android.minisdk.utils
 * desc    : Sensors API接口类
 */
 interface ISensorsDataAPI {
    /**
     * 返回预置属性
     * @return JSONObject 预置属性
     */
    public JSONObject getPresetProperties();

    /**
     * 设置当前 serverUrl
     * @param serverUrl 当前 serverUrl
     */
    public void setServerUrl(String serverUrl);

    /**
     * 设置是否开启 log
     * @param enable boolean
     */
    public void enableLog(boolean enable);

    /**
     * 获取本地缓存上限制
     * @return 字节
     */
    public long getMaxCacheSize();

    /**
     * 返回档期是否是开启 debug 模式
     * @return true：是，false：不是
     */
    public boolean isDebugMode();

    /**
     * 设置本地缓存上限值，单位 byte，默认为 32MB：32 * 1024 * 1024
     * @param maxCacheSize 单位 byte
     */
    public void setMaxCacheSize(long maxCacheSize);

    /**
     * 设置 flush 时网络发送策略，默认 3G、4G、WI-FI 环境下都会尝试 flush
     * @param networkType int 网络类型
     */
    public void setFlushNetworkPolicy(int networkType);

    /**
     * 两次数据发送的最小时间间隔，单位毫秒
     *
     * 默认值为15 * 1000毫秒
     * 在每次调用track、signUp以及profileSet等接口的时候，都会检查如下条件，以判断是否向服务器上传数据:
     *
     * 1. 是否是WIFI/3G/4G网络条件
     * 2. 是否满足发送条件之一:
     * 1) 与上次发送的时间间隔是否大于 flushInterval
     * 2) 本地缓存日志数目是否大于 flushBulkSize
     *
     * 如果满足这两个条件，则向服务器发送一次数据；如果不满足，则把数据加入到队列中，等待下次检查时把整个队列的内
     * 容一并发送。需要注意的是，为了避免占用过多存储，队列最多只缓存20MB数据。
     *
     * @return 返回时间间隔，单位毫秒
     */
    public int getFlushInterval();

    /**
     * 设置两次数据发送的最小时间间隔
     *
     * @param flushInterval 时间间隔，单位毫秒
     */
    public void setFlushInterval(int flushInterval);

    /**
     * 返回本地缓存日志的最大条目数
     * @return 条数
     */
    public int getFlushBulkSize();

    /**
     * 设置本地缓存日志的最大条目数
     *
     * @param flushBulkSize 缓存数目
     */
    public void setFlushBulkSize(int flushBulkSize);


    /**
     * 自动收集 App Crash 日志，该功能默认是关闭的
     */
    public void trackAppCrash();


    /**
     * 获取当前用户的distinctId
     *
     * 若调用前未调用 {@link #identify(String)} 设置用户的 distinctId，SDK 会调用 {@link java.util.UUID} 随机生成
     * UUID，作为用户的 distinctId
     *
     * 该方法已不推荐使用，请参考 {@link #getAnonymousId()}
     *
     * @return 当前用户的distinctId
     */
    @Deprecated
    public String getDistinctId();

    /**
     * 获取当前用户的匿名id
     *
     * 若调用前未调用 {@link #identify(String)} 设置用户的匿名id，SDK 会调用 {@link java.util.UUID} 随机生成
     * UUID，作为用户的匿名id
     *
     * @return 当前用户的匿名id
     */
    public String getAnonymousId();

    /**
     * 重置默认匿名id
     */
    public void resetAnonymousId();

    /**
     * 获取当前用户的 loginId
     *
     * 若调用前未调用 {@link #login(String)} 设置用户的 loginId，会返回null
     *
     * @return 当前用户的 loginId
     */
    public String getLoginId();

    /**
     * 设置当前用户的distinctId。一般情况下，如果是一个注册用户，则应该使用注册系统内
     * 的user_id，如果是个未注册用户，则可以选择一个不会重复的匿名ID，如设备ID等，如果
     * 客户没有调用identify，则使用SDK自动生成的匿名ID
     *
     * @param distinctId 当前用户的distinctId，仅接受数字、下划线和大小写字母
     */
    public void identify(String distinctId);

    /**
     * 登录，设置当前用户的 loginId
     *
     * @param loginId 当前用户的 loginId，不能为空，且长度不能大于255
     */
    public void login(String loginId);

    /**
     * 注销，清空当前用户的 loginId
     */
    public void logout();


    /**
     * 调用track接口，追踪一个带有属性的事件
     *
     * @param eventName  事件的名称
     * @param properties 事件的属性
     */
    public void track(String eventName, JSONObject properties);

    /**
     * 与 {@link #track(String, JSONObject)} 类似，无事件属性
     *
     * @param eventName 事件的名称
     */
    public void track(String eventName);

    /**
     * 初始化事件的计时器，计时单位为秒。
     *
     * @param eventName 事件的名称
     */
    public void trackTimerStart(final String eventName);

    /**
     * 停止事件计时器
     * @param eventName 事件的名称
     * @param properties 事件的属性
     */
    public void trackTimerEnd(final String eventName, JSONObject properties);

    /**
     * 停止事件计时器
     * @param eventName 事件的名称
     */
    public void trackTimerEnd(final String eventName);

    /**
     * 清除所有事件计时器
     */
    public void clearTrackTimer();


    /**
     * Track 进入页面事件 ($AppViewScreen)
     * @param url String
     * @param properties JSONObject
     */

    /**
     * 将所有本地缓存的日志发送到 Sensors Analytics.
     */
    public void flush();

    /**
     * 以阻塞形式将所有本地缓存的日志发送到 Sensors Analytics，该方法不能在 UI 线程调用。
     */
    public void flushSync();

    /**
     * 获取事件公共属性
     *
     * @return 当前所有Super属性
     */
    public JSONObject getSuperProperties();

    /**
     * 注册所有事件都有的公共属性
     *
     * @param superProperties 事件公共属性
     */
    public void registerSuperProperties(JSONObject superProperties);

    /**
     * 删除事件公共属性
     *
     * @param superPropertyName 事件属性名称
     */
    public void unregisterSuperProperty(String superPropertyName);

    /**
     * 删除所有事件公共属性
     */
    public void clearSuperProperties();

    /**
     * 设置用户的一个或多个Profile。
     * Profile如果存在，则覆盖；否则，新创建。
     *
     * @param properties 属性列表
     */
    public void profileSet(JSONObject properties);

    /**
     * 设置用户的一个Profile，如果之前存在，则覆盖，否则，新创建
     *
     * @param property 属性名称
     * @param value    属性的值，值的类型只允许为
     *                 {@link String}, {@link Number}, {@link java.util.Date}, {@link List}
     */
    public void profileSet(String property, Object value);

    /**
     * 首次设置用户的一个或多个Profile。
     * 与profileSet接口不同的是，如果之前存在，则忽略，否则，新创建
     *
     * @param properties 属性列表
     */
    public void profileSetOnce(JSONObject properties);

    /**
     * 首次设置用户的一个Profile
     * 与profileSet接口不同的是，如果之前存在，则忽略，否则，新创建
     *
     * @param property 属性名称
     * @param value    属性的值，值的类型只允许为
     *                 {@link String}, {@link Number}, {@link java.util.Date}, {@link List}
     */
    public void profileSetOnce(String property, Object value);

    /**
     * 给一个或多个数值类型的Profile增加一个数值。只能对数值型属性进行操作，若该属性
     * 未设置，则添加属性并设置默认值为0
     *
     * @param properties 一个或多个属性集合
     */
    public void profileIncrement(Map<String, ? extends Number> properties);

    /**
     * 给一个数值类型的Profile增加一个数值。只能对数值型属性进行操作，若该属性
     * 未设置，则添加属性并设置默认值为0
     *
     * @param property 属性名称
     * @param value    属性的值，值的类型只允许为 {@link Number}
     */
    public void profileIncrement(String property, Number value);

    /**
     * 给一个列表类型的Profile增加一个元素
     *
     * @param property 属性名称
     * @param value    新增的元素
     */
    public void profileAppend(String property, String value);

    /**
     * 给一个列表类型的Profile增加一个或多个元素
     *
     * @param property 属性名称
     * @param values   新增的元素集合
     */
    public void profileAppend(String property, Set<String> values);

    /**
     * 删除用户的一个Profile
     *
     * @param property 属性名称
     */
    public void profileUnset(String property);

    /**
     * 删除用户所有Profile
     */
    public void profileDelete();

    /**
     * 更新 GPS 位置信息
     * @param latitude 纬度
     * @param longitude 经度
     */
    public void setGPSLocation(double latitude, double longitude);

    /**
     * 清楚 GPS 位置信息
     */
    public void clearGPSLocation();

    /**
     * 开启/关闭采集屏幕方向
     * @param enable true：开启 false：关闭
     */
    public void enableTrackScreenOrientation(boolean enable);

    /**
     * 恢复采集屏幕方向
     */
    public void resumeTrackScreenOrientation();

    /**
     * 暂停采集屏幕方向
     */
    public void stopTrackScreenOrientation();

    /**
     * 获取当前屏幕方向
     * @return portrait:竖屏 landscape:横屏
     */
    public String getScreenOrientation();
}
