package com.yodo1.sdk.kit;

import android.content.Context;
import android.text.TextUtils;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

/**
 * 获取配置文件
 *
 * @author yodo1
 */
public class Yodo1PropertiesUtils {
    /**
     * 配置文件名，存放在游戏res/raw目录下
     */
    private static final String FILE_NAME_PROPERTIES = "yodo1_games_config";

    private static Yodo1PropertiesUtils mInstance;
    private Properties mBasicConfig;
    private Properties mBasicConfigAssets;

    private Yodo1PropertiesUtils() {
    }

    public static Yodo1PropertiesUtils getInstance() {
        if (mInstance == null) {
            mInstance = new Yodo1PropertiesUtils();
        }
        return mInstance;
    }

    /**
     * 从raw文件夹中获取某个配置文件
     *
     * @param context  Context
     * @param rawResId 资源id，必须在raw文件夹
     * @return 配置文件的实体
     */
    public static Properties getPropertiesFromRaw(Context context, int rawResId) {
        Properties properties = null;
        InputStream inputStream = null;
        try {
            if (rawResId > 0) {
                properties = new Properties();
                inputStream = context.getResources().openRawResource(rawResId);
                properties.load(inputStream);
            } else {
                YLog.d("res not found : " + rawResId);
            }
        } catch (IOException e) {
            e.printStackTrace();
            YLog.d("getPropertiesFromRaw failed");
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return properties;
    }

    /**
     * 从assets文件夹中获取某个配置文件
     *
     * @param context  Context
     * @param fileName 文件名称
     * @return Properties
     */
    public static Properties getPropertiesFromAssets(Context context, String fileName) {
        Properties properties = null;
        InputStream inputStream = null;
        try {
            if (!TextUtils.isEmpty(fileName)) {
                properties = new Properties();
                inputStream = context.getAssets().open(fileName);

                properties.load(inputStream);
            } else {
                YLog.d("filePath is null : ");
            }
        } catch (Exception e) {
            YLog.d("getProperties failed : " + fileName);
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return properties;
    }

    /**
     * 初始化
     *
     * @param context Context
     */
    public void initProperties(Context context) {
        if (mBasicConfig == null) {
            int resId = RR.raw(context, FILE_NAME_PROPERTIES);
            if (resId > 0) {
                mBasicConfig = getPropertiesFromRaw(context, resId);
            } else {
                YLog.d("yodo1properties file is not find, filename : " + FILE_NAME_PROPERTIES);
            }
        }
    }

    /**
     * 从配置文件中读取某字段
     *
     * @param key 字段名
     * @return 值
     */
    public String getBasicConfigValue(String key) {
        if (mBasicConfig == null) {
            YLog.d("get error, yodo1properties is not init ...");
            return null;
        }
        String parameter = mBasicConfig.getProperty(key);
        if (TextUtils.isEmpty(parameter)) {
            YLog.d("yodo1properties, property not found : " + key);
        }
        return parameter;
    }

    /**
     * 往配置文件中加入或覆盖某字段
     *
     * @param key   字段名
     * @param value 值
     * @return 成功与否
     */
    public boolean putBasicConfigValue(String key, String value) {
        if (mBasicConfig == null) {
            YLog.d("put error, yodo1properties is not init ...");
            return false;
        }
        mBasicConfig.put(key, value);
        return true;
    }

    /**
     * 从配置文件中读取某字段
     *
     * @param context  Context
     * @param fileName 文件名
     * @param key      字段名
     * @return 值
     */
    public String getBasicConfigValueFromAssets(Context context, String fileName, String key) {
        if (mBasicConfigAssets == null) {
            mBasicConfigAssets = getPropertiesFromAssets(context, fileName);
        }
        String parameter = mBasicConfigAssets.getProperty(key);
        if (TextUtils.isEmpty(parameter)) {
            YLog.d("yodo1properties, property not found : " + key);
        }
        return parameter;
    }
}
