package com.yodo1.plugin.dmp.yodo1.helper;

import android.app.Activity;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.text.TextUtils;

import com.yodo1.onlineconfig.Yodo1OnlineConfig;
import com.yodo1.plugin.dmp.yodo1.constants.AnalyticsConstant;
import com.yodo1.sdk.kit.SysUtils;
import com.yodo1.sdk.kit.YLog;
import com.yodo1.sdk.kit.Yodo1CommonUtils;
import com.yodo1.sdk.kit.Yodo1DeviceUtils;
import com.yodo1.sdk.kit.Yodo1SharedPreferences;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Created by yodo1 on 18/5/8.
 */
public class Yodo1AnalyticsManage {

    private static final String TAG = "[Yodo1AnalyticsManage] ";

    /**
     * handler消息标记
     */
    private static final int WHAT_REFRESH_DATA_IMMEDIATE = 5000;
    private static final int WHAT_REFRESH_DATA_TIMED = 5001;
    private static final int WHAT_REFRESH_CLOSE_HANDLER = 5002;

    private static Yodo1AnalyticsManage instance;
    private static Yodo1AnalyticsForYodo1 helper = new Yodo1AnalyticsForYodo1();
    private static Activity mActivity;

    /**
     * 设备唯一识别码
     */
    private String deviceId;
    /**
     * sessionId 应该由客户端通过设备唯一识别码和当前时间生成以保证全局唯一
     */
    private static String sessionId;
    /**
     * 启动时间 单位：毫秒
     */
    private static long launchTime;
    /**
     * 一次数据刷新结束的时间 单位：毫秒
     */
    private static long terminalTime;
    /**
     * duration为持续时间, 单位：秒
     */
    private static long duration = 0;
    /**
     * 是否需要更新session
     */
    private static boolean newSession = false;
    /**
     * 初始化状态
     */
    private boolean isInitialized = false;
    /**
     * 前后台状态
     */
    private static boolean isAppOnForeground = false;
    /**
     * 定时器和定时任务
     */
    private static Timer timer, timer1;
    private static TimerTask task, task1;
    /**
     * 数据刷新和上报的handler
     */
    private static Yodo1Handler handler;

    public static Yodo1AnalyticsManage getInstance() {
        if (instance == null) {
            instance = new Yodo1AnalyticsManage();
        }
        return instance;
    }

    /**
     * 只处理应用启动初始化
     * <p>
     * 1：启动与否受限于后台配置的统计开关
     * 2：初始化只允许一次，多次会拦截
     * 3：先获取上一次储存的 sessionId 和储存的时长 lastDuration，避免被覆盖
     * 4：生成并储存新的 sessionId
     * 5：初始化helper，并记录启动事件，启动定时器
     */
    public void initSDK(Activity activity, String appKey) {
        YLog.d(TAG + "initSDK...");
        String isEnabled = Yodo1OnlineConfig.getInstance().getConfigParam(AnalyticsConstant.KEY_ANALYTICS_SWITCH, "on");
        if (isEnabled.equals("off")) {
            YLog.d(TAG + "Yodo1数据统计关闭");
            return;
        }
        if (isInitialized) {
            YLog.d(TAG + "重复 initSDK ");
            return;
        }

        mActivity = activity;
        handler = new Yodo1Handler(activity);

        deviceId = Yodo1DeviceUtils.getDeviceId(activity);
        launchTime = Yodo1CommonUtils.getUTCTime();

        /**
         *  先取出上一次启动时储存的sessionId 和未上报的时长 lastDuration
         */
        String lastSessionId = Yodo1SharedPreferences.getString(mActivity, AnalyticsConstant.KEY_SESSION_ID);
        String lastDuration = Yodo1SharedPreferences.getString(mActivity, AnalyticsConstant.KEY_DURATION);
        /**
         *  生成新的 sessionId 并保存到SP
         */
        sessionId = generateSessionId(deviceId, launchTime + "");
        Yodo1SharedPreferences.put(mActivity, AnalyticsConstant.KEY_SESSION_ID, sessionId);
        /**
         *  如果是第一次安装启动 lastSessionId 肯定是空，以第一次生成的 sessionId 作为 lastSessionId
         */
        if (TextUtils.isEmpty(lastSessionId)) {
            lastSessionId = sessionId;
        }

        /**
         *  初始化db 保存启动事件
         */
        helper.initSDK(activity, appKey);
        helper.saveLaunchEvent(activity, launchTime + "", sessionId, lastSessionId, lastDuration);

        isInitialized = true;

        startUpdate();
        refreshDate();
    }

    /**
     * 只处理应用从后台到前台
     */
    public void onResume(final Activity activity) {
        if (!isInitialized) {
            return;
        }
        if (!isAppOnForeground) {
            YLog.d(TAG + "onResume...  应用没有到过后台 不处理 resume");
            return;
        }
        YLog.d(TAG + "onResume...");

        try {
            /**
             * 获取回到前台的时间 记录的 session ，记录的最后刷新时间 terminalTime  ，记录的已玩时长 duration
             */
            launchTime = Yodo1CommonUtils.getUTCTime();
            sessionId = Yodo1SharedPreferences.getString(mActivity, AnalyticsConstant.KEY_SESSION_ID);
            String t = Yodo1SharedPreferences.getString(mActivity, AnalyticsConstant.KEY_TERMINAL_TIME);
            terminalTime = Yodo1CommonUtils.getLongValues(t);
            String d = Yodo1SharedPreferences.getString(mActivity, AnalyticsConstant.KEY_DURATION);
            duration = Yodo1CommonUtils.getLongValues(d);

            /**
             *  后台回归前台的时差超过30秒 或者 session 为空，判定为需要更新 session
             */
            boolean needNewSessionId = ((launchTime - terminalTime > 30000) || TextUtils.isEmpty(sessionId));
            YLog.d(TAG + "Need new session id ？ : " + needNewSessionId);

            if (needNewSessionId) {
                /**
                 *  取出旧的session 和 duration
                 */
                final String lastDuration = d;
                final String lastTimeSessionId = sessionId;

                /**
                 *  生成新的 session 并储存
                 */
                sessionId = generateSessionId(deviceId, Yodo1CommonUtils.getUTCTime() + "");
                Yodo1SharedPreferences.put(mActivity, AnalyticsConstant.KEY_SESSION_ID, sessionId);

                /**
                 *  新的 session 的时长需重置和记录
                 */
                duration = 0;
                Yodo1SharedPreferences.put(mActivity, AnalyticsConstant.KEY_DURATION, "0");

                /**
                 *  新session标志位置 true
                 */
                newSession = true;

                YLog.d(TAG + "onResume saveLaunchEvent lastDuration : " + lastDuration + " ,  lastTimeSessionId : " + lastTimeSessionId);
                YLog.d(TAG + "onResume newSession , sessionId : " + sessionId);

                //启动后延时1s保存launch信息    确保SDK已经初始化完成
                helper.saveLaunchEvent(activity, launchTime + "", sessionId, lastTimeSessionId, lastDuration);
            }

            startUpdate();//定时5min上报事件信息
            refreshDate();//定时5s刷新本地信息
        } catch (Exception ignored) {
        } catch (Error ignored) {
        }
    }

    /**
     * 如果应用推到后台 关闭定时器
     */
    public void onPause(Activity activity) {
        if (!isInitialized) {
            return;
        }
        if (SysUtils.isAppOnForeground(activity)) {
            YLog.d(TAG + "应用在前台, 不处理onPause");
            return;
        }
        isAppOnForeground = true;
        YLog.d(TAG + "onPause...");

        toBackRefreshDate();
        closeUpdate();//关闭定时器
    }

    public void onDestroy(Activity activity) {
        if (!isInitialized) {
            return;
        }
        YLog.d(TAG + "onDestroy...");

        if (helper != null) {
            helper.onDestroy(activity);
        }
    }

    /**
     * 记录事件
     */
    public void onEvent(final String eventId, final HashMap<String, Object> map) {
        if (!isInitialized) {
            return;
        }

        new Thread(new Runnable() {
            @Override
            public void run() {
                //线程执行内容
                Looper.prepare();
                if (helper != null) {
                    helper.onEvent(sessionId, eventId, map);
                }
                Looper.loop();
            }
        }).start();
    }

    /**
     * 生成 Session Id
     *
     * @param deviceId 设备唯一识别码
     * @param utcTime  UTC时间戳
     * @return session id
     */
    private static String generateSessionId(String deviceId, String utcTime) {
        return deviceId + utcTime;
    }

    /**
     * 本地缓存数据刷新和储存 时长和最后刷新时间
     */
    private static void toBackRefreshDate() {
        if (!SysUtils.isAppOnForeground(mActivity)) {
            YLog.d(TAG + "onPause  应用在后台 停止数据刷新");
            handler.sendEmptyMessage(WHAT_REFRESH_CLOSE_HANDLER);
            isAppOnForeground = true;
            return;
        }

        terminalTime = Yodo1CommonUtils.getUTCTime();

        /**
         *  这一轮数据刷新时此时时间和启动时间的差
         */
        long currentDuration = (terminalTime - launchTime) / 1000;
        /**
         *  以刷新时间作为下一次刷新的开始时间
         */
        launchTime = terminalTime;
        /**
         *  如果是新的session 把这一轮的时间记在新的 session 头上，重置 newSession 的值为 false
         *  如果不是新的session 把这一轮的时间累加在 duration 上
         */
        if (newSession) {
            duration = currentDuration;
            newSession = false;
        } else {
            duration = duration + currentDuration;
        }
        YLog.d(TAG + "total duration: " + duration + " seconds");

        /**
         *  保存 duration  保存 terminalTime
         */
        Yodo1SharedPreferences.put(mActivity, AnalyticsConstant.KEY_DURATION, String.valueOf(duration));
        Yodo1SharedPreferences.put(mActivity, AnalyticsConstant.KEY_TERMINAL_TIME, String.valueOf(terminalTime));
    }

    /**
     * 定时器  调用update方法 上传数据
     */
    private void startUpdate() {
        task = new TimerTask() {
            @Override
            public void run() {
                YLog.d(TAG + "发送数据");
                Message message = new Message();
                message.what = WHAT_REFRESH_DATA_IMMEDIATE;
                handler.sendMessage(message);
            }
        };

        timer = new Timer();
        YLog.d(TAG + "开启定时上传");
        timer.schedule(task, 5000, AnalyticsConstant.REPORT_DATA_INTERVAL);
    }

    /**
     * 定时器 刷新数据
     */
    private void refreshDate() {
        task1 = new TimerTask() {
            @Override
            public void run() {
                Message message = new Message();
                message.what = WHAT_REFRESH_DATA_TIMED;
                handler.sendMessage(message);
            }
        };

        timer1 = new Timer();
        timer1.schedule(task1, 1000, AnalyticsConstant.REFRESH_DATA_TIME);
    }

    /**
     * 关闭定时器
     */
    private static void closeUpdate() {
        handler.removeMessages(WHAT_REFRESH_DATA_IMMEDIATE);
        handler.removeMessages(WHAT_REFRESH_DATA_TIMED);
        if (timer != null) {
            timer.cancel();
            timer = null;
        }
        if (task != null) {
            task.cancel();
            task = null;
        }

        if (timer1 != null) {
            timer1.cancel();
            timer1 = null;
        }
        if (task1 != null) {
            task1.cancel();
            task1 = null;
        }
    }


    /**
     * 以静态内部类使用handler 避免内存泄露
     */
    static class Yodo1Handler extends Handler {

        private WeakReference<Activity> weakReference;

        public Yodo1Handler(Activity activity) {
            weakReference = new WeakReference<>(activity);
        }

        @Override
        public void handleMessage(Message msg) {
            YLog.d(TAG + "Yodo1Handler  handleMessage  msg.what : " + msg.what);

            switch (msg.what) {
                case WHAT_REFRESH_DATA_IMMEDIATE:
                    helper.Update(helper);
                    break;
                case WHAT_REFRESH_DATA_TIMED:
                    //每5s刷新一次数据
                    toBackRefreshDate();
                    break;
                case WHAT_REFRESH_CLOSE_HANDLER:
                    closeUpdate();
                default:
                    break;
            }
        }
    }
}
