package com.sensorsdata.analytics.android.minisdk;

import android.app.Application;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.graphics.Point;
import android.hardware.SensorManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.SystemClock;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.WindowManager;

import com.sensorsdata.analytics.android.minisdk.exceptions.InvalidDataException;
import com.sensorsdata.analytics.android.minisdk.util.JSONUtils;
import com.sensorsdata.analytics.android.minisdk.util.SensorsDataUtils;
import com.yodo1.sdk.kit.SysUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

/**
 * @author :huangguofeng
 * date :2020-03-24
 * package :com.sensorsdata.analytics.android.minisdk
 * desc    : Sensors 核心API处理类
 */
public class Yodo1SensorsDataAPI implements ISensorsDataAPI {

    /**
     * Debug 模式，用于检验数据导入是否正确。该模式下，事件会逐条实时发送到 Sensors Analytics，并根据返回值检查
     * 数据导入是否正确。
     * <p>
     * Debug 模式的具体使用方式，请参考:
     * http://www.sensorsdata.cn/manual/debug_mode.html
     * <p>
     * Debug 模式有三种：
     * DEBUG_OFF - 关闭DEBUG模式
     * DEBUG_ONLY - 打开DEBUG模式，但该模式下发送的数据仅用于调试，不进行数据导入
     * DEBUG_AND_TRACK - 打开DEBUG模式，并将数据导入到SensorsAnalytics中
     */
    public enum DebugMode {
        DEBUG_OFF(false, false),
        DEBUG_ONLY(true, false),
        DEBUG_AND_TRACK(true, true);

        private final boolean debugMode;
        private final boolean debugWriteData;

        DebugMode(boolean debugMode, boolean debugWriteData) {
            this.debugMode = debugMode;
            this.debugWriteData = debugWriteData;
        }

        boolean isDebugMode() {
            return debugMode;
        }

        boolean isDebugWriteData() {
            return debugWriteData;
        }
    }

    /**
     * 网络类型
     */
    public final class NetworkType {
        public static final int TYPE_NONE = 0;//NULL
        public static final int TYPE_2G = 1;//2G
        public static final int TYPE_3G = 1 << 1;//3G
        public static final int TYPE_4G = 1 << 2;//4G
        public static final int TYPE_WIFI = 1 << 3;//WIFI
        public static final int TYPE_ALL = 0xFF;//ALL
    }

    protected boolean isShouldFlush(String networkType) {
        return (toNetworkType(networkType) & mFlushNetworkPolicy) != 0;
    }

    private int toNetworkType(String networkType) {
        if ("NULL".equals(networkType)) {
            return NetworkType.TYPE_ALL;
        } else if ("WIFI".equals(networkType)) {
            return NetworkType.TYPE_WIFI;
        } else if ("2G".equals(networkType)) {
            return NetworkType.TYPE_2G;
        } else if ("3G".equals(networkType)) {
            return NetworkType.TYPE_3G;
        } else if ("4G".equals(networkType)) {
            return NetworkType.TYPE_4G;
        }
        return NetworkType.TYPE_ALL;
    }


    //private
    Yodo1SensorsDataAPI() {
        mContext = null;
        mMessages = null;
        mDistinctId = null;
        mLoginId = null;
        mSuperProperties = null;
        mFirstStart = null;
        mFirstDay = null;
        mDeviceInfo = null;
        mTrackTimer = null;
    }

    Yodo1SensorsDataAPI(Context context, String serverURL, DebugMode debugMode) {
        mContext = context;
        mDebugMode = debugMode;

        final String packageName = context.getApplicationContext().getPackageName();


        try {
            SensorsDataUtils.cleanUserAgent(mContext);
        } catch (Exception e) {
            e.printStackTrace();
        }

        try {
            SALog.init(this);
            final ApplicationInfo appInfo = context.getApplicationContext().getPackageManager()
                    .getApplicationInfo(packageName, PackageManager.GET_META_DATA);
            Bundle configBundle = appInfo.metaData;
            if (null == configBundle) {
                configBundle = new Bundle();
            }

            setServerUrl(serverURL);

            if (debugMode == DebugMode.DEBUG_OFF) {
                ENABLE_LOG = configBundle.getBoolean("com.sensorsdata.analytics.android.EnableLogging",
                        false);
            } else {
                ENABLE_LOG = configBundle.getBoolean("com.sensorsdata.analytics.android.EnableLogging",
                        true);
            }

            mFlushInterval = configBundle.getInt("com.sensorsdata.analytics.android.FlushInterval",
                    15000);
            mFlushBulkSize = configBundle.getInt("com.sensorsdata.analytics.android.FlushBulkSize",
                    100);
            mEnableAndroidId = configBundle.getBoolean("com.sensorsdata.analytics.android.AndroidId",
                    true);
        } catch (final PackageManager.NameNotFoundException e) {
            throw new RuntimeException("Can't configure Yodo1SensorsDataAPI with package name " + packageName,
                    e);
        }

        mMessages = AnalyticsMessages.getInstance(mContext, SysUtils.getPackageName(mContext)+Yodo1SensorsDataAPI.SENSORS_DATA_API_FILE_NAME);

        final SharedPreferencesLoader.OnPrefsLoadedListener listener =
                new SharedPreferencesLoader.OnPrefsLoadedListener() {
                    @Override
                    public void onPrefsLoaded(SharedPreferences preferences) {
                    }
                };

        final Future<SharedPreferences> storedPreferences =
                sPrefsLoader.loadPreferences(context, SysUtils.getPackageName(context)+Yodo1SensorsDataAPI.SENSORS_DATA_API_FILE_NAME, listener);

        mDistinctId = new PersistentDistinctId(storedPreferences);
        if (mEnableAndroidId) {
            try {
                if (TextUtils.isEmpty(mDistinctId.get())) {
                    String androidId = SensorsDataUtils.getAndroidID(mContext);
                    if (SensorsDataUtils.isValidAndroidId(androidId)) {
                        identify(androidId);
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        mLoginId = new PersistentLoginId(storedPreferences);
        mSuperProperties = new PersistentSuperProperties(storedPreferences);
        mFirstStart = new PersistentFirstStart(storedPreferences);

        mFirstDay = new PersistentFirstDay(storedPreferences);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.ICE_CREAM_SANDWICH) {
            final Application app = (Application) context.getApplicationContext();
            app.registerActivityLifecycleCallbacks(new SensorsDataActivityLifecycleCallbacks(this, mFirstStart, mFirstDay));
        }

        if (debugMode != DebugMode.DEBUG_OFF) {
            Log.i(TAG, String.format(Locale.CHINA, "Initialized the instance of Sensors Analytics SDK with server"
                    + " url '%s', flush interval %d ms, debugMode: %s", mServerUrl, mFlushInterval, debugMode));
        }

        final Map<String, Object> deviceInfo = new HashMap<>();

        {
            deviceInfo.put("$lib", "Android");
            deviceInfo.put("$lib_version", VERSION);
            deviceInfo.put("$os", "Android");
            deviceInfo.put("$os_version",
                    Build.VERSION.RELEASE == null ? "UNKNOWN" : Build.VERSION.RELEASE);
            deviceInfo
                    .put("$manufacturer", Build.MANUFACTURER == null ? "UNKNOWN" : Build.MANUFACTURER);
            if (TextUtils.isEmpty(Build.MODEL)) {
                deviceInfo.put("$model", "UNKNOWN");
            } else {
                deviceInfo.put("$model", Build.MODEL.trim());
            }
            try {
                final PackageManager manager = mContext.getPackageManager();
                final PackageInfo info = manager.getPackageInfo(mContext.getPackageName(), 0);
                deviceInfo.put("$app_version", info.versionName);
            } catch (final Exception e) {
                if (debugMode != DebugMode.DEBUG_OFF) {
                    Log.i(TAG, "Exception getting app version name", e);
                }
            }
            final DisplayMetrics displayMetrics = context.getResources().getDisplayMetrics();
            deviceInfo.put("$screen_height", displayMetrics.heightPixels);
            deviceInfo.put("$screen_width", displayMetrics.widthPixels);

            try {
                WindowManager windowManager = (WindowManager) mContext.getSystemService(Context.WINDOW_SERVICE);
                if (Build.VERSION.SDK_INT >= 17) {
                    Point point = new Point();
                    if (windowManager != null) {
                        windowManager.getDefaultDisplay().getRealSize(point);
                        deviceInfo.put("$screen_height", point.y);
                    }
                }
            } catch (Exception e) {
                deviceInfo.put("$screen_height", displayMetrics.heightPixels);
            }

            String $carrier = SensorsDataUtils.getCarrier(mContext);
            if (!TextUtils.isEmpty($carrier)) {
                deviceInfo.put("$carrier", $carrier);
            }

            String androidID = SensorsDataUtils.getAndroidID(mContext);
            if (!TextUtils.isEmpty(androidID)) {
                deviceInfo.put("$device_id", androidID);
            }
        }

        mDeviceInfo = Collections.unmodifiableMap(deviceInfo);
        mTrackTimer = new HashMap<>();
    }

    /**
     * 获取SensorsDataAPI单例
     *
     * @param context App的Context
     * @return SensorsDataAPI单例
     */
    public static Yodo1SensorsDataAPI sharedInstance(Context context) {

        if (null == context) {
            return new SensorsDataAPIEmptyImplementation();
        }

        synchronized (sInstanceMap) {
            final Context appContext = context.getApplicationContext();
            Yodo1SensorsDataAPI instance = sInstanceMap.get(appContext);

            if (null == instance) {
                Log.i(TAG, "The static method sharedInstance(context, serverURL, debugMode) should be called before calling sharedInstance()");
                return new SensorsDataAPIEmptyImplementation();
            }
            return instance;
        }
    }


    /**
     * 初始化并获取SensorsDataAPI单例
     *
     * @param context   App 的 Context
     * @param serverURL 用于收集事件的服务地址
     * @param debugMode Debug模式,
     *                  {@link DebugMode}
     * @return SensorsDataAPI单例
     */
    public static Yodo1SensorsDataAPI sharedInstance(Context context, String serverURL, DebugMode debugMode) {
        if (null == context) {
            return new SensorsDataAPIEmptyImplementation();
        }

        synchronized (sInstanceMap) {
            final Context appContext = context.getApplicationContext();

            Yodo1SensorsDataAPI instance = sInstanceMap.get(appContext);
            if (null == instance && ConfigurationChecker.checkBasicConfiguration(appContext)) {
                instance = new Yodo1SensorsDataAPI(appContext, serverURL, debugMode);
                sInstanceMap.put(appContext, instance);
            }

            if (instance != null) {
                return instance;
            } else {
                return new SensorsDataAPIEmptyImplementation();
            }
        }
    }

    public static Yodo1SensorsDataAPI sharedInstance() {

        synchronized (sInstanceMap) {
            if (sInstanceMap.size() > 0) {
                Iterator<Yodo1SensorsDataAPI> iterator = sInstanceMap.values().iterator();
                if (iterator.hasNext()) {
                    return iterator.next();
                }
            }
            return new SensorsDataAPIEmptyImplementation();
        }
    }


    /**
     * 返回预置属性
     *
     * @return JSONObject 预置属性
     */
    @Override
    public JSONObject getPresetProperties() {
        JSONObject properties = new JSONObject();
        try {
            properties.put("$app_version", mDeviceInfo.get("$app_version"));
            properties.put("$lib", "Android");
            properties.put("$lib_version", VERSION);
            properties.put("$manufacturer", mDeviceInfo.get("$manufacturer"));
            properties.put("$model", mDeviceInfo.get("$model"));
            properties.put("$os", "Android");
            properties.put("$os_version", mDeviceInfo.get("$os_version"));
            properties.put("$screen_height", mDeviceInfo.get("$screen_height"));
            properties.put("$screen_width", mDeviceInfo.get("$screen_width"));
            String networkType = SensorsDataUtils.networkType(mContext);
            properties.put("$wifi", networkType.equals("WIFI"));
            properties.put("$network_type", networkType);
            properties.put("$carrier", mDeviceInfo.get("$carrier"));
            properties.put("$is_first_day", isFirstDay());
            properties.put("$device_id", mDeviceInfo.get("$device_id"));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return properties;
    }

    /**
     * 设置当前 serverUrl
     *
     * @param serverUrl 当前 serverUrl
     */
    @Override
    public void setServerUrl(String serverUrl) {
        try {
            mOriginServerUrl = serverUrl;
            if (TextUtils.isEmpty(serverUrl) || mDebugMode == DebugMode.DEBUG_OFF) {
                mServerUrl = serverUrl;
                disableDebugMode();
            } else {
                Uri serverURI = Uri.parse(serverUrl);

                int pathPrefix = serverURI.getPath().lastIndexOf('/');
                if (pathPrefix != -1) {
                    String newPath = serverURI.getPath().substring(0, pathPrefix) + "/debug";

                    // 将 URI Path 中末尾的部分替换成 '/debug'
                    mServerUrl = serverURI.buildUpon().path(newPath).build().toString();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 设置是否开启 log
     *
     * @param enable boolean
     */
    @Override
    public void enableLog(boolean enable) {
        this.ENABLE_LOG = enable;
    }

    @Override
    public long getMaxCacheSize() {
        return mMaxCacheSize;
    }

    /**
     * 设置本地缓存上限值，单位 byte，默认为 32MB：32 * 1024 * 1024
     *
     * @param maxCacheSize 单位 byte
     */
    @Override
    public void setMaxCacheSize(long maxCacheSize) {
        if (maxCacheSize > 0) {
            //防止设置的值太小导致事件丢失
            if (maxCacheSize < 16 * 1024 * 1024) {
                maxCacheSize = 16 * 1024 * 1024;
            }
            this.mMaxCacheSize = maxCacheSize;
        }
    }

    /**
     * 设置 flush 时网络发送策略，默认 3G、4G、WI-FI 环境下都会尝试 flush
     *
     * @param networkType int 网络类型
     */
    @Override
    public void setFlushNetworkPolicy(int networkType) {
        mFlushNetworkPolicy = networkType;
    }

    /**
     * 两次数据发送的最小时间间隔，单位毫秒
     * <p>
     * 默认值为15 * 1000毫秒
     * 在每次调用track、signUp以及profileSet等接口的时候，都会检查如下条件，以判断是否向服务器上传数据:
     * <p>
     * 1. 是否是WIFI/3G/4G网络条件
     * 2. 是否满足发送条件之一:
     * 1) 与上次发送的时间间隔是否大于 flushInterval
     * 2) 本地缓存日志数目是否大于 flushBulkSize
     * <p>
     * 如果满足这两个条件，则向服务器发送一次数据；如果不满足，则把数据加入到队列中，等待下次检查时把整个队列的内
     * 容一并发送。需要注意的是，为了避免占用过多存储，队列最多只缓存20MB数据。
     *
     * @return 返回时间间隔，单位毫秒
     */
    @Override
    public int getFlushInterval() {
        return mFlushInterval;
    }

    /**
     * 设置两次数据发送的最小时间间隔
     *
     * @param flushInterval 时间间隔，单位毫秒
     */
    @Override
    public void setFlushInterval(int flushInterval) {
        if (flushInterval < 5 * 1000) {
            flushInterval = 5 * 1000;
        }
        mFlushInterval = flushInterval;
    }

    /**
     * 返回本地缓存日志的最大条目数
     * <p>
     * 默认值为100条
     * 在每次调用track、signUp以及profileSet等接口的时候，都会检查如下条件，以判断是否向服务器上传数据:
     * <p>
     * 1. 是否是WIFI/3G/4G网络条件
     * 2. 是否满足发送条件之一:
     * 1) 与上次发送的时间间隔是否大于 flushInterval
     * 2) 本地缓存日志数目是否大于 flushBulkSize
     * <p>
     * 如果满足这两个条件，则向服务器发送一次数据；如果不满足，则把数据加入到队列中，等待下次检查时把整个队列的内
     * 容一并发送。需要注意的是，为了避免占用过多存储，队列最多只缓存32MB数据。
     *
     * @return 返回本地缓存日志的最大条目数
     */
    @Override
    public int getFlushBulkSize() {
        return mFlushBulkSize;
    }

    /**
     * 设置本地缓存日志的最大条目数
     *
     * @param flushBulkSize 缓存数目
     */
    @Override
    public void setFlushBulkSize(int flushBulkSize) {
        mFlushBulkSize = flushBulkSize;
    }

    /**
     * 更新 GPS 位置信息
     *
     * @param latitude  纬度
     * @param longitude 经度
     */
    @Override
    public void setGPSLocation(double latitude, double longitude) {
        try {
            if (mGPSLocation == null) {
                mGPSLocation = new SensorsDataGPSLocation();
            }

            mGPSLocation.setLatitude((long) (latitude * Math.pow(10, 6)));
            mGPSLocation.setLongitude((long) (longitude * Math.pow(10, 6)));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 清楚 GPS 位置信息
     */
    @Override
    public void clearGPSLocation() {
        mGPSLocation = null;
    }

    @Override
    public void enableTrackScreenOrientation(boolean enable) {
        try {
            if (enable) {
                if (mOrientationDetector == null) {
                    mOrientationDetector = new SensorsDataScreenOrientationDetector(mContext, SensorManager.SENSOR_DELAY_NORMAL);
                }
                mOrientationDetector.enable();
            } else {
                if (mOrientationDetector != null) {
                    mOrientationDetector.disable();
                    mOrientationDetector = null;
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void resumeTrackScreenOrientation() {
        try {
            if (mOrientationDetector != null) {
                mOrientationDetector.enable();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void stopTrackScreenOrientation() {
        try {
            if (mOrientationDetector != null) {
                mOrientationDetector.disable();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public String getScreenOrientation() {
        try {
            if (mOrientationDetector != null) {
                return mOrientationDetector.getOrientation();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }


    /**
     * 自动收集 App Crash 日志，该功能默认是关闭的
     */
    @Override
    public void trackAppCrash() {
        SensorsDataExceptionHandler.init();
    }

    // Package-level access. Used (at least) by GCMReceiver
    // when OS-level events occur.
    /* package */ interface InstanceProcessor {
        public void process(Yodo1SensorsDataAPI m);
    }

    /* package */
    static void allInstances(InstanceProcessor processor) {
        synchronized (sInstanceMap) {
            for (final Yodo1SensorsDataAPI instance : sInstanceMap.values()) {
                processor.process(instance);
            }
        }
    }

    /**
     * 获取当前用户的distinctId
     * <p>
     * 若调用前未调用 {@link #identify(String)} 设置用户的 distinctId，SDK 会调用 {@link UUID} 随机生成
     * UUID，作为用户的 distinctId
     * <p>
     * 该方法已不推荐使用，请参考 {@link #getAnonymousId()}
     *
     * @return 当前用户的distinctId
     */
    @Deprecated
    @Override
    public String getDistinctId() {
        synchronized (mDistinctId) {
            return mDistinctId.get();
        }
    }

    /**
     * 获取当前用户的匿名id
     * <p>
     * 若调用前未调用 {@link #identify(String)} 设置用户的匿名id，SDK 会调用 {@link UUID} 随机生成
     * UUID，作为用户的匿名id
     *
     * @return 当前用户的匿名id
     */
    @Override
    public String getAnonymousId() {
        synchronized (mDistinctId) {
            return mDistinctId.get();
        }
    }

    /**
     * 重置默认匿名id
     */
    @Override
    public void resetAnonymousId() {
        synchronized (mDistinctId) {
            if (mEnableAndroidId) {
                String androidId = SensorsDataUtils.getAndroidID(mContext);
                if (SensorsDataUtils.isValidAndroidId(androidId)) {
                    mDistinctId.commit(androidId);
                    return;
                }
            }
            mDistinctId.commit(UUID.randomUUID().toString());
        }
    }

    /**
     * 获取当前用户的 loginId
     * <p>
     * 若调用前未调用 {@link #login(String)} 设置用户的 loginId，会返回null
     *
     * @return 当前用户的 loginId
     */
    @Override
    public String getLoginId() {
        synchronized (mLoginId) {
            return mLoginId.get();
        }
    }

    /**
     * 设置当前用户的distinctId。一般情况下，如果是一个注册用户，则应该使用注册系统内
     * 的user_id，如果是个未注册用户，则可以选择一个不会重复的匿名ID，如设备ID等，如果
     * 客户没有调用identify，则使用SDK自动生成的匿名ID
     *
     * @param distinctId 当前用户的distinctId，仅接受数字、下划线和大小写字母
     */
    @Override
    public void identify(String distinctId) {
        try {
            assertDistinctId(distinctId);
            synchronized (mDistinctId) {
                mDistinctId.commit(distinctId);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 登录，设置当前用户的 loginId
     *
     * @param loginId 当前用户的 loginId，不能为空，且长度不能大于255
     */
    @Override
    public void login(String loginId) {
        try {
            assertDistinctId(loginId);
            synchronized (mLoginId) {
                if (!loginId.equals(mLoginId.get())) {
                    mLoginId.commit(loginId);
                    if (!loginId.equals(getAnonymousId())) {
                        trackEvent(EventType.TRACK_SIGNUP, "$SignUp", null, getAnonymousId());
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 注销，清空当前用户的 loginId
     */
    @Override
    public void logout() {
        synchronized (mLoginId) {
            mLoginId.commit(null);
        }
    }


    /**
     * 调用track接口，追踪一个带有属性的事件
     *
     * @param eventName  事件的名称
     * @param properties 事件的属性
     */
    @Override
    public void track(String eventName, JSONObject properties) {
        try {
            trackEvent(EventType.TRACK, eventName, properties, null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 与 {@link #track(String, JSONObject)} 类似，无事件属性
     *
     * @param eventName 事件的名称
     */
    @Override
    public void track(String eventName) {
        try {
            trackEvent(EventType.TRACK, eventName, null, null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    /**
     * 初始化事件的计时器，计时单位为秒。
     *
     * @param eventName 事件的名称
     */
    @Override
    public void trackTimerStart(final String eventName) {
        try {
            assertKey(eventName);
            synchronized (mTrackTimer) {
                mTrackTimer.put(eventName, new EventTimer(TimeUnit.SECONDS));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 停止事件计时器
     *
     * @param eventName  事件的名称
     * @param properties 事件的属性
     */
    @Override
    public void trackTimerEnd(final String eventName, JSONObject properties) {
        track(eventName, properties);
    }

    /**
     * 停止事件计时器
     *
     * @param eventName 事件的名称
     */
    @Override
    public void trackTimerEnd(final String eventName) {
        track(eventName);
    }

    /**
     * 清除所有事件计时器
     */
    @Override
    public void clearTrackTimer() {
        synchronized (mTrackTimer) {
            mTrackTimer.clear();
        }
    }

    /**
     * app进入后台
     * 遍历mTrackTimer
     * eventAccumulatedDuration = eventAccumulatedDuration + System.currentTimeMillis() - startTime
     */
    protected void appEnterBackground() {
        synchronized (mTrackTimer) {
            try {
                Iterator iter = mTrackTimer.entrySet().iterator();
                while (iter.hasNext()) {
                    Map.Entry entry = (Map.Entry) iter.next();
                    if (entry != null) {
                        if ("$AppEnd".equals(entry.getKey().toString())) {
                            continue;
                        }
                        EventTimer eventTimer = (EventTimer) entry.getValue();
                        if (eventTimer != null) {
                            long eventAccumulatedDuration = eventTimer.getEventAccumulatedDuration() + SystemClock.elapsedRealtime() - eventTimer.getStartTime();
                            eventTimer.setEventAccumulatedDuration(eventAccumulatedDuration);
                            eventTimer.setStartTime(SystemClock.elapsedRealtime());
                        }
                    }
                }
            } catch (Exception e) {
                SALog.i(TAG, "appEnterBackground error:" + e.getMessage());
            }
        }
    }

    /**
     * app从后台恢复
     * 遍历mTrackTimer
     * startTime = System.currentTimeMillis()
     */
    protected void appBecomeActive() {
        synchronized (mTrackTimer) {
            try {
                Iterator iter = mTrackTimer.entrySet().iterator();
                while (iter.hasNext()) {
                    Map.Entry entry = (Map.Entry) iter.next();
                    if (entry != null) {
                        EventTimer eventTimer = (EventTimer) entry.getValue();
                        if (eventTimer != null) {
                            eventTimer.setStartTime(SystemClock.elapsedRealtime());
                        }
                    }
                }
            } catch (Exception e) {
                SALog.i(TAG, "appBecomeActive error:" + e.getMessage());
            }
        }
    }

    /**
     * 将所有本地缓存的日志发送到 Sensors Analytics.
     */
    @Override
    public void flush() {
        mMessages.flush();
    }

    /**
     * 以阻塞形式将所有本地缓存的日志发送到 Sensors Analytics，该方法不能在 UI 线程调用。
     */
    @Override
    public void flushSync() {
        mMessages.sendData();
    }

    /**
     * 获取事件公共属性
     *
     * @return 当前所有Super属性
     */
    @Override
    public JSONObject getSuperProperties() {
        synchronized (mSuperProperties) {
            return mSuperProperties.get();
        }
    }

    /**
     * 注册所有事件都有的公共属性
     *
     * @param superProperties 事件公共属性
     */
    @Override
    public void registerSuperProperties(JSONObject superProperties) {
        try {
            if (superProperties == null) {
                return;
            }
            assertPropertyTypes(EventType.REGISTER_SUPER_PROPERTIES, superProperties);
            synchronized (mSuperProperties) {
                JSONObject properties = mSuperProperties.get();
                SensorsDataUtils.mergeJSONObject(superProperties, properties);
                mSuperProperties.commit(properties);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 删除事件公共属性
     *
     * @param superPropertyName 事件属性名称
     */
    @Override
    public void unregisterSuperProperty(String superPropertyName) {
        try {
            synchronized (mSuperProperties) {
                JSONObject superProperties = mSuperProperties.get();
                superProperties.remove(superPropertyName);
                mSuperProperties.commit(superProperties);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 删除所有事件公共属性
     */
    @Override
    public void clearSuperProperties() {
        synchronized (mSuperProperties) {
            mSuperProperties.commit(new JSONObject());
        }
    }

    /**
     * 设置用户的一个或多个Profile。
     * Profile如果存在，则覆盖；否则，新创建。
     *
     * @param properties 属性列表
     */
    @Override
    public void profileSet(JSONObject properties) {
        try {
            trackEvent(EventType.PROFILE_SET, null, properties, null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 设置用户的一个Profile，如果之前存在，则覆盖，否则，新创建
     *
     * @param property 属性名称
     * @param value    属性的值，值的类型只允许为
     *                 {@link String}, {@link Number}, {@link Date}, {@link java.util.List}
     */
    @Override
    public void profileSet(String property, Object value) {
        try {
            trackEvent(EventType.PROFILE_SET, null, new JSONObject().put(property, value), null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 首次设置用户的一个或多个Profile。
     * 与profileSet接口不同的是，如果之前存在，则忽略，否则，新创建
     *
     * @param properties 属性列表
     */
    @Override
    public void profileSetOnce(JSONObject properties) {
        try {
            trackEvent(EventType.PROFILE_SET_ONCE, null, properties, null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 首次设置用户的一个Profile
     * 与profileSet接口不同的是，如果之前存在，则忽略，否则，新创建
     *
     * @param property 属性名称
     * @param value    属性的值，值的类型只允许为
     *                 {@link String}, {@link Number}, {@link Date}, {@link java.util.List}
     */
    @Override
    public void profileSetOnce(String property, Object value) {
        try {
            trackEvent(EventType.PROFILE_SET_ONCE, null, new JSONObject().put(property, value), null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 给一个或多个数值类型的Profile增加一个数值。只能对数值型属性进行操作，若该属性
     * 未设置，则添加属性并设置默认值为0
     *
     * @param properties 一个或多个属性集合
     */
    @Override
    public void profileIncrement(Map<String, ? extends Number> properties) {
        try {
            trackEvent(EventType.PROFILE_INCREMENT, null, new JSONObject(properties), null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 给一个数值类型的Profile增加一个数值。只能对数值型属性进行操作，若该属性
     * 未设置，则添加属性并设置默认值为0
     *
     * @param property 属性名称
     * @param value    属性的值，值的类型只允许为 {@link Number}
     */
    @Override
    public void profileIncrement(String property, Number value) {
        try {
            trackEvent(EventType.PROFILE_INCREMENT, null, new JSONObject().put(property, value), null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 给一个列表类型的Profile增加一个元素
     *
     * @param property 属性名称
     * @param value    新增的元素
     */
    @Override
    public void profileAppend(String property, String value) {
        try {
            final JSONArray append_values = new JSONArray();
            append_values.put(value);
            final JSONObject properties = new JSONObject();
            properties.put(property, append_values);
            trackEvent(EventType.PROFILE_APPEND, null, properties, null);
        } catch (final Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 给一个列表类型的Profile增加一个或多个元素
     *
     * @param property 属性名称
     * @param values   新增的元素集合
     */
    @Override
    public void profileAppend(String property, Set<String> values) {
        try {
            final JSONArray append_values = new JSONArray();
            for (String value : values) {
                append_values.put(value);
            }
            final JSONObject properties = new JSONObject();
            properties.put(property, append_values);
            trackEvent(EventType.PROFILE_APPEND, null, properties, null);
        } catch (final Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 删除用户的一个Profile
     *
     * @param property 属性名称
     */
    @Override
    public void profileUnset(String property) {
        try {
            trackEvent(EventType.PROFILE_UNSET, null, new JSONObject().put(property, true), null);
        } catch (final Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 删除用户所有Profile
     */
    @Override
    public void profileDelete() {
        try {
            trackEvent(EventType.PROFILE_DELETE, null, null, null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public boolean isDebugMode() {
        return mDebugMode.isDebugMode();
    }

    boolean isDebugWriteData() {
        return mDebugMode.isDebugWriteData();
    }

    private void disableDebugMode() {
        mDebugMode = DebugMode.DEBUG_OFF;
        enableLog(false);
        mServerUrl = mOriginServerUrl;
    }

    String getServerUrl() {
        return mServerUrl;
    }


    private void trackEvent(final EventType eventType, final String eventName, final JSONObject properties, final String
            originalDistinctId) {
        final EventTimer eventTimer;
        if (eventName != null) {
            synchronized (mTrackTimer) {
                eventTimer = mTrackTimer.get(eventName);
                mTrackTimer.remove(eventName);
            }
        } else {
            eventTimer = null;
        }

        SensorsDataThreadPool.getInstance().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    if (eventType.isTrack()) {
                        assertKey(eventName);
                    }
                    assertPropertyTypes(eventType, properties);

                    try {
                        JSONObject sendProperties;

                        if (eventType.isTrack()) {
                            sendProperties = new JSONObject(mDeviceInfo);

                            //之前可能会因为没有权限无法获取运营商信息，检测再次获取
                            try {
                                if (TextUtils.isEmpty(sendProperties.optString("$carrier"))) {
                                    String carrier = SensorsDataUtils.getCarrier(mContext);
                                    if (!TextUtils.isEmpty(carrier)) {
                                        sendProperties.put("$carrier", carrier);
                                    }
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }

                            synchronized (mSuperProperties) {
                                JSONObject superProperties = mSuperProperties.get();
                                SensorsDataUtils.mergeJSONObject(superProperties, sendProperties);
                            }

                            // 当前网络状况
                            String networkType = SensorsDataUtils.networkType(mContext);
                            sendProperties.put("$wifi", networkType.equals("WIFI"));
                            sendProperties.put("$network_type", networkType);


                            // GPS
                            try {
                                if (mGPSLocation != null) {
                                    sendProperties.put("$latitude", mGPSLocation.getLatitude());
                                    sendProperties.put("$longitude", mGPSLocation.getLongitude());
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }

                            // 屏幕方向
                            try {
                                String screenOrientation = getScreenOrientation();
                                if (!TextUtils.isEmpty(screenOrientation)) {
                                    sendProperties.put("$screen_orientation", screenOrientation);
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        } else if (eventType.isProfile()) {
                            sendProperties = new JSONObject();
                        } else {
                            return;
                        }

                        String libDetail = null;
                        if (null != properties) {
                            try {
                                if (properties.has("$lib_detail")) {
                                    libDetail = properties.getString("$lib_detail");
                                    properties.remove("$lib_detail");
                                }
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                            SensorsDataUtils.mergeJSONObject(properties, sendProperties);
                        }

                        if (null != eventTimer) {
                            try {
                                Double duration = Double.valueOf(eventTimer.duration());
                                if (duration > 0) {
                                    sendProperties.put("event_duration", duration);
                                }
                            } catch (Exception e) {
                                //ignore
                                e.printStackTrace();
                            }
                        }

                        JSONObject libProperties = new JSONObject();
                        libProperties.put("$lib", "Android");
                        libProperties.put("$lib_version", VERSION);

                        if (mDeviceInfo.containsKey("$app_version")) {
                            libProperties.put("$app_version", mDeviceInfo.get("$app_version"));
                        }

                        //update lib $app_version from super properties
                        JSONObject superProperties = mSuperProperties.get();
                        if (superProperties != null) {
                            if (superProperties.has("$app_version")) {
                                libProperties.put("$app_version", superProperties.get("$app_version"));
                            }
                        }

                        final JSONObject dataObj = new JSONObject();

                        try {
                            Random random = new Random();
                            dataObj.put("_track_id", random.nextInt());
                        } catch (Exception e) {
                            //ignore
                        }

                        final long now = System.currentTimeMillis();
                        dataObj.put("time", now);
                        dataObj.put("type", eventType.getEventType());

                        try {
                            if (sendProperties.has("$project")) {
                                dataObj.put("project", sendProperties.optString("$project"));
                                sendProperties.remove("$project");
                            }

                            if (sendProperties.has("$token")) {
                                dataObj.put("token", sendProperties.optString("$token"));
                                sendProperties.remove("$token");
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }

                        dataObj.put("properties", sendProperties);
                        if (!TextUtils.isEmpty(getLoginId())) {
                            dataObj.put("distinct_id", getLoginId());
                        } else {
                            dataObj.put("distinct_id", getAnonymousId());
                        }
                        dataObj.put("lib", libProperties);

                        if (eventType == EventType.TRACK) {
                            dataObj.put("event", eventName);
                            //是否首日访问
                            sendProperties.put("$is_first_day", isFirstDay());
                        } else if (eventType == EventType.TRACK_SIGNUP) {
                            dataObj.put("event", eventName);
                            dataObj.put("original_id", originalDistinctId);
                        }

                        libProperties.put("$lib_method", "code");


                        if (TextUtils.isEmpty(libDetail)) {
                            StackTraceElement[] trace = (new Exception()).getStackTrace();
                            if (trace.length > 2) {
                                StackTraceElement traceElement = trace[2];
                                libDetail = String.format("%s##%s##%s##%s", traceElement
                                                .getClassName(), traceElement.getMethodName(), traceElement.getFileName(),
                                        traceElement.getLineNumber());
                            }
                        }

                        libProperties.put("$lib_detail", libDetail);
                        mMessages.enqueueEventMessage(eventType.getEventType(), dataObj);
                        SALog.i(TAG, "track event:\n" + JSONUtils.formatJson(dataObj.toString()));
                    } catch (JSONException e) {
                        throw new InvalidDataException("Unexpected property");
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
    }

    private boolean isFirstDay() {
        String firstDay = mFirstDay.get();
        if (firstDay == null) {
            return true;
        }
        String current = mIsFirstDayDateFormat.format(System.currentTimeMillis());
        return firstDay.equals(current);
    }

    private void assertPropertyTypes(EventType eventType, JSONObject properties) throws
            InvalidDataException {
        if (properties == null) {
            return;
        }

        for (Iterator iterator = properties.keys(); iterator.hasNext(); ) {
            String key = (String) iterator.next();

            // Check Keys
            assertKey(key);

            try {
                Object value = properties.get(key);

                if (!(value instanceof String || value instanceof Number || value
                        instanceof JSONArray || value instanceof Boolean || value instanceof Date)) {
                    throw new InvalidDataException("The property value must be an instance of "
                            + "String/Number/Boolean/JSONArray. [key='" + key + "', value='" + value.toString()
                            + "']");
                }

                if ("app_crashed_reason".equals(key)) {
                    if (value instanceof String && !key.startsWith("$") && ((String) value).length() > 8191 * 2) {
                        SALog.d(TAG, "The property value is too long. [key='" + key
                                + "', value='" + value.toString() + "']");
                    }
                } else {
                    if (value instanceof String && !key.startsWith("$") && ((String) value).length() > 8191) {
                        SALog.d(TAG, "The property value is too long. [key='" + key
                                + "', value='" + value.toString() + "']");
                    }
                }
            } catch (JSONException e) {
                throw new InvalidDataException("Unexpected property key. [key='" + key + "']");
            }
        }
    }

    private void assertKey(String key) throws InvalidDataException {
        if (null == key || key.length() < 1) {
            throw new InvalidDataException("The key is empty.");
        }
        if (!(KEY_PATTERN.matcher(key).matches())) {
            throw new InvalidDataException("The key '" + key + "' is invalid.");
        }
    }

    private void assertDistinctId(String key) throws InvalidDataException {
        if (key == null || key.length() < 1) {
            throw new InvalidDataException("The distinct_id or original_id or login_id is empty.");
        }
        if (key.length() > 255) {
            throw new InvalidDataException("The max length of distinct_id or original_id or login_id is 255.");
        }
    }

    // 可视化埋点功能最低API版本
    static final int VTRACK_SUPPORTED_MIN_API = 16;

    static Boolean ENABLE_LOG = false;

    private static final Pattern KEY_PATTERN = Pattern.compile(
            "^((?!^distinct_id$|^original_id$|^time$|^properties$|^id$|^first_id$|^second_id$|^users$|^events$|^event$|^user_id$|^date$|^datetime$)[a-zA-Z_$][a-zA-Z\\d_$]{0,99})$",
            Pattern.CASE_INSENSITIVE);

    // Maps each token to a singleton Yodo1SensorsDataAPI instance
    private static final Map<Context, Yodo1SensorsDataAPI> sInstanceMap = new HashMap<>();
    private static final SharedPreferencesLoader sPrefsLoader = new SharedPreferencesLoader();
    private static SensorsDataGPSLocation mGPSLocation;

    // Configures
    /* SensorsAnalytics 地址 */
    private String mServerUrl;
    private String mOriginServerUrl;
    /* Debug模式选项 */
    private DebugMode mDebugMode;
    /* Flush时间间隔 */
    private int mFlushInterval;
    /* Flush数据量阈值 */
    private int mFlushBulkSize;
    /* AndroidId 作为默认匿名Id */
    private boolean mEnableAndroidId;

    private final Context mContext;
    private final AnalyticsMessages mMessages;
    private final PersistentDistinctId mDistinctId;
    private final PersistentLoginId mLoginId;
    private final PersistentSuperProperties mSuperProperties;
    private final PersistentFirstStart mFirstStart;
    private final PersistentFirstDay mFirstDay;
    private final Map<String, Object> mDeviceInfo;
    private final Map<String, EventTimer> mTrackTimer;
    private int mFlushNetworkPolicy = NetworkType.TYPE_3G | NetworkType.TYPE_4G | NetworkType.TYPE_WIFI;
    private long mMaxCacheSize = 32 * 1024 * 1024; //default 32MB

    private SensorsDataScreenOrientationDetector mOrientationDetector;

    private static final SimpleDateFormat mIsFirstDayDateFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());

    private static final String TAG = "Yodo1SensorsDataAPI";

    // SDK版本
    static final String VERSION = "1.10.4-mini";
    // 原有代码是第二行，需要的值为游戏包名加上.sa，现修改所有引用此静态属性的位置，改成SysUtils.getPackageName(mContext)+此属性的后缀.sa
    public static final String SENSORS_DATA_API_FILE_NAME = ".sa";
//    public static final String SENSORS_DATA_API_FILE_NAME = "you package name"+".sa";
}
