package com.yodo1.sdk.kit;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.IBinder;
import android.os.IInterface;
import android.os.Looper;
import android.os.Parcel;
import android.os.RemoteException;
import android.support.v4.content.ContextCompat;
import android.telephony.TelephonyManager;
import android.text.TextUtils;

import java.io.IOException;
import java.util.concurrent.LinkedBlockingQueue;

/**
 * @author yodo1
 */
public class Yodo1DeviceUtils {
    private static final String TAG = "Yodo1DeviceUtils";


    /**
     * 获取手机品牌
     *
     * @return 手机品牌
     */
    public static String getPhoneBrand() {
        return android.os.Build.BRAND;
    }

    /**
     * 获取手机型号
     *
     * @return 手机型号
     */
    public static String getPhoneVersion() {
        return android.os.Build.MODEL;
    }

    /**
     * 获取手机版本
     *
     * @return 系统版本
     */
    public static String getOsVersion() {
        return android.os.Build.VERSION.RELEASE;
    }


    /**
     * 获取MAC地址
     *
     * @param context Context
     * @return mac address
     */
    public static String getMacAddress(Context context) {
        String macStr = "";
        WifiManager wifi = (WifiManager) context.getApplicationContext().getSystemService(Context.WIFI_SERVICE);
        if (wifi != null) {
            WifiInfo info = wifi.getConnectionInfo();
            if (info != null) {
                macStr = info.getMacAddress();
                if (TextUtils.isEmpty(macStr)) {
                    macStr = "";
                }
            }
        }
        return macStr;
    }


    /**
     * 这个方法是耗时的，不能在主线程调用
     *
     * @param context Context
     * @return google gaid
     * @throws Exception
     */
    public static String getGoogleAdId(Context context) throws Exception {
        if (Looper.getMainLooper() == Looper.myLooper()) {
            throw new IllegalStateException("Cannot call in the main thread, You must call in the other thread");
        }

        try {
            PackageManager pm = context.getPackageManager();
            pm.getPackageInfo("com.android.vending", 0);
        } catch (Exception e) {
            throw e;
        }

        AdvertisingConnection connection = new AdvertisingConnection();
        Intent intent = new Intent("com.google.android.gms.ads.identifier.service.START");
        intent.setPackage("com.google.android.gms");
        if (context.bindService(intent, connection, Context.BIND_AUTO_CREATE)) {
            try {
                AdvertisingInterface adInterface = new AdvertisingInterface(connection.getBinder());
                return adInterface.getId();
            } catch (Exception e) {
                throw e;
            } finally {
                context.unbindService(connection);
            }
        }
        throw new IOException("Google Play connection failed");
    }

    /**
     * 获取设备ID。
     * <p>
     * 目前采取的策略是：第一次读取后在本地存储至本地，以后都从存储空间中读取。
     * <p>
     * 在6.0或者没有READ_PHONE_STATE的设备上，采取androidId + serial的方式。
     * 在非6.0并且有READ_PHONE_STATE权限的设备上，依旧采取老的imei方式
     *
     * @param ctx Context
     * @return 可以的话返回曾经的设备ID, 否则返回新策略生成的
     */
    public static String getDeviceId(Context ctx) {

        String deviceId = YFileUtils.readFilesFromSDCard(YFileUtils.YODO1FILE);
        if (TextUtils.isEmpty(deviceId) || deviceId.contains("openYodo1Log")) {
            deviceId = getDeviceIdForSave(ctx);
        }

        if (TextUtils.isEmpty(deviceId)) {
            deviceId = getIMEI(ctx);
        }

        if (TextUtils.isEmpty(deviceId)) {
            deviceId = java.util.UUID.randomUUID().toString();
            saveDeviceId(ctx, deviceId);
            YFileUtils.writeFilesToSDCard(deviceId, YFileUtils.YODO1FILE);
        }
        return deviceId;
    }

    @SuppressLint("MissingPermission")
    private static String getIMEI(Context context) {
        String imei = "";
        int permissionCheck = ContextCompat.checkSelfPermission(context, Manifest.permission.READ_PHONE_STATE);
        if (permissionCheck == PackageManager.PERMISSION_GRANTED) {

            TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            if (telephonyManager != null) {
                imei = telephonyManager.getDeviceId();
                if (TextUtils.isEmpty(imei) || "0".equals(imei) || "000000000000000".equals(imei)) {
                    imei = "";
                }
            }
        } else {
            YLog.d("没READ_PHONE_STATE权限,无法读取IMEI");
        }

        return imei;
    }

    private static void saveDeviceId(Context context, String value) {
        SharedPreferences sp = context.getSharedPreferences("sp_yodo1games", Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sp.edit();
        editor.putString("user_device_id", value);
        editor.apply();
    }

    private static String getDeviceIdForSave(Context context) {
        SharedPreferences sp = context.getSharedPreferences("sp_yodo1games", Context.MODE_PRIVATE);
        return sp.getString("user_device_id", null);
    }

    private static final class AdvertisingConnection implements ServiceConnection {
        private final LinkedBlockingQueue<IBinder> queue = new LinkedBlockingQueue<>(1);
        boolean retrieved = false;

        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            try {
                this.queue.put(service);
            } catch (InterruptedException localInterruptedException) {
            }
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
        }

        public IBinder getBinder() throws InterruptedException {
            if (this.retrieved) {
                throw new IllegalStateException();
            }
            this.retrieved = true;
            return this.queue.take();
        }
    }

    private static final class AdvertisingInterface implements IInterface {
        private IBinder binder;

        public AdvertisingInterface(IBinder pBinder) {
            binder = pBinder;
        }

        @Override
        public IBinder asBinder() {
            return binder;
        }

        public String getId() throws RemoteException {
            Parcel data = Parcel.obtain();
            Parcel reply = Parcel.obtain();
            String id;
            try {
                data.writeInterfaceToken("com.google.android.gms.ads.identifier.internal.IAdvertisingIdService");
                binder.transact(1, data, reply, 0);
                reply.readException();
                id = reply.readString();
            } finally {
                reply.recycle();
                data.recycle();
            }
            return id;
        }


        public boolean isLimitAdTrackingEnabled(boolean paramBoolean) throws RemoteException {
            Parcel data = Parcel.obtain();
            Parcel reply = Parcel.obtain();
            boolean limitAdTracking;
            try {
                data.writeInterfaceToken("com.google.android.gms.ads.identifier.internal.IAdvertisingIdService");
                data.writeInt(paramBoolean ? 1 : 0);
                binder.transact(2, data, reply, 0);
                reply.readException();
                limitAdTracking = 0 != reply.readInt();
            } finally {
                reply.recycle();
                data.recycle();
            }
            return limitAdTracking;
        }
    }
}
