package com.yodo1.sensor;

import android.content.Context;
import android.text.TextUtils;
import android.util.Log;

import com.sensorsdata.analytics.android.minisdk.Yodo1SensorsDataAPI;
import com.sensorsdata.analytics.android.minisdk.util.JSONUtils;
import com.yodo1.sdk.kit.SysUtils;
import com.yodo1.sdk.kit.YLog;
import com.yodo1.sdk.kit.Yodo1SdkUtils;

import org.json.JSONObject;

/**
 * @author :huangguofeng
 * date    :2020-03-24
 * package :com.yodo1.sensor
 * desc    : 神策埋点中转类
 */
public class Yodo1SensorHelper {
    private static final String TAG = "[Yodo1SensorHelper] ";
    private static Yodo1SensorHelper instance;
    /**
     * 初始化状态
     */
    private boolean isInit = false;

    public static Yodo1SensorHelper getInstance() {
        if (instance == null) {
            instance = new Yodo1SensorHelper();
        }
        return instance;
    }

    /**
     * @param context   App 的 Context
     * @param serverUrl 用于收集事件的服务地址
     * @param debugMode Debug模式 true: Yodo1SensorsDataAPI.DebugMode.DEBUG_AND_TRACK, false: Yodo1SensorsDataAPI.DebugMode.DEBUG_OFF
     */
    public void init(Context context, String serverUrl, boolean debugMode) {
        if (isInit) {
            Log.d(TAG, "sensor 禁止多次初始化 ");
            return;
        }
        isInit = true;
        Log.d(TAG, "The sensor init is called...");

        Yodo1SensorsDataAPI.sharedInstance(context, serverUrl, getDebugMode(debugMode));
        registerSuperProperties(context);
    }

    public boolean isInitialized() {
        return isInit;
    }

    private Yodo1SensorsDataAPI.DebugMode getDebugMode(boolean debugMode) {
        if (debugMode) {
            return Yodo1SensorsDataAPI.DebugMode.DEBUG_AND_TRACK;
        }
        return Yodo1SensorsDataAPI.DebugMode.DEBUG_OFF;
    }

    /**
     * 备用方法
     */
    public void onDestroy() {
        YLog.d(TAG + "The onDestroy of  method is called...");
    }

    /**
     * 1:请在初始化 SDK 后立即调用 registerSuperProperties() 方法，确保每个事件都会添加已设置的公共属性。
     * 2:神策事件属性大小写敏感（详见官网文档:数据格式），设置公共属性前需确保事件表中不存在仅大小写不同的同名属性，否则会导致所有事件因存在同名属性而不能入库。
     * 3:重复调用registerSuperProperties()会覆盖之前已设置的公共属性，公共属性会保存在App本地SharedPreferences中
     * 4:可以通过 unregisterSuperProperty() 删除一个静态公共属性，使用clearSuperProperties()会删除所有已设置的事件公共属性。
     */
    private void registerSuperProperties(Context context) {
        if (!isInit) {
            YLog.d(TAG + "sensor 没有初始化 无法registerSuperProperties ");
            return;
        }
        Log.d(TAG, " registerSuperProperties  is call ");
        try {
            JSONObject properties = new JSONObject();

            properties.put("gameName", SysUtils.getAppName(context) + "");
            properties.put("gameVersion", SysUtils.getVersionName(context) + "");
            properties.put("gameBundleId", SysUtils.getPackageName(context) + "");
            properties.put("sdkType", Yodo1SdkUtils.getSdkType(context) + "");
            properties.put("sdkVersion", Yodo1SdkUtils.getSdkVersion(context) + "");
            properties.put("masVersion", Yodo1SdkUtils.getMasSdkVersion(context) + "");
            properties.put("publishChannelCode", Yodo1SdkUtils.getPublishCode(context) + "");
            Log.i(TAG, " registerSuperProperties content :\n" + JSONUtils.formatJson(properties.toString()));
            Yodo1SensorsDataAPI.sharedInstance().registerSuperProperties(properties);
        } catch (Exception e) {
            YLog.d(TAG + " registerSuperProperties  Exception e : " + e.getMessage());
        }
    }

    /**
     * 1:当事件公共属性和事件属性的 Key 冲突时，事件属性优先级最高，它会覆盖事件公共属性。
     * 2:事件属性是一个 JSONObject 对象
     * 3:JSONObject 中每个元素描述一个属性，Key 为属性名称，必需是 String 类型
     * 4:JSONObject 中，每个元素的Value是属性的值，支持String、Number、Boolean、JSONArray 和 Date，对于JSONArray，其中所有元素必须是String类型
     *
     * @param eventId    埋点事件名
     * @param properties 埋点事件附加属性
     */
    public void track(String eventId, JSONObject properties) {
        if (!isInit) {
            YLog.d(TAG + "sensor 没有初始化 无法track eventId ： " + eventId);
            return;
        }
        try {
            if (!TextUtils.isEmpty(eventId) && properties != null) {
                Yodo1SensorsDataAPI.sharedInstance().track(eventId, properties);
            } else {
                YLog.d(TAG + " eventId is null or properties is null ");
            }
        } catch (Exception e) {
            YLog.d(TAG + " track  Exception e : " + e.getMessage());
        }
    }

    /**
     * 记录用户属性
     *
     * @param properties 用户属性
     */
    public void profileSet(JSONObject properties) {
        if (!isInit) {
            YLog.d(TAG + "profileSet fail : sensor 没有初始化 无法记录用户属性");
            return;
        }
        try {
            if (properties != null) {
                Yodo1SensorsDataAPI.sharedInstance().profileSet(properties);
            } else {
                YLog.d(TAG + "profileSet fail : properties is null  ");
            }
        } catch (Exception e) {
            YLog.d(TAG + " profileSet  Exception e : " + e.getMessage());
        }
    }

    /**
     * 获取神策绑定后的登录ID
     */
    public String getLoginId() {
        if (!isInit) {
            YLog.d(TAG + "getLoginId fail : sensor 没有初始化 无法记录用户属性");
            return "";
        }
        return Yodo1SensorsDataAPI.sharedInstance().getLoginId();
    }

    /**
     * 获取神策绑定之前自动生成的匿名ID
     */
    public String getAnonymousId() {
        if (!isInit) {
            YLog.d(TAG + "getAnonymousId fail : sensor 没有初始化 无法记录用户属性");
            return "";
        }
        return Yodo1SensorsDataAPI.sharedInstance().getAnonymousId();
    }

    /**
     * 绑定登录ID
     *
     * @param loginId 自己的用户系统标识用户的ID
     */
    public void login(String loginId) {
        if (!isInit) {
            YLog.d(TAG + "login fail : sensor 没有初始化 无法记录用户属性");
            return;
        }
        Yodo1SensorsDataAPI.sharedInstance().login(loginId);
    }

    /**
     * 解绑登录ID
     */
    public void loginOut() {
        if (!isInit) {
            YLog.d(TAG + "loginOut fail : sensor 没有初始化 无法记录用户属性");
            return;
        }
        Yodo1SensorsDataAPI.sharedInstance().logout();
    }
}
