package com.yodo1.sdk.kit;

import android.app.ActivityManager;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.Signature;
import android.os.Build;
import android.os.Environment;
import android.text.TextUtils;

import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;

/**
 * @author yodo1
 */
public class SysUtils {
    /**
     * 获得应用签名
     *
     * @param ctx Context
     * @return 该app的签名
     */
    public static String getSignature(Context ctx) {
        try {
            PackageManager manager = ctx.getPackageManager();
            PackageInfo packageInfo = manager.getPackageInfo(ctx.getPackageName(), PackageManager.GET_SIGNATURES);
            Signature[] signatures = packageInfo.signatures;
            StringBuilder sb = new StringBuilder();
            for (Signature signature : signatures) {
                sb.append(signature.toCharsString());
            }
            return sb.toString();
        } catch (Exception e) {
            YLog.d("getSignature", e);
        }
        return null;
    }

    /**
     * 获取sd卡目录
     *
     * @return sdcard path
     */
    public static String getSDCardPath() {
        String sdcardDir = null;
        //判断SDCard是否存在
        boolean sdcardExist = Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED);
        if (sdcardExist) {
            sdcardDir = Environment.getExternalStorageDirectory().toString();
        }
        return sdcardDir;
    }

    /**
     * 读取assets里的txt文件
     *
     * @param context  Context
     * @param filename 文件名
     * @return 内容
     */
    public static String loadAssetsForTxt(Context context, String filename) {
        String content = "";
        try {
            InputStream is = context.getAssets().open(filename);
            int size = is.available();

            // Read the entire asset into a local byte buffer.
            byte[] buffer = new byte[size];
            is.read(buffer);
            is.close();

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                content = new String(buffer, StandardCharsets.UTF_8);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return content;
    }

    /**
     * 获取当前进程的名字
     *
     * @param cxt Context
     * @param pid 进程id
     * @return 进程名
     */
    public static String getProcessName(Context cxt, int pid) {
        ActivityManager am = (ActivityManager) cxt.getSystemService(Context.ACTIVITY_SERVICE);
        try {
            List<ActivityManager.RunningAppProcessInfo> runningApps = am.getRunningAppProcesses();
            if(runningApps == null) {
                return "";
            }

            for (ActivityManager.RunningAppProcessInfo procInfo : runningApps) {
                if (procInfo.pid == pid) {
                    return procInfo.processName;
                }
            }
        } catch (Exception e) {
            YLog.d("getProcessName", e);
        }
        return "";
    }

    public static String getCurrentProcessName(Context cxt) {
        return getProcessName(cxt, android.os.Process.myPid());
    }

    public static boolean isMainProcess(Context cxt) {
        return TextUtils.equals(cxt.getPackageName(), getCurrentProcessName(cxt));
    }

    /**
     * 获得程序版本号
     *
     * @param context Context
     * @return versionCode
     */
    public static int getVersionCode(Context context) {
        try {
            PackageInfo pinfo = context.getApplicationContext().getPackageManager().getPackageInfo(context.getApplicationContext().getPackageName(), 0);
            return pinfo.versionCode;
        } catch (Exception e) {
            YLog.d("getVersionCode", e);
        }
        return -1;
    }

    /**
     * 获得当前版本名
     *
     * @param context Context
     * @return versionName
     */
    public static String getVersionName(Context context) {
        try {
            PackageInfo pinfo = context.getApplicationContext().getPackageManager().getPackageInfo(context.getApplicationContext().getPackageName(), 0);
            return pinfo.versionName;
        } catch (Exception e) {
            YLog.d("getVersionName", e);
        }
        return "";
    }

    /**
     * 获得当前APP名字
     *
     * @param context Context
     * @return app名字
     */
    public static String getAppName(Context context) {
        try {
            PackageManager pm = context.getApplicationContext().getPackageManager();
            ApplicationInfo applicationInfo = pm.getApplicationInfo(context.getPackageName(), 0);
            return (String) pm.getApplicationLabel(applicationInfo);
        } catch (Exception e) {
            YLog.d("getAppName", e);
        }
        return "";
    }

    /**
     * 判断当前应用是否正在运行。是否是前台进程。
     *
     * @param context Context
     * @return 是否运行
     */
    public static boolean isAppOnForeground(Context context) {
        try {
            ActivityManager am = (ActivityManager) context.getApplicationContext().getSystemService(Context.ACTIVITY_SERVICE);
            String packageName = context.getApplicationContext().getPackageName();
            List<ActivityManager.RunningAppProcessInfo> appProcesses = am.getRunningAppProcesses();
            for (ActivityManager.RunningAppProcessInfo appProcess : appProcesses) {
                if (appProcess.importance == ActivityManager.RunningAppProcessInfo.IMPORTANCE_FOREGROUND && appProcess.processName.equals(packageName)) {
                    return true;
                }
            }
        } catch (Exception e) {
            YLog.d("isAppOnForeground", e);
        }
        return false;
    }

    public static String getTopActivity(Context context) {
        try {
            ActivityManager am = (ActivityManager) context.getApplicationContext().getSystemService(Context.ACTIVITY_SERVICE);
            List<ActivityManager.RunningTaskInfo> runningTasks = am.getRunningTasks(1);
            return runningTasks.get(0).topActivity.getClassName();
        } catch (Exception e) {
            return "";
        }
    }

    /**
     * 获得当前游戏程序包名
     *
     * @param context Context
     * @return packageName
     */
    public static String getPackageName(Context context) {
        try {
            PackageInfo pinfo = context.getApplicationContext().getPackageManager().getPackageInfo(context.getApplicationContext().getPackageName(), 0);
            return pinfo.packageName;
        } catch (Exception e) {
            YLog.d("getPackageName", e);
        }
        return "";
    }
}
