package com.yodo1.onlineconfig;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.Context;
import android.location.Location;
import android.location.LocationManager;
import android.os.Looper;
import android.text.TextUtils;

import com.yodo1.android.ops.net.HttpListener;
import com.yodo1.android.ops.net.HttpStringListener;
import com.yodo1.android.ops.net.Yodo1HttpManage;
import com.yodo1.android.ops.net.Yodo1RequestListener;
import com.yodo1.android.ops.net.Yodo1SDKResponse;
import com.yodo1.nohttp.NoHttp;
import com.yodo1.nohttp.rest.Request;
import com.yodo1.nohttp.rest.Response;
import com.yodo1.sdk.kit.MD5EncodeUtil;
import com.yodo1.sdk.kit.SysUtils;
import com.yodo1.sdk.kit.YLog;
import com.yodo1.sdk.kit.Yodo1DeviceUtils;
import com.yodo1.sdk.kit.Yodo1PermissonUtils;
import com.yodo1.sdk.kit.Yodo1SdkUtils;
import com.yodo1.sdk.kit.Yodo1SharedPreferences;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.concurrent.Executors;

/**
 * Created by yodo1 on 2017/8/7.
 *
 * @author yodo1
 */
public class Yodo1OnlineConfig {
    private static final String TAG = "[Yodo1OnlineConfig] ";
    private static final String KEY_CACHE_FILE_JSON = "onlineconfig";
    private static final String URL_ONLINE_CONFIG_PRODUCT = "https://olc.yodo1api.com/config/getDataV2/";
    private static final String URL_ONLINE_CONFIG_CDN = "https://ocd.yodo1api.com/configfiles/";
    private static final String KEY_DATA_IDENTIFER = "data_identifer";
    private static final String KEY_LOCATION_IDENTIFER = "location_identifer";
    private static final String KEY_LOCATION_IDENTIFER_TTL = "location_identifer_ttl";
    private static final String KEY_GAME_APP_KEY = "game_appkey";
    private static final String KEY_GAME_CHANNEL_CODE = "game_channelcode";
    private static final String KEY_GAME_VERSION_NAME = "game_version";
    private static final String KEY_GAME_PACKAGE_NAME = "game_packagename";
    private static final String KEY_TIMESTAMP_GET_DATA = "timestamp_getdata";
    private static final String KEY_ONLINE_CONFIG_TEST_INFO = "OnlineConfigParams_TestInfo";
    private static final String KEY_ONLINE_CONFIG_TEST_DEVICE = "OnlineConfigParams_TestDevice";
    private static final String KEY_ONLINE_CONFIG_TEST_DEVICE_SOURCE = "OnlineConfigParams_TestDeviceSource";
    private static final String KEY_ONLINE_CONFIG_REPORT_FIELDS = "report_fields";

    @SuppressLint("StaticFieldLeak")
    private static volatile Yodo1OnlineConfig instance;
    @SuppressLint("StaticFieldLeak")
    private static Context context;

    private String buildEnvUrl = URL_ONLINE_CONFIG_PRODUCT;
    private String appKey;
    private static long time;

    private String dataIdentifier;
    private String locationIdentifier;
    /**
     * 手机当前经度，保留小数点后两位
     */
    private String locationLongitude;
    /**
     * 手机当前纬度，保留小数点后两位
     */
    private String locationLatitude;
    private JSONObject dataObj;
    private JSONObject testObj;
    private volatile boolean initialized;

    private Yodo1OnlineConfig() {
        this.initialized = false;
        time = 0;
        this.dataObj = null;
        this.testObj = null;
        this.locationLongitude = "";
        this.locationLatitude = "";
    }

    public static Yodo1OnlineConfig getInstance() {
        if (instance == null) {
            synchronized (Yodo1OnlineConfig.class) {
                if (instance == null) {
                    instance = new Yodo1OnlineConfig();
                }
            }
        }
        return instance;
    }

    public void onDestroy() {
        time = 0;
        this.initialized = false;
        this.dataObj = null;
        this.testObj = null;
        this.locationLongitude = "";
        this.locationLatitude = "";
    }

    public void setBuildEnvironment(String url) {
        this.buildEnvUrl = url;
    }

    private String getRequestUrl() {
        return this.buildEnvUrl;
    }

    private JSONObject getRequestBody(String deviceId) {
        String currentTimeMillis = System.currentTimeMillis() + "";

        JSONObject dataJson = new JSONObject();
        try {
            dataJson.put("game_appkey", this.appKey);
            dataJson.put("device_id", deviceId);
            dataJson.put("version", SysUtils.getVersionName(context));
            dataJson.put("channel", Yodo1SdkUtils.getPublishCode(context));
            dataJson.put("sdk_type", Yodo1SdkUtils.getSdkType(context));
            dataJson.put("sdk_version", Yodo1SdkUtils.getSdkVersion(context));
            dataJson.put("data_identifer", this.dataIdentifier);
            dataJson.put("location_identifer", this.locationIdentifier);
            dataJson.put("location_lng", this.locationLongitude);
            dataJson.put("location_lat", this.locationLatitude);
            dataJson.put("timestamp", currentTimeMillis);
            dataJson.put("sign", getSign(this.appKey, SysUtils.getVersionName(context), Yodo1SdkUtils.getPublishCode(context), currentTimeMillis));
        } catch (Exception e) {
            YLog.d(TAG, e);
        }
        return dataJson;
    }

    private String getSign(String appKey, String versionName, String publishCode, String currentTimeMillis) {
        return MD5EncodeUtil.MD5Encode(appKey + versionName + publishCode + currentTimeMillis + "yodo1");
    }

    private void checkIdentifier() {
        String data = Yodo1SharedPreferences.getString(context, KEY_DATA_IDENTIFER);
        this.dataIdentifier = !TextUtils.isEmpty(data) ? !"null".equals(data) ? data : "0" : "0";
        String cacheFile = JsonCacheUtils.readJson(context, KEY_CACHE_FILE_JSON);
        if (TextUtils.isEmpty(cacheFile)) {
            YLog.d("onlineconfig.json not exist ");
            this.dataIdentifier = "0";
        } else {
            YLog.d("onlineconfig.json exist " + cacheFile.length());
        }

        this.locationIdentifier = Yodo1SharedPreferences.getString(context, KEY_LOCATION_IDENTIFER);
        if (!TextUtils.isEmpty(this.locationIdentifier)) {
            String locationIdentifierTtl = Yodo1SharedPreferences.getString(context, KEY_LOCATION_IDENTIFER_TTL);
            String getDataTimestamp = Yodo1SharedPreferences.getString(context, KEY_TIMESTAMP_GET_DATA);
            try {
                long timeTtl = Long.parseLong(locationIdentifierTtl);
                long getDataTime = Long.parseLong(getDataTimestamp);
                long nowTime = Long.parseLong(System.currentTimeMillis() + "");

                //在线参数本次请求的时间戳和上次请求返回的时间戳比较   大于上次请求返回的有效时长  清空location_identifer和data_identifer
                if ((getDataTime - nowTime) > (timeTtl * 60 * 60 * 1000)) {
                    this.locationIdentifier = "";
                    this.dataIdentifier = "0";
                }
            } catch (NumberFormatException e) {
                this.locationIdentifier = "";
            }
        } else {
            this.locationIdentifier = "";
        }

        Yodo1SharedPreferences.put(context, KEY_DATA_IDENTIFER, this.dataIdentifier);
        Yodo1SharedPreferences.put(context, KEY_LOCATION_IDENTIFER, this.locationIdentifier);
    }

    /**
     * 根据接口获取在线参数
     *
     * @param listener 回调
     */
    private void postRequest(final HttpStringListener listener) {
        Runnable onlineConfig = new Runnable() {
            @Override
            public void run() {
                Looper.prepare();
                YLog.i(TAG, "子线程开始请求OnLineConfig数据。");
                String deviceId = null;
                try {
                    deviceId = Yodo1DeviceUtils.getGoogleAdId(context);
                } catch (Exception e) {
                    YLog.i(TAG, e);
                }
                if (TextUtils.isEmpty(deviceId)) {
                    deviceId = Yodo1DeviceUtils.getDeviceId(context);
                }
                Yodo1SharedPreferences.put(context, "YDEVICEID", deviceId);//MAS展示信息用。不是严格yodo1DeviceId.zjq2020年08月25日13:56:14
                checkIdentifier();
                Yodo1HttpManage.getInstance().post(getRequestUrl(), getRequestBody(deviceId), listener);
                Looper.loop();
            }
        };
        Executors.newCachedThreadPool().execute(onlineConfig);
    }

    /**
     * 获取广告数据
     *
     * @return Json Object
     * @throws JSONException Json Exception
     */
    public JSONObject getJsonData() throws JSONException {
        if (context == null) {
            return null;
        }

        if (this.dataObj != null) {
            return this.dataObj;
        }

        String data = JsonCacheUtils.readJson(context, KEY_CACHE_FILE_JSON);
        if (!TextUtils.isEmpty(data)) {
            this.dataObj = new JSONObject(data);
        }
        return this.dataObj;
    }

    /**
     * 判断本地是否有数据
     */
    public boolean hasData() {
        boolean ret = false;
        if (context == null) {
            return false;
        }
        String data = JsonCacheUtils.readJson(context, KEY_CACHE_FILE_JSON);
        if (!TextUtils.isEmpty(data)) {
            ret = true;
        }
        return ret;
    }

    /**
     * 获取高级测试信息
     */
    public JSONObject getJsonTestData() throws JSONException {
        if (this.testObj != null) {
            return this.testObj;
        }

        String data = Yodo1SharedPreferences.getString(context, KEY_ONLINE_CONFIG_TEST_INFO);
        if (!TextUtils.isEmpty(data)) {
            this.testObj = new JSONObject(data);
        }
        return this.testObj;
    }

    private void saveTestModeData(JSONObject jsonObj) {
        if (jsonObj == null) {
            return;
        }
        //存储Test Mode功能信息
        Yodo1SharedPreferences.put(context, KEY_ONLINE_CONFIG_TEST_DEVICE, jsonObj.optBoolean("is_test_model"));
        Yodo1SharedPreferences.put(context, KEY_ONLINE_CONFIG_TEST_DEVICE_SOURCE, jsonObj.optString("device_source"));
        JSONObject testListObject = jsonObj.optJSONObject("test_list");
        Yodo1SharedPreferences.put(context, KEY_ONLINE_CONFIG_TEST_INFO, testListObject == null ? "" : testListObject.toString());
    }

    private void saveReportFields(JSONObject jsonObject) {
        if (jsonObject == null) {
            return;
        }

        JSONArray array = jsonObject.optJSONArray(KEY_ONLINE_CONFIG_REPORT_FIELDS);
        Yodo1SharedPreferences.put(context, KEY_ONLINE_CONFIG_REPORT_FIELDS, array == null ? "" : array.toString());
    }

    public JSONArray getReportFields() {
        String reportFields = Yodo1SharedPreferences.getString(context, KEY_ONLINE_CONFIG_REPORT_FIELDS);
        if (TextUtils.isEmpty(reportFields)) {
            return null;
        }
        JSONArray array = null;
        try {
            array = new JSONArray(reportFields);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return array;
    }

    /**
     * 提供给外部调用
     */
    public void getOnlineConfig(final Yodo1OnlineConfigListener listener) {
        if (!this.initialized) {
            YLog.i(TAG, "已经初始化，return");
            return;
        }
        //后台进入前台二十分钟后重新请求数据
        long duration = System.currentTimeMillis() - time;
        if (duration < 60000 * 20) {
            YLog.i(TAG, "重新请求时间未到,缓存20分钟,duration:" + duration);
            return;
        }
        time = System.currentTimeMillis();

        postRequest(new HttpStringListener(null) {
            @Override
            public void onSuccess(int i, String s, JSONObject jsonObj) {
                YLog.d(TAG + "onSuccess, error_code: " + i + ", error_message: " + s + ", jsonObj:" + jsonObj.toString());

                String dataIdentifer = jsonObj.optString(KEY_DATA_IDENTIFER);
                String locationIdentifer = jsonObj.optString(KEY_LOCATION_IDENTIFER);//地理标识  客户端本地存
                String locationIdentiferTtl = jsonObj.optString(KEY_LOCATION_IDENTIFER_TTL);//地理标识有效时长   小时，客户端本地存

                Yodo1SharedPreferences.put(context, KEY_DATA_IDENTIFER, dataIdentifer);
                //当本地地理标识为空的时候再保存获取到的地理标识
                if (TextUtils.isEmpty(Yodo1SharedPreferences.getString(context, KEY_LOCATION_IDENTIFER))) {
                    Yodo1SharedPreferences.put(context, KEY_LOCATION_IDENTIFER, locationIdentifer);
                }
                Yodo1SharedPreferences.put(context, KEY_LOCATION_IDENTIFER_TTL, locationIdentiferTtl);
                Yodo1SharedPreferences.put(context, KEY_TIMESTAMP_GET_DATA, System.currentTimeMillis() + "");

                JSONObject objData = jsonObj.optJSONObject("data");
                JsonCacheUtils.writeJson(context, objData, KEY_CACHE_FILE_JSON);

                saveTestModeData(jsonObj);
                saveReportFields(jsonObj);

                listener.getDataFinish(i, objData.toString());
            }

            @Override
            public void onFailure(int i, String s) {
                YLog.d(TAG + "onFailure, error_code: " + i + ", error_message: " + s);
                if (i == 10) {
                    String onlineconfig = JsonCacheUtils.readJson(context, KEY_CACHE_FILE_JSON);
                    listener.getDataFinish(i, onlineconfig);
                } else {
                    YLog.d(TAG + "获取在线参数异常, 需获取CDN静态参数...");
                    //需拼接静态参数，获取在线参数静态数据
                    time = 0;
                    getCdnOnlineConfig(listener);
                }
            }
        });
    }

    /**
     * 根据接口获取在线参数静态数据
     */
    private void getCdnOnlineConfig(final Yodo1OnlineConfigListener listener) {
        getCdnOnlineConfig(new Yodo1RequestListener() {
            @Override
            public void onYodo1RequestComplete(Yodo1SDKResponse responseObject) {
                String resMsg = responseObject.getResponseString();
                YLog.d(TAG + "CDN onYodo1RequestComplete, the response string:" + resMsg);
                try {
                    if (resMsg != null) {
                        JSONObject jsonObj = new JSONObject(resMsg.replace(" ", ""));

                        String dataIdentifer = jsonObj.optString(KEY_DATA_IDENTIFER);
                        String localData = Yodo1SharedPreferences.getString(context, KEY_DATA_IDENTIFER);
                        if (TextUtils.isEmpty(localData)) {
                            localData = "0";
                        }
                        if (Long.parseLong(dataIdentifer) > Long.parseLong(localData)) {
                            Yodo1SharedPreferences.put(context, KEY_DATA_IDENTIFER, dataIdentifer);

                            JSONObject objData = jsonObj.optJSONObject("data");
                            JsonCacheUtils.writeJson(context, objData, KEY_CACHE_FILE_JSON);

                            saveTestModeData(jsonObj);
                            saveReportFields(jsonObj);

                            listener.getDataFinish(0, objData.toString());
                        } else {
                            String onlineConfig = JsonCacheUtils.readJson(context, KEY_CACHE_FILE_JSON);
                            listener.getDataFinish(0, onlineConfig);
                        }
                    } else {
                        listener.getDataFinish(-1, "");
                    }
                } catch (Exception e) {
                    YLog.d(TAG + "解析CDN静态参数异常, message: " + e.getMessage() + ", cause: " + e.getCause());
                    listener.getDataFinish(-1, "");
                }
            }
        });
    }

    /**
     * 根据接口获取在线参数
     *
     * @param listener 回调
     */
    private void getCdnOnlineConfig(final Yodo1RequestListener listener) {
        String url = URL_ONLINE_CONFIG_CDN + MD5EncodeUtil.MD5Encode(appKey) + ".json";

        Request<String> request = NoHttp.createStringRequest(url);
        YLog.d(TAG + "Connecting to the CDN configuration, the request url: " + request.url());
        Yodo1HttpManage.getInstance().connect(0, request, new HttpListener<String>() {
            @Override
            public void onSucceed(int what, Response<String> response) {
                YLog.d(TAG + "Connect CDN configuration onSucceed, the response: " + response.get());
                Yodo1SDKResponse resp = Yodo1HttpManage.getInstance().getResponseObject(0, response);
                listener.onYodo1RequestComplete(resp);
            }

            @Override
            public void onFailed(int what, Response<String> response) {
                YLog.d(TAG + "Connect CDN configuration onFailed");
                Yodo1SDKResponse resp = Yodo1HttpManage.getInstance().getResponseObject(0, response);
                listener.onYodo1RequestComplete(resp);
            }
        }, false);
    }

    /**
     * 获取在线参数
     *
     * @param key          在线参数Key
     * @param defaultValue 默认值
     * @return 在线参数值
     */
    public String getConfigParam(String key, String defaultValue) {
        String result = defaultValue;
        try {
            JSONObject obj = getJsonData();
            if (obj != null && !TextUtils.isEmpty(obj.optString(key))) {
                result = obj.optString(key);
            }

        } catch (JSONException e) {
            e.printStackTrace();
        }
        YLog.d(TAG + "getConfigParam " + key + " = " + result);
        return result;
    }

    /**
     * 在线参数初始化
     *
     * @param context Context
     * @param appKey  App key
     */
    public void init(final Context context, final String appKey) {
        if (context == null || TextUtils.isEmpty(appKey)) {
            YLog.i(TAG, "appKey 为空，初始化失败。");
            return;
        }
        if (this.initialized) {
            YLog.i(TAG, "init 已经初始化，return.");
            return;
        }

        this.initialized = true;
        this.appKey = appKey;
        Yodo1OnlineConfig.context = context;
        YLog.v(TAG, "*********** Yodo1 Info ***********");
        YLog.v(TAG, "App Key: " + this.appKey);
        YLog.v(TAG, "AppPackageName: " + SysUtils.getPackageName(context));
        YLog.v(TAG, "AppVersionName: " + SysUtils.getVersionName(context));
        YLog.d(TAG, "Publish Channel: " + Yodo1SdkUtils.getPublishCode(context));
        YLog.v(TAG, "Sdk Type: " + Yodo1SdkUtils.getSdkType(context));
        YLog.v(TAG, "Sdk Version: " + Yodo1SdkUtils.getSdkVersion(context));
        YLog.d(TAG, "Sdk Mode: " + Yodo1SdkUtils.getSdkMode());
        YLog.v(TAG, "*********** Yodo1 Info ***********");
        String localAppKey = Yodo1SharedPreferences.getString(context, KEY_GAME_APP_KEY);
        String localAppPackageName = Yodo1SharedPreferences.getString(context, KEY_GAME_PACKAGE_NAME);
        String localAppVersionName = Yodo1SharedPreferences.getString(context, KEY_GAME_VERSION_NAME);
        if (!TextUtils.isEmpty(localAppKey) && !localAppKey.equals(this.appKey)) {
            Yodo1SharedPreferences.put(context, KEY_DATA_IDENTIFER, "0");
        }
        if (!TextUtils.isEmpty(localAppPackageName) && !localAppPackageName.equals(SysUtils.getPackageName(context))) {
            Yodo1SharedPreferences.put(context, KEY_DATA_IDENTIFER, "0");
        }
        if (!TextUtils.isEmpty(localAppVersionName) && !localAppVersionName.equals(SysUtils.getVersionName(context))) {
            Yodo1SharedPreferences.put(context, KEY_DATA_IDENTIFER, "0");
        }
        //升级和包覆盖，重新拉配置。
        Yodo1SharedPreferences.put(context, KEY_GAME_APP_KEY, appKey);
        Yodo1SharedPreferences.put(context, KEY_GAME_PACKAGE_NAME, SysUtils.getPackageName(context));
        Yodo1SharedPreferences.put(context, KEY_GAME_VERSION_NAME, SysUtils.getVersionName(context));
        Yodo1SharedPreferences.put(context, KEY_GAME_CHANNEL_CODE, Yodo1SdkUtils.getPublishCode(context));
    }

    /**
     * 在线参数初始化
     *
     * @param context Context
     */
    public void initConfig(final Context context) {
        Yodo1OnlineConfig.context = context;
        getLocation();//读取当前位置
    }

    /**
     * 获取当前的位置信息
     */
    @SuppressLint({"MissingPermission", "DefaultLocale"})
    private void getLocation() {
        try {
            String[] permissions1 = new String[]{Manifest.permission.ACCESS_COARSE_LOCATION};
            String[] permissions2 = new String[]{Manifest.permission.ACCESS_FINE_LOCATION};

            boolean permissionCheck = Yodo1PermissonUtils.hasPermissionForManifest(context, permissions1);
            boolean permissionCheck2 = Yodo1PermissonUtils.hasPermissionForManifest(context, permissions2);

            if (permissionCheck || permissionCheck2) {
                boolean permissionCheck3 = Yodo1PermissonUtils.hasPermission(context, permissions1);
                boolean permissionCheck4 = Yodo1PermissonUtils.hasPermission(context, permissions2);

                if (permissionCheck3 || permissionCheck4) {
                    LocationManager locationManager = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);

                    Location location = locationManager.getLastKnownLocation(LocationManager.GPS_PROVIDER);
                    if (location != null) {
                        YLog.d(TAG + "getLocation 当前GPS定位经纬度为: getLatitude = " + location.getLatitude() + ", getLongitude = " + location.getLongitude());
                        this.locationLongitude = String.format("%.2f", location.getLongitude());
                        this.locationLatitude = String.format("%.2f", location.getLatitude());
                    } else {
                        location = locationManager.getLastKnownLocation(LocationManager.NETWORK_PROVIDER);

                        if (location != null) {
                            YLog.d(TAG + "getLocation 当前网络定位经纬度为: getLatitude = " + location.getLatitude() + ", getLongitude = " + location.getLongitude());

                            locationLongitude = String.format("%.2f", location.getLongitude());
                            locationLatitude = String.format("%.2f", location.getLatitude());
                        }
                    }
                } else {
                    YLog.d(TAG + "getLocation 当前应用未授权获取到定位权限");
                }

            } else {
                YLog.d(TAG + "getLocation Manifest中缺少 ACCESS_COARSE_LOCATION 定位权限");
            }
        } catch (Exception e) {
            YLog.d(TAG + "getLocation 当前定位异常");
        }
    }
}
