/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.zookeeper;

import com.yahoo.cloud.config.ZookeeperServerConfig;
import com.yahoo.concurrent.DaemonThreadFactory;
import com.yahoo.protect.Process;
import com.yahoo.time.ExponentialBackoff;
import com.yahoo.vespa.defaults.Defaults;
import com.yahoo.vespa.zookeeper.Configurator;
import com.yahoo.vespa.zookeeper.Sleeper;
import com.yahoo.vespa.zookeeper.server.VespaZooKeeperServer;
import com.yahoo.yolean.Exceptions;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class ZooKeeperRunner
implements Runnable {
    private static final Logger log = Logger.getLogger(ZooKeeperRunner.class.getName());
    private static final Duration STOP_TIMEOUT = Duration.ofSeconds(10L);
    private static final Duration START_TIMEOUT = Duration.ofMinutes(10L);
    private final ExecutorService executorService;
    private final ZookeeperServerConfig zookeeperServerConfig;
    private final VespaZooKeeperServer server;
    private final ExponentialBackoff backoff = new ExponentialBackoff(Duration.ofSeconds(5L), Duration.ofSeconds(15L));
    private final Sleeper sleeper = new Sleeper();

    public ZooKeeperRunner(ZookeeperServerConfig zookeeperServerConfig, VespaZooKeeperServer server) {
        this.zookeeperServerConfig = zookeeperServerConfig;
        this.server = server;
        new Configurator(zookeeperServerConfig).writeConfigToDisk();
        this.executorService = Executors.newSingleThreadExecutor((ThreadFactory)new DaemonThreadFactory("zookeeper-server-"));
        this.executorService.submit(this);
    }

    void shutdown() {
        log.log(Level.INFO, "Triggering shutdown");
        this.server.shutdown();
        this.executorService.shutdownNow();
        try {
            if (!this.executorService.awaitTermination(STOP_TIMEOUT.toMillis(), TimeUnit.MILLISECONDS)) {
                log.log(Level.WARNING, "Failed to shut down within " + STOP_TIMEOUT);
            }
        }
        catch (InterruptedException e) {
            log.log(Level.WARNING, "Interrupted waiting for executor to complete", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        Path path = Paths.get(Defaults.getDefaults().underVespaHome(this.zookeeperServerConfig.zooKeeperConfigFile()), new String[0]);
        Instant now = Instant.now();
        Instant end = now.plus(START_TIMEOUT);
        int attempt = 1;
        while (now.isBefore(end) && !this.executorService.isShutdown()) {
            try {
                this.startServer(path, attempt);
                return;
            }
            catch (RuntimeException e) {
                String messagePart = "Starting " + this.serverDescription() + " failed on attempt " + attempt;
                if (this.server.reconfigurable()) {
                    Duration delay = this.backoff.delay(attempt);
                    log.log(Level.WARNING, messagePart + ". Retrying in " + delay + ", time left " + Duration.between(now, end), e);
                    this.sleeper.sleep(delay);
                } else {
                    Process.logAndDie((String)(messagePart + ". Forcing shutdown"), (Throwable)e);
                }
            }
            finally {
                now = Instant.now();
            }
            ++attempt;
        }
        log.log(Level.INFO, "Current content of zookeeper config file at '" + path + "':\n" + (String)Exceptions.uncheck(() -> Files.readString(path)));
    }

    private String serverDescription() {
        return (this.server.reconfigurable() ? "" : "non-") + "reconfigurable ZooKeeper server";
    }

    private void startServer(Path path, int attempt) {
        if (attempt > 1) {
            log.log(Level.INFO, "Starting ZooKeeper server with " + path.toFile().getAbsolutePath() + ". Trying to establish ZooKeeper quorum (members: " + Configurator.zookeeperServerHostnames(this.zookeeperServerConfig) + ", attempt " + attempt + ")");
        }
        ClassLoader tccl = Thread.currentThread().getContextClassLoader();
        Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());
        try {
            this.server.start(path);
        }
        catch (Throwable e) {
            throw new RuntimeException("Starting " + this.serverDescription() + " failed", e);
        }
        finally {
            Thread.currentThread().setContextClassLoader(tccl);
        }
    }
}

