/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.compress;

import com.yahoo.path.Path;
import com.yahoo.yolean.Exceptions;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.util.Objects;
import java.util.OptionalLong;
import java.util.function.Predicate;
import java.util.zip.GZIPInputStream;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveInputStream;

public class ArchiveStreamReader
implements AutoCloseable {
    private final ArchiveInputStream archiveInputStream;
    private final Options options;
    private long totalRead = 0L;
    private long entriesRead = 0L;

    private ArchiveStreamReader(ArchiveInputStream archiveInputStream, Options options) {
        this.archiveInputStream = Objects.requireNonNull(archiveInputStream);
        this.options = Objects.requireNonNull(options);
    }

    public static ArchiveStreamReader ofTarGzip(InputStream inputStream, Options options) {
        return new ArchiveStreamReader((ArchiveInputStream)new TarArchiveInputStream((InputStream)Exceptions.uncheck(() -> new GZIPInputStream(inputStream))), options);
    }

    public static ArchiveStreamReader ofZip(InputStream inputStream, Options options) {
        return new ArchiveStreamReader((ArchiveInputStream)new ZipArchiveInputStream(inputStream), options);
    }

    public ArchiveFile readNextTo(OutputStream outputStream) {
        try {
            ArchiveEntry entry;
            while ((entry = this.archiveInputStream.getNextEntry()) != null) {
                int read;
                Path path = Path.fromString(ArchiveStreamReader.requireNormalized(entry.getName(), this.options.allowDotSegment));
                if (ArchiveStreamReader.isSymlink(entry)) {
                    throw new IllegalArgumentException("Archive entry " + path + " is a symbolic link, which is unsupported");
                }
                if (entry.isDirectory() || !this.options.pathPredicate.test(path.toString())) continue;
                if (++this.entriesRead > this.options.maxEntries) {
                    throw new IllegalArgumentException("Attempted to read more entries than entry limit of " + this.options.maxEntries);
                }
                long size = 0L;
                byte[] buffer = new byte[2048];
                while ((read = this.archiveInputStream.read(buffer)) != -1) {
                    this.totalRead += (long)read;
                    size += (long)read;
                    if (this.totalRead > this.options.maxSize) {
                        throw new IllegalArgumentException("Total size of archive exceeds size limit of " + this.options.maxSize + " bytes");
                    }
                    if ((long)read > this.options.maxEntrySize) {
                        if (this.options.truncateEntry) continue;
                        throw new IllegalArgumentException("Size of entry " + path + " exceeded entry size limit of " + this.options.maxEntrySize + " bytes");
                    }
                    outputStream.write(buffer, 0, read);
                }
                return new ArchiveFile(path, ArchiveStreamReader.crc32(entry), size);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return null;
    }

    @Override
    public void close() {
        Exceptions.uncheck(() -> this.archiveInputStream.close());
    }

    private static OptionalLong crc32(ArchiveEntry entry) {
        long crc32 = -1L;
        if (entry instanceof ZipArchiveEntry) {
            crc32 = ((ZipArchiveEntry)entry).getCrc();
        }
        return crc32 > -1L ? OptionalLong.of(crc32) : OptionalLong.empty();
    }

    private static boolean isSymlink(ArchiveEntry entry) {
        if (entry instanceof ZipArchiveEntry) {
            ZipArchiveEntry zipEntry = (ZipArchiveEntry)entry;
            return zipEntry.isUnixSymlink();
        }
        if (entry instanceof TarArchiveEntry) {
            TarArchiveEntry tarEntry = (TarArchiveEntry)entry;
            return tarEntry.isSymbolicLink();
        }
        throw new IllegalArgumentException("Unsupported archive entry " + entry.getClass().getSimpleName() + ", cannot check for symbolic link");
    }

    private static String requireNormalized(String name, boolean allowDotSegment) {
        for (String part : name.split("/")) {
            if (!part.isEmpty() && (allowDotSegment || !part.equals(".")) && !part.equals("..")) continue;
            throw new IllegalArgumentException("Unexpected non-normalized path found in zip content: '" + name + "'");
        }
        return name;
    }

    private static long requireNonNegative(String field, long n) {
        if (n < 0L) {
            throw new IllegalArgumentException(field + " cannot be negative, got " + n);
        }
        return n;
    }

    public static class Options {
        private long maxSize = 8L * (long)Math.pow(1024.0, 3.0);
        private long maxEntrySize = Long.MAX_VALUE;
        private long maxEntries = Long.MAX_VALUE;
        private boolean truncateEntry = false;
        private boolean allowDotSegment = false;
        private Predicate<String> pathPredicate = path -> true;

        private Options() {
        }

        public static Options standard() {
            return new Options();
        }

        public Options maxSize(long size) {
            this.maxSize = ArchiveStreamReader.requireNonNegative("size", size);
            return this;
        }

        public Options maxEntrySize(long size) {
            this.maxEntrySize = ArchiveStreamReader.requireNonNegative("size", size);
            return this;
        }

        public Options maxEntries(long count) {
            this.maxEntries = ArchiveStreamReader.requireNonNegative("count", count);
            return this;
        }

        public Options truncateEntry(boolean truncate) {
            this.truncateEntry = truncate;
            return this;
        }

        public Options pathPredicate(Predicate<String> predicate) {
            this.pathPredicate = predicate;
            return this;
        }

        public Options allowDotSegment(boolean allow) {
            this.allowDotSegment = allow;
            return this;
        }
    }

    public static class ArchiveFile {
        private final Path path;
        private final OptionalLong crc32;
        private final long size;

        public ArchiveFile(Path name, OptionalLong crc32, long size) {
            this.path = Objects.requireNonNull(name);
            this.crc32 = Objects.requireNonNull(crc32);
            if (crc32.isPresent()) {
                ArchiveStreamReader.requireNonNegative("crc32", crc32.getAsLong());
            }
            this.size = ArchiveStreamReader.requireNonNegative("size", size);
        }

        public Path path() {
            return this.path;
        }

        public OptionalLong crc32() {
            return this.crc32;
        }

        public long size() {
            return this.size;
        }
    }
}

