/*
 * Decompiled with CFR 0.152.
 */
package ai.vespa.metricsproxy.service;

import ai.vespa.metricsproxy.metric.Metric;
import ai.vespa.metricsproxy.metric.Metrics;
import ai.vespa.metricsproxy.metric.model.MetricId;
import ai.vespa.metricsproxy.service.CpuJiffies;
import ai.vespa.metricsproxy.service.VespaService;
import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

public class SystemPoller {
    private static final Logger log = Logger.getLogger(SystemPoller.class.getName());
    private static final int memoryTypeVirtual = 0;
    private static final int memoryTypeResident = 1;
    private static final MetricId CPU = MetricId.toMetricId("cpu");
    private static final MetricId CPU_UTIL = MetricId.toMetricId("cpu_util");
    private static final MetricId MEMORY_VIRT = MetricId.toMetricId("memory_virt");
    private static final MetricId MEMORY_RSS = MetricId.toMetricId("memory_rss");
    private final Duration interval;
    private final List<VespaService> services;
    private final Map<VespaService, Long> lastCpuJiffiesMetrics = new ConcurrentHashMap<VespaService, Long>();
    private final Timer systemPollTimer;
    private final GetJiffies jiffiesInterface;
    private JiffiesAndCpus lastTotalCpuJiffies;

    public SystemPoller(List<VespaService> services, Duration interval) {
        this.services = services;
        this.interval = interval;
        this.systemPollTimer = new Timer("systemPollTimer", true);
        this.jiffiesInterface = new GetJiffies(){

            @Override
            public JiffiesAndCpus getTotalSystemJiffies() {
                return SystemPoller.getTotalSystemJiffies();
            }

            @Override
            public long getJiffies(VespaService service) {
                return SystemPoller.getPidJiffies(service);
            }
        };
        this.lastTotalCpuJiffies = this.jiffiesInterface.getTotalSystemJiffies();
        for (VespaService s : services) {
            this.lastCpuJiffiesMetrics.put(s, this.jiffiesInterface.getJiffies(s));
        }
    }

    void stop() {
        this.systemPollTimer.cancel();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static long[] getMemoryUsage(VespaService service) {
        BufferedReader br;
        int pid = service.getPid();
        try {
            br = new BufferedReader(new FileReader("/proc/" + pid + "/smaps"));
        }
        catch (FileNotFoundException ex) {
            service.setAlive(false);
            return new long[2];
        }
        try {
            long[] ex = SystemPoller.getMemoryUsage(br);
            return ex;
        }
        catch (IOException ex) {
            log.log(Level.FINE, "Unable to read line from smaps file", ex);
            long[] lArray = new long[2];
            return lArray;
        }
        finally {
            try {
                br.close();
            }
            catch (IOException ex) {
                log.log(Level.FINE, "Closing of smaps file failed", ex);
            }
        }
    }

    static long[] getMemoryUsage(BufferedReader br) throws IOException {
        String line;
        long[] size = new long[2];
        while ((line = br.readLine()) != null) {
            String remain;
            if (line.startsWith("Rss:")) {
                remain = line.substring(4).trim();
                size[1] = size[1] + Long.parseLong(remain.substring(0, remain.indexOf(32))) * 1024L;
                continue;
            }
            if (!line.startsWith("Size:")) continue;
            remain = line.substring(5).trim();
            size[0] = size[0] + Long.parseLong(remain.substring(0, remain.indexOf(32))) * 1024L;
        }
        return size;
    }

    void poll() {
        Instant startTime = Instant.now();
        if (this.services.isEmpty()) {
            this.schedule();
            return;
        }
        log.log(Level.FINE, () -> "Monitoring system metrics for " + this.services.size() + " services");
        boolean someAlive = this.services.stream().anyMatch(VespaService::isAlive);
        this.lastTotalCpuJiffies = SystemPoller.updateMetrics(this.lastTotalCpuJiffies, startTime, this.jiffiesInterface, this.services, this.lastCpuJiffiesMetrics);
        if (!someAlive) {
            this.reschedule(Duration.between(startTime, Instant.now()));
        } else {
            this.schedule();
        }
    }

    static JiffiesAndCpus updateMetrics(JiffiesAndCpus prevTotalJiffies, Instant timeStamp, GetJiffies getJiffies, List<VespaService> services, Map<VespaService, Long> lastCpuJiffiesMetrics) {
        HashMap<VespaService, Long> currentServiceJiffies = new HashMap<VespaService, Long>();
        for (VespaService s : services) {
            currentServiceJiffies.put(s, getJiffies.getJiffies(s));
        }
        JiffiesAndCpus sysJiffies = getJiffies.getTotalSystemJiffies();
        JiffiesAndCpus sysJiffiesDiff = sysJiffies.diff(prevTotalJiffies);
        log.log(Level.FINE, () -> "Total jiffies: " + sysJiffies.jiffies + " - " + prevTotalJiffies.jiffies + " = " + sysJiffiesDiff.jiffies);
        for (VespaService s : services) {
            Metrics metrics = new Metrics();
            long[] size = SystemPoller.getMemoryUsage(s);
            log.log(Level.FINE, () -> "Updating memory metric for service " + s);
            metrics.add(new Metric(MEMORY_VIRT, (Number)size[0], timeStamp));
            metrics.add(new Metric(MEMORY_RSS, (Number)size[1], timeStamp));
            long procJiffies = (Long)currentServiceJiffies.get(s);
            long last = lastCpuJiffiesMetrics.get(s);
            long diff = procJiffies - last;
            log.log(Level.FINE, () -> "Service " + s + " jiffies: " + procJiffies + " - " + last + " = " + diff);
            if (diff >= 0L) {
                metrics.add(new Metric(CPU, (Number)(100.0 * sysJiffiesDiff.ratioSingleCoreJiffies(diff)), timeStamp));
                metrics.add(new Metric(CPU_UTIL, (Number)(100.0 * sysJiffiesDiff.ratioJiffies(diff)), timeStamp));
            }
            lastCpuJiffiesMetrics.put(s, procJiffies);
            s.setSystemMetrics(metrics);
            log.log(Level.FINE, () -> "Current size of system metrics for service  " + s + " is " + metrics.size());
        }
        return sysJiffies;
    }

    static long getPidJiffies(VespaService service) {
        int pid = service.getPid();
        try {
            BufferedReader in = new BufferedReader(new FileReader("/proc/" + pid + "/stat"));
            return SystemPoller.getPidJiffies(in);
        }
        catch (FileNotFoundException ex) {
            log.log(Level.FINE, () -> "Unable to find pid " + pid + " in proc directory, for service " + service.getInstanceName());
            service.setAlive(false);
            return 0L;
        }
    }

    static long getPidJiffies(BufferedReader in) {
        String line;
        try {
            line = in.readLine();
            in.close();
        }
        catch (IOException ex) {
            log.log(Level.FINE, "Unable to read line from process stat file", ex);
            return 0L;
        }
        String[] elems = line.split(" ");
        return Long.parseLong(elems[13]) + Long.parseLong(elems[14]);
    }

    private static JiffiesAndCpus getTotalSystemJiffies() {
        try {
            BufferedReader in = new BufferedReader(new FileReader("/proc/stat"));
            return SystemPoller.getTotalSystemJiffies(in);
        }
        catch (FileNotFoundException ex) {
            log.log(Level.SEVERE, "Unable to open stat file", ex);
            return new JiffiesAndCpus();
        }
    }

    static JiffiesAndCpus getTotalSystemJiffies(BufferedReader in) {
        ArrayList<CpuJiffies> jiffies = new ArrayList<CpuJiffies>();
        CpuJiffies total = null;
        try {
            String line;
            while ((line = in.readLine()) != null) {
                if (line.startsWith("cpu ")) {
                    total = new CpuJiffies(line);
                    continue;
                }
                if (!line.startsWith("cpu")) continue;
                jiffies.add(new CpuJiffies(line));
            }
            in.close();
        }
        catch (IOException ex) {
            log.log(Level.SEVERE, "Unable to read line from stat file", ex);
            return new JiffiesAndCpus();
        }
        return total != null ? new JiffiesAndCpus(total.getTotalJiffies(), jiffies.size()) : new JiffiesAndCpus();
    }

    void schedule(Duration time) {
        try {
            this.systemPollTimer.schedule((TimerTask)new PollTask(this), time.toMillis());
        }
        catch (IllegalStateException e) {
            log.info("Tried to schedule task, but timer was already shut down.");
        }
    }

    void schedule() {
        this.schedule(this.interval);
    }

    private void reschedule(Duration skew) {
        Duration sleep = this.interval.minus(skew);
        if (sleep.compareTo(Duration.ofMinutes(1L)) < 0) {
            this.schedule(Duration.ofMinutes(1L));
        } else {
            this.schedule(sleep);
        }
    }

    private static class PollTask
    extends TimerTask {
        private final SystemPoller poller;

        PollTask(SystemPoller poller) {
            this.poller = poller;
        }

        @Override
        public void run() {
            this.poller.poll();
        }
    }

    static interface GetJiffies {
        public JiffiesAndCpus getTotalSystemJiffies();

        public long getJiffies(VespaService var1);
    }

    static class JiffiesAndCpus {
        final long jiffies;
        final int cpus;

        JiffiesAndCpus() {
            this(0L, 1);
        }

        JiffiesAndCpus(long jiffies, int cpus) {
            this.jiffies = jiffies;
            this.cpus = Math.max(1, cpus);
        }

        double ratioSingleCoreJiffies(long partJiffies) {
            return (double)(partJiffies * (long)this.cpus) / Math.max(1.0, (double)this.jiffies);
        }

        double ratioJiffies(long partJiffies) {
            return (double)partJiffies / Math.max(1.0, (double)this.jiffies);
        }

        JiffiesAndCpus diff(JiffiesAndCpus prev) {
            return this.cpus == prev.cpus ? new JiffiesAndCpus(this.jiffies - prev.jiffies, this.cpus) : new JiffiesAndCpus();
        }
    }
}

