/*
 * Decompiled with CFR 0.152.
 */
package ai.vespa.metricsproxy.service;

import ai.vespa.metricsproxy.metric.Metric;
import ai.vespa.metricsproxy.metric.Metrics;
import ai.vespa.metricsproxy.metric.model.MetricId;
import ai.vespa.metricsproxy.service.CpuJiffies;
import ai.vespa.metricsproxy.service.VespaService;
import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

public class SystemPoller {
    private static final Logger log = Logger.getLogger(SystemPoller.class.getName());
    private static final int memoryTypeVirtual = 0;
    private static final int memoryTypeResident = 1;
    private static final MetricId CPU = MetricId.toMetricId("cpu");
    private static final MetricId CPU_UTIL = MetricId.toMetricId("cpu_util");
    private static final MetricId MEMORY_VIRT = MetricId.toMetricId("memory_virt");
    private static final MetricId MEMORY_RSS = MetricId.toMetricId("memory_rss");
    private final int pollingIntervalSecs;
    private final List<VespaService> services;
    private final Map<VespaService, Long> lastCpuJiffiesMetrics = new ConcurrentHashMap<VespaService, Long>();
    private final Timer systemPollTimer;
    private final GetJiffies jiffiesInterface;
    private JiffiesAndCpus lastTotalCpuJiffies;

    public SystemPoller(List<VespaService> services, int pollingIntervalSecs) {
        this.services = services;
        this.pollingIntervalSecs = pollingIntervalSecs;
        this.systemPollTimer = new Timer("systemPollTimer", true);
        this.jiffiesInterface = new GetJiffies(){

            @Override
            public JiffiesAndCpus getTotalSystemJiffies() {
                return SystemPoller.getTotalSystemJiffies();
            }

            @Override
            public long getJiffies(VespaService service) {
                return SystemPoller.getPidJiffies(service);
            }
        };
        this.lastTotalCpuJiffies = this.jiffiesInterface.getTotalSystemJiffies();
        for (VespaService s : services) {
            this.lastCpuJiffiesMetrics.put(s, this.jiffiesInterface.getJiffies(s));
        }
    }

    void stop() {
        this.systemPollTimer.cancel();
    }

    static long[] getMemoryUsage(VespaService service) {
        BufferedReader br;
        long[] size = new long[2];
        int pid = service.getPid();
        try {
            br = new BufferedReader(new FileReader("/proc/" + pid + "/smaps"));
        }
        catch (FileNotFoundException ex) {
            service.setAlive(false);
            return size;
        }
        try {
            String line;
            while ((line = br.readLine()) != null) {
                String[] elems = line.split("\\s+");
                if (line.startsWith("Rss:")) {
                    size[1] = size[1] + Long.parseLong(elems[1]) * 1024L;
                    continue;
                }
                if (!line.startsWith("Size:")) continue;
                size[0] = size[0] + Long.parseLong(elems[1]) * 1024L;
            }
            br.close();
        }
        catch (IOException ex) {
            log.log(Level.FINE, "Unable to read line from smaps file", ex);
            return size;
        }
        return size;
    }

    void poll() {
        long startTime = System.currentTimeMillis();
        if (this.services.isEmpty()) {
            this.schedule();
            return;
        }
        log.log(Level.FINE, () -> "Monitoring system metrics for " + this.services.size() + " services");
        boolean someAlive = this.services.stream().anyMatch(VespaService::isAlive);
        this.lastTotalCpuJiffies = SystemPoller.updateMetrics(this.lastTotalCpuJiffies, startTime / 1000L, this.jiffiesInterface, this.services, this.lastCpuJiffiesMetrics);
        if (!someAlive) {
            this.reschedule(System.currentTimeMillis() - startTime);
        } else {
            this.schedule();
        }
    }

    static JiffiesAndCpus updateMetrics(JiffiesAndCpus prevTotalJiffies, long timeStamp, GetJiffies getJiffies, List<VespaService> services, Map<VespaService, Long> lastCpuJiffiesMetrics) {
        JiffiesAndCpus sysJiffies = getJiffies.getTotalSystemJiffies();
        JiffiesAndCpus sysJiffiesDiff = sysJiffies.diff(prevTotalJiffies);
        for (VespaService s : services) {
            Metrics metrics = new Metrics();
            log.log(Level.FINE, () -> "Current size of system metrics for service  " + s + " is " + metrics.size());
            long[] size = SystemPoller.getMemoryUsage(s);
            log.log(Level.FINE, () -> "Updating memory metric for service " + s);
            metrics.add(new Metric(MEMORY_VIRT, size[0], timeStamp));
            metrics.add(new Metric(MEMORY_RSS, size[1], timeStamp));
            long procJiffies = getJiffies.getJiffies(s);
            long last = lastCpuJiffiesMetrics.get(s);
            long diff = procJiffies - last;
            if (diff >= 0L) {
                metrics.add(new Metric(CPU, 100.0 * sysJiffiesDiff.ratioSingleCoreJiffies(diff), timeStamp));
                metrics.add(new Metric(CPU_UTIL, 100.0 * sysJiffiesDiff.ratioJiffies(diff), timeStamp));
            }
            lastCpuJiffiesMetrics.put(s, procJiffies);
            s.setSystemMetrics(metrics);
        }
        return sysJiffies;
    }

    static long getPidJiffies(VespaService service) {
        int pid = service.getPid();
        try {
            BufferedReader in = new BufferedReader(new FileReader("/proc/" + pid + "/stat"));
            return SystemPoller.getPidJiffies(in);
        }
        catch (FileNotFoundException ex) {
            log.log(Level.FINE, () -> "Unable to find pid " + pid + " in proc directory, for service " + service.getInstanceName());
            service.setAlive(false);
            return 0L;
        }
    }

    static long getPidJiffies(BufferedReader in) {
        String line;
        try {
            line = in.readLine();
            in.close();
        }
        catch (IOException ex) {
            log.log(Level.FINE, "Unable to read line from process stat file", ex);
            return 0L;
        }
        String[] elems = line.split(" ");
        return Long.parseLong(elems[13]) + Long.parseLong(elems[14]);
    }

    private static JiffiesAndCpus getTotalSystemJiffies() {
        try {
            BufferedReader in = new BufferedReader(new FileReader("/proc/stat"));
            return SystemPoller.getTotalSystemJiffies(in);
        }
        catch (FileNotFoundException ex) {
            log.log(Level.SEVERE, "Unable to open stat file", ex);
            return new JiffiesAndCpus();
        }
    }

    static JiffiesAndCpus getTotalSystemJiffies(BufferedReader in) {
        ArrayList<CpuJiffies> jiffies = new ArrayList<CpuJiffies>();
        CpuJiffies total = null;
        try {
            String line;
            while ((line = in.readLine()) != null) {
                if (line.startsWith("cpu ")) {
                    total = new CpuJiffies(line);
                    continue;
                }
                if (!line.startsWith("cpu")) continue;
                jiffies.add(new CpuJiffies(line));
            }
            in.close();
        }
        catch (IOException ex) {
            log.log(Level.SEVERE, "Unable to read line from stat file", ex);
            return new JiffiesAndCpus();
        }
        return total != null ? new JiffiesAndCpus(total.getTotalJiffies(), jiffies.size()) : new JiffiesAndCpus();
    }

    private void schedule(long time) {
        try {
            this.systemPollTimer.schedule((TimerTask)new PollTask(this), time);
        }
        catch (IllegalStateException e) {
            log.info("Tried to schedule task, but timer was already shut down.");
        }
    }

    public void schedule() {
        this.schedule((long)this.pollingIntervalSecs * 1000L);
    }

    private void reschedule(long skew) {
        long sleep = (long)this.pollingIntervalSecs * 1000L - skew;
        sleep = Math.max(60000L, sleep);
        this.schedule(sleep);
    }

    private static class PollTask
    extends TimerTask {
        private final SystemPoller poller;

        PollTask(SystemPoller poller) {
            this.poller = poller;
        }

        @Override
        public void run() {
            this.poller.poll();
        }
    }

    static interface GetJiffies {
        public JiffiesAndCpus getTotalSystemJiffies();

        public long getJiffies(VespaService var1);
    }

    static class JiffiesAndCpus {
        final long jiffies;
        final int cpus;

        JiffiesAndCpus() {
            this(0L, 1);
        }

        JiffiesAndCpus(long jiffies, int cpus) {
            this.jiffies = jiffies;
            this.cpus = Math.max(1, cpus);
        }

        double ratioSingleCoreJiffies(long partJiffies) {
            return (double)(partJiffies * (long)this.cpus) / Math.max(1.0, (double)this.jiffies);
        }

        double ratioJiffies(long partJiffies) {
            return (double)partJiffies / Math.max(1.0, (double)this.jiffies);
        }

        JiffiesAndCpus diff(JiffiesAndCpus prev) {
            return this.cpus == prev.cpus ? new JiffiesAndCpus(this.jiffies - prev.jiffies, this.cpus) : new JiffiesAndCpus();
        }
    }
}

