/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.jrt;

import com.yahoo.jrt.Buffer;
import com.yahoo.jrt.CryptoSocket;
import com.yahoo.jrt.TransportMetrics;
import com.yahoo.security.tls.authz.AuthorizationResult;
import com.yahoo.security.tls.authz.PeerAuthorizerTrustManager;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SocketChannel;
import java.util.logging.Logger;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLHandshakeException;
import javax.net.ssl.SSLSession;

public class TlsCryptoSocket
implements CryptoSocket {
    private static final ByteBuffer NULL_BUFFER = ByteBuffer.allocate(0);
    private static final Logger log = Logger.getLogger(TlsCryptoSocket.class.getName());
    private final TransportMetrics metrics;
    private final SocketChannel channel;
    private final SSLEngine sslEngine;
    private final Buffer wrapBuffer;
    private final Buffer unwrapBuffer;
    private int sessionPacketBufferSize;
    private int sessionApplicationBufferSize;
    private ByteBuffer handshakeDummyBuffer;
    private HandshakeState handshakeState;
    private AuthorizationResult authorizationResult;

    public TlsCryptoSocket(TransportMetrics metrics, SocketChannel channel, SSLEngine sslEngine) {
        this.metrics = metrics;
        this.channel = channel;
        this.sslEngine = sslEngine;
        SSLSession nullSession = sslEngine.getSession();
        this.wrapBuffer = new Buffer(nullSession.getPacketBufferSize() * 2);
        this.unwrapBuffer = new Buffer(nullSession.getPacketBufferSize() * 2);
        this.handshakeDummyBuffer = ByteBuffer.allocate(nullSession.getApplicationBufferSize());
        this.handshakeState = HandshakeState.NOT_STARTED;
        log.fine(() -> "Initialized with " + sslEngine.toString());
    }

    public void injectReadData(Buffer data) {
        this.unwrapBuffer.getWritable(data.bytes()).put(data.getReadable());
    }

    @Override
    public SocketChannel channel() {
        return this.channel;
    }

    @Override
    public CryptoSocket.HandshakeResult handshake() throws IOException {
        HandshakeState newHandshakeState = this.processHandshakeState(this.handshakeState);
        log.fine(() -> String.format("Handshake state '%s -> %s'", new Object[]{this.handshakeState, newHandshakeState}));
        this.handshakeState = newHandshakeState;
        return TlsCryptoSocket.toHandshakeResult(newHandshakeState);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private HandshakeState processHandshakeState(HandshakeState state) throws IOException {
        try {
            switch (state) {
                case NOT_STARTED: {
                    log.fine(() -> "Initiating handshake");
                    this.sslEngine.beginHandshake();
                    break;
                }
                case NEED_WRITE: {
                    this.channelWrite();
                    break;
                }
                case NEED_READ: {
                    this.channelRead();
                    break;
                }
                case COMPLETED: {
                    return HandshakeState.COMPLETED;
                }
                default: {
                    throw TlsCryptoSocket.unhandledStateException(state);
                }
            }
            block14: while (true) {
                log.fine(() -> "SSLEngine.getHandshakeStatus(): " + (Object)((Object)this.sslEngine.getHandshakeStatus()));
                switch (this.sslEngine.getHandshakeStatus()) {
                    case NOT_HANDSHAKING: {
                        if (this.wrapBuffer.bytes() > 0) {
                            return HandshakeState.NEED_WRITE;
                        }
                        this.sslEngine.setEnableSessionCreation(false);
                        this.handshakeDummyBuffer = null;
                        SSLSession session = this.sslEngine.getSession();
                        this.sessionApplicationBufferSize = session.getApplicationBufferSize();
                        this.sessionPacketBufferSize = session.getPacketBufferSize();
                        log.fine(() -> String.format("Handshake complete: protocol=%s, cipherSuite=%s", session.getProtocol(), session.getCipherSuite()));
                        if (this.sslEngine.getUseClientMode()) {
                            this.metrics.incrementClientTlsConnectionsEstablished();
                            return HandshakeState.COMPLETED;
                        } else {
                            this.metrics.incrementServerTlsConnectionsEstablished();
                        }
                        return HandshakeState.COMPLETED;
                    }
                    case NEED_TASK: {
                        this.sslEngine.getDelegatedTask().run();
                        if (this.authorizationResult == null) continue block14;
                        PeerAuthorizerTrustManager.getAuthorizationResult((SSLEngine)this.sslEngine).ifPresent(result -> {
                            if (!result.succeeded()) {
                                this.metrics.incrementPeerAuthorizationFailures();
                            }
                            this.authorizationResult = result;
                        });
                        continue block14;
                    }
                    case NEED_UNWRAP: {
                        if (this.wrapBuffer.bytes() > 0) {
                            return HandshakeState.NEED_WRITE;
                        }
                        if (this.handshakeUnwrap()) continue block14;
                        return HandshakeState.NEED_READ;
                    }
                    case NEED_WRAP: {
                        if (!this.handshakeWrap()) return HandshakeState.NEED_WRITE;
                        continue block14;
                    }
                }
                break;
            }
            throw new IllegalStateException("Unexpected handshake status: " + (Object)((Object)this.sslEngine.getHandshakeStatus()));
        }
        catch (SSLHandshakeException e) {
            if (this.authorizationResult != null && !this.authorizationResult.succeeded()) throw e;
            this.metrics.incrementTlsCertificateVerificationFailures();
            throw e;
        }
    }

    private static CryptoSocket.HandshakeResult toHandshakeResult(HandshakeState state) {
        switch (state) {
            case NEED_READ: {
                return CryptoSocket.HandshakeResult.NEED_READ;
            }
            case NEED_WRITE: {
                return CryptoSocket.HandshakeResult.NEED_WRITE;
            }
            case COMPLETED: {
                return CryptoSocket.HandshakeResult.DONE;
            }
        }
        throw TlsCryptoSocket.unhandledStateException(state);
    }

    @Override
    public int getMinimumReadBufferSize() {
        return this.sessionApplicationBufferSize;
    }

    @Override
    public int read(ByteBuffer dst) throws IOException {
        this.verifyHandshakeCompleted();
        int bytesUnwrapped = this.drain(dst);
        if (bytesUnwrapped > 0) {
            return bytesUnwrapped;
        }
        int bytesRead = this.channelRead();
        if (bytesRead == 0) {
            return 0;
        }
        return this.drain(dst);
    }

    @Override
    public int drain(ByteBuffer dst) throws IOException {
        int bytesUnwrapped;
        this.verifyHandshakeCompleted();
        int totalBytesUnwrapped = 0;
        do {
            bytesUnwrapped = this.applicationDataUnwrap(dst);
            totalBytesUnwrapped += bytesUnwrapped;
        } while (bytesUnwrapped > 0);
        return totalBytesUnwrapped;
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        int bytesWrapped;
        this.verifyHandshakeCompleted();
        if (this.flush() == CryptoSocket.FlushResult.NEED_WRITE) {
            return 0;
        }
        int totalBytesWrapped = 0;
        do {
            bytesWrapped = this.applicationDataWrap(src);
            totalBytesWrapped += bytesWrapped;
        } while (bytesWrapped > 0 && this.wrapBuffer.bytes() < this.sessionPacketBufferSize);
        return totalBytesWrapped;
    }

    @Override
    public CryptoSocket.FlushResult flush() throws IOException {
        this.verifyHandshakeCompleted();
        this.channelWrite();
        return this.wrapBuffer.bytes() > 0 ? CryptoSocket.FlushResult.NEED_WRITE : CryptoSocket.FlushResult.DONE;
    }

    private boolean handshakeWrap() throws IOException {
        SSLEngineResult result = this.sslEngineWrap(NULL_BUFFER);
        switch (result.getStatus()) {
            case OK: {
                return true;
            }
            case BUFFER_OVERFLOW: {
                return false;
            }
        }
        throw TlsCryptoSocket.unexpectedStatusException(result.getStatus());
    }

    private int applicationDataWrap(ByteBuffer src) throws IOException {
        SSLEngineResult result = this.sslEngineWrap(src);
        if (result.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING) {
            throw new SSLException("Renegotiation detected");
        }
        switch (result.getStatus()) {
            case OK: {
                return result.bytesConsumed();
            }
            case BUFFER_OVERFLOW: {
                return 0;
            }
        }
        throw TlsCryptoSocket.unexpectedStatusException(result.getStatus());
    }

    private SSLEngineResult sslEngineWrap(ByteBuffer src) throws IOException {
        SSLEngineResult result = this.sslEngine.wrap(src, this.wrapBuffer.getWritable(this.sessionPacketBufferSize));
        if (result.getStatus() == SSLEngineResult.Status.CLOSED) {
            throw new ClosedChannelException();
        }
        return result;
    }

    private boolean handshakeUnwrap() throws IOException {
        SSLEngineResult result = this.sslEngineUnwrap(this.handshakeDummyBuffer);
        switch (result.getStatus()) {
            case OK: {
                if (result.bytesProduced() > 0) {
                    throw new SSLException("Got application data in handshake unwrap");
                }
                return true;
            }
            case BUFFER_UNDERFLOW: {
                return false;
            }
        }
        throw TlsCryptoSocket.unexpectedStatusException(result.getStatus());
    }

    private int applicationDataUnwrap(ByteBuffer dst) throws IOException {
        SSLEngineResult result = this.sslEngineUnwrap(dst);
        if (result.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING) {
            throw new SSLException("Renegotiation detected");
        }
        switch (result.getStatus()) {
            case OK: {
                return result.bytesProduced();
            }
            case BUFFER_OVERFLOW: 
            case BUFFER_UNDERFLOW: {
                return 0;
            }
        }
        throw TlsCryptoSocket.unexpectedStatusException(result.getStatus());
    }

    private SSLEngineResult sslEngineUnwrap(ByteBuffer dst) throws IOException {
        SSLEngineResult result = this.sslEngine.unwrap(this.unwrapBuffer.getReadable(), dst);
        if (result.getStatus() == SSLEngineResult.Status.CLOSED) {
            throw new ClosedChannelException();
        }
        return result;
    }

    private int channelRead() throws IOException {
        int read = this.channel.read(this.unwrapBuffer.getWritable(this.sessionPacketBufferSize));
        if (read == -1) {
            throw new ClosedChannelException();
        }
        return read;
    }

    private int channelWrite() throws IOException {
        return this.channel.write(this.wrapBuffer.getReadable());
    }

    private static IllegalStateException unhandledStateException(HandshakeState state) {
        return new IllegalStateException("Unhandled state: " + (Object)((Object)state));
    }

    private static IllegalStateException unexpectedStatusException(SSLEngineResult.Status status) {
        return new IllegalStateException("Unexpected status: " + (Object)((Object)status));
    }

    private void verifyHandshakeCompleted() throws SSLException {
        if (this.handshakeState != HandshakeState.COMPLETED) {
            throw new SSLException("Handshake not completed: handshakeState=" + (Object)((Object)this.handshakeState));
        }
    }

    private static enum HandshakeState {
        NOT_STARTED,
        NEED_READ,
        NEED_WRITE,
        COMPLETED;

    }
}

