/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.filedistribution;

import com.yahoo.concurrent.DaemonThreadFactory;
import com.yahoo.config.FileReference;
import com.yahoo.jrt.Int32Value;
import com.yahoo.jrt.Request;
import com.yahoo.jrt.Spec;
import com.yahoo.jrt.StringArray;
import com.yahoo.jrt.StringValue;
import com.yahoo.jrt.Value;
import com.yahoo.vespa.config.Connection;
import com.yahoo.vespa.config.ConnectionPool;
import com.yahoo.vespa.filedistribution.Downloads;
import com.yahoo.vespa.filedistribution.FileApiErrorCodes;
import com.yahoo.vespa.filedistribution.FileDownloader;
import com.yahoo.vespa.filedistribution.FileReferenceData;
import com.yahoo.vespa.filedistribution.FileReferenceDownload;
import java.io.File;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

public class FileReferenceDownloader {
    private static final Logger log = Logger.getLogger(FileReferenceDownloader.class.getName());
    private static final Set<FileReferenceData.CompressionType> defaultAcceptedCompressionTypes = Set.of(FileReferenceData.CompressionType.gzip, FileReferenceData.CompressionType.lz4, FileReferenceData.CompressionType.zstd);
    private final ExecutorService downloadExecutor = Executors.newFixedThreadPool(Math.max(8, Runtime.getRuntime().availableProcessors()), (ThreadFactory)new DaemonThreadFactory("filereference downloader"));
    private final ConnectionPool connectionPool;
    private final Downloads downloads;
    private final Duration downloadTimeout;
    private final Duration backoffInitialTime;
    private final Optional<Duration> rpcTimeout;
    private final File downloadDirectory;
    private final AtomicBoolean shutDown = new AtomicBoolean(false);

    FileReferenceDownloader(ConnectionPool connectionPool, Downloads downloads, Duration timeout, Duration backoffInitialTime, File downloadDirectory) {
        this.connectionPool = connectionPool;
        this.downloads = downloads;
        this.downloadTimeout = timeout;
        this.backoffInitialTime = backoffInitialTime;
        this.downloadDirectory = downloadDirectory;
        Optional<String> timeoutString = Optional.ofNullable(System.getenv("VESPA_FILE_DOWNLOAD_RPC_TIMEOUT"));
        this.rpcTimeout = timeoutString.map(t -> Duration.ofSeconds(Integer.parseInt(t)));
    }

    private void waitUntilDownloadStarted(FileReferenceDownload fileReferenceDownload) {
        Instant end = Instant.now().plus(this.downloadTimeout);
        FileReference fileReference = fileReferenceDownload.fileReference();
        int retryCount = 0;
        Connection connection = this.connectionPool.getCurrent();
        do {
            if (retryCount > 0) {
                this.backoff(retryCount, end);
            }
            if (this.shutDown.get()) {
                return;
            }
            if (FileDownloader.fileReferenceExists(fileReference, this.downloadDirectory)) {
                return;
            }
            Duration timeout = this.rpcTimeout.orElse(Duration.between(Instant.now(), end));
            log.log(Level.FINE, "Wait until download of " + fileReference + " has started, retryCount " + retryCount + " timeout" + timeout + " (request from client " + fileReferenceDownload.client() + ")");
            if (!timeout.isNegative() && this.startDownloadRpc(fileReferenceDownload, retryCount, connection, timeout)) {
                return;
            }
            ++retryCount;
            connection = this.connectionPool.switchConnection(connection);
        } while (Instant.now().isBefore(end));
        fileReferenceDownload.future().completeExceptionally(new RuntimeException("Failed getting " + fileReference));
        this.downloads.remove(fileReference);
    }

    private void backoff(int retryCount, Instant end) {
        try {
            long sleepTime = Math.min(120000L, Math.min((long)Math.pow(2.0, retryCount) * this.backoffInitialTime.toMillis(), Duration.between(Instant.now(), end).toMillis()));
            if (sleepTime <= 0L) {
                return;
            }
            Instant endSleep = Instant.now().plusMillis(sleepTime);
            do {
                Thread.sleep(Math.min(100L, sleepTime));
            } while (Instant.now().isBefore(endSleep) && !this.shutDown.get());
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    CompletableFuture<Optional<File>> startDownload(FileReferenceDownload fileReferenceDownload) {
        FileReference fileReference = fileReferenceDownload.fileReference();
        Optional<FileReferenceDownload> inProgress = this.downloads.get(fileReference);
        if (inProgress.isPresent()) {
            return inProgress.get().future();
        }
        this.downloads.add(fileReferenceDownload);
        this.downloadExecutor.submit(() -> this.waitUntilDownloadStarted(fileReferenceDownload));
        return fileReferenceDownload.future();
    }

    void startDownloadFromSource(FileReferenceDownload fileReferenceDownload, Spec spec) {
        FileReference fileReference = fileReferenceDownload.fileReference();
        for (Connection connection : this.connectionPool.connections()) {
            if (!connection.getAddress().equals(spec.toString())) continue;
            this.downloadExecutor.submit(() -> {
                if (this.downloads.get(fileReference).isPresent()) {
                    return;
                }
                log.log(Level.FINE, () -> "Will download " + fileReference + " with timeout " + this.downloadTimeout + " from " + spec);
                this.downloads.add(fileReferenceDownload);
                boolean downloading = this.startDownloadRpc(fileReferenceDownload, 1, connection, this.downloadTimeout);
                if (!downloading) {
                    this.downloads.remove(fileReference);
                }
            });
        }
    }

    void failedDownloading(FileReference fileReference) {
        this.downloads.remove(fileReference);
    }

    private boolean startDownloadRpc(FileReferenceDownload fileReferenceDownload, int retryCount, Connection connection, Duration timeout) {
        Request request = this.createRequest(fileReferenceDownload);
        connection.invokeSync(request, timeout);
        Level logLevel = retryCount > 3 ? Level.INFO : Level.FINE;
        FileReference fileReference = fileReferenceDownload.fileReference();
        String address = connection.getAddress();
        if (this.validateResponse(request)) {
            log.log(Level.FINE, () -> "Request callback, OK. Req: " + request + "\nSpec: " + connection);
            int errorCode = request.returnValues().get(0).asInt32();
            if (errorCode == 0) {
                log.log(Level.FINE, () -> "Found " + fileReference + " available at " + address);
                return true;
            }
            FileApiErrorCodes error = FileApiErrorCodes.get(errorCode);
            log.log(logLevel, "Downloading " + fileReference + " from " + address + " failed (" + error + ")");
            return false;
        }
        log.log(logLevel, "Downloading " + fileReference + " from " + address + " failed: error code " + request.errorCode() + " (" + request.errorMessage() + "). (retry " + retryCount + ", rpc timeout " + timeout + ")");
        return false;
    }

    private Request createRequest(FileReferenceDownload fileReferenceDownload) {
        Request request = new Request("filedistribution.serveFile");
        request.parameters().add((Value)new StringValue(fileReferenceDownload.fileReference().value()));
        request.parameters().add((Value)new Int32Value(fileReferenceDownload.downloadFromOtherSourceIfNotFound() ? 0 : 1));
        String[] temp = new String[defaultAcceptedCompressionTypes.size()];
        defaultAcceptedCompressionTypes.stream().map(Enum::name).toList().toArray(temp);
        request.parameters().add((Value)new StringArray(temp));
        return request;
    }

    private boolean validateResponse(Request request) {
        if (request.isError()) {
            return false;
        }
        if (request.returnValues().size() == 0) {
            return false;
        }
        if (!request.checkReturnTypes("is")) {
            log.log(Level.WARNING, "Invalid return types for response: " + request.errorMessage());
            return false;
        }
        return true;
    }

    public void close() {
        this.shutDown.set(true);
        this.downloadExecutor.shutdown();
        try {
            if (!this.downloadExecutor.awaitTermination(30L, TimeUnit.SECONDS)) {
                log.log(Level.WARNING, "FileReferenceDownloader failed to shutdown within 30 seconds");
            }
        }
        catch (InterruptedException e) {
            Thread.interrupted();
        }
    }
}

