/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.filedistribution;

import com.yahoo.config.FileReference;
import com.yahoo.jrt.Spec;
import com.yahoo.jrt.Supervisor;
import com.yahoo.vespa.config.Connection;
import com.yahoo.vespa.config.ConnectionPool;
import com.yahoo.vespa.defaults.Defaults;
import com.yahoo.vespa.filedistribution.Downloads;
import com.yahoo.vespa.filedistribution.FileReceiver;
import com.yahoo.vespa.filedistribution.FileReferenceDownload;
import com.yahoo.vespa.filedistribution.FileReferenceDownloader;
import java.io.File;
import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;

public class FileDownloader
implements AutoCloseable {
    private static final Logger log = Logger.getLogger(FileDownloader.class.getName());
    private static final Duration backoffInitialTime;
    public static final File defaultDownloadDirectory;
    private static final boolean forceDownload;
    private final ConnectionPool connectionPool;
    private final Supervisor supervisor;
    private final File downloadDirectory;
    private final Duration timeout;
    private final FileReferenceDownloader fileReferenceDownloader;
    private final Downloads downloads = new Downloads();

    public FileDownloader(ConnectionPool connectionPool, Supervisor supervisor, Duration timeout) {
        this(connectionPool, supervisor, defaultDownloadDirectory, timeout, backoffInitialTime);
    }

    public FileDownloader(ConnectionPool connectionPool, Supervisor supervisor, File downloadDirectory, Duration timeout) {
        this(connectionPool, supervisor, downloadDirectory, timeout, backoffInitialTime);
    }

    public FileDownloader(ConnectionPool connectionPool, Supervisor supervisor, File downloadDirectory, Duration timeout, Duration backoffInitialTime) {
        this.connectionPool = connectionPool;
        this.supervisor = supervisor;
        this.downloadDirectory = downloadDirectory;
        this.timeout = timeout;
        new FileReceiver(supervisor, this.downloads, downloadDirectory);
        this.fileReferenceDownloader = new FileReferenceDownloader(connectionPool, this.downloads, timeout, backoffInitialTime, downloadDirectory);
        if (forceDownload) {
            log.log(Level.INFO, "Force download of file references (download even if file reference exists on disk)");
        }
    }

    public Optional<File> getFile(FileReferenceDownload fileReferenceDownload) {
        try {
            return this.getFutureFile(fileReferenceDownload).get(this.timeout.toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException | ExecutionException | TimeoutException e) {
            this.fileReferenceDownloader.failedDownloading(fileReferenceDownload.fileReference());
            return Optional.empty();
        }
    }

    public Future<Optional<File>> getFutureFileOrTimeout(FileReferenceDownload fileReferenceDownload) {
        return this.getFutureFile(fileReferenceDownload).orTimeout(this.timeout.toMillis(), TimeUnit.MILLISECONDS).exceptionally(thrown -> {
            this.fileReferenceDownloader.failedDownloading(fileReferenceDownload.fileReference());
            return Optional.empty();
        });
    }

    CompletableFuture<Optional<File>> getFutureFile(FileReferenceDownload fileReferenceDownload) {
        FileReference fileReference = fileReferenceDownload.fileReference();
        Optional<File> file = this.getFileFromFileSystem(fileReference);
        if (file.isPresent()) {
            this.downloads.setDownloadStatus(fileReference, 1.0);
            return CompletableFuture.completedFuture(file);
        }
        return this.startDownload(fileReferenceDownload);
    }

    public Map<FileReference, Double> downloadStatus() {
        return this.downloads.downloadStatus();
    }

    public ConnectionPool connectionPool() {
        return this.connectionPool;
    }

    public Downloads downloads() {
        return this.downloads;
    }

    File downloadDirectory() {
        return this.downloadDirectory;
    }

    private Optional<File> getFileFromFileSystem(FileReference fileReference) {
        return FileDownloader.getFileFromFileSystem(fileReference, this.downloadDirectory);
    }

    private static Optional<File> getFileFromFileSystem(FileReference fileReference, File downloadDirectory) {
        if (forceDownload) {
            return Optional.empty();
        }
        File[] files = new File(downloadDirectory, fileReference.value()).listFiles();
        if (files == null) {
            return Optional.empty();
        }
        if (files.length == 0) {
            return Optional.empty();
        }
        if (files.length > 1) {
            throw new RuntimeException("More than one file reference found for " + fileReference);
        }
        File file = files[0];
        if (!file.exists()) {
            throw new RuntimeException("File reference '" + fileReference.value() + "' does not exist");
        }
        if (!file.canRead()) {
            throw new RuntimeException("File reference '" + fileReference.value() + "' exists, but unable to read it");
        }
        log.log(Level.FINE, () -> "File reference '" + fileReference.value() + "' found: " + file.getAbsolutePath());
        return Optional.of(file);
    }

    static boolean fileReferenceExists(FileReference fileReference, File downloadDirectory) {
        return FileDownloader.getFileFromFileSystem(fileReference, downloadDirectory).isPresent();
    }

    boolean isDownloading(FileReference fileReference) {
        return this.downloads.get(fileReference).isPresent();
    }

    public boolean downloadFromSource(FileReferenceDownload fileReferenceDownload, Spec source) {
        if (FileDownloader.fileReferenceExists(fileReferenceDownload.fileReference(), this.downloadDirectory)) {
            return false;
        }
        this.fileReferenceDownloader.startDownloadFromSource(fileReferenceDownload, source);
        return true;
    }

    private synchronized CompletableFuture<Optional<File>> startDownload(FileReferenceDownload fileReferenceDownload) {
        return this.fileReferenceDownloader.startDownload(fileReferenceDownload);
    }

    @Override
    public void close() {
        this.fileReferenceDownloader.close();
        this.supervisor.transport().shutdown().join();
    }

    public static ConnectionPool emptyConnectionPool() {
        return new EmptyConnectionPool();
    }

    static {
        defaultDownloadDirectory = new File(Defaults.getDefaults().underVespaHome("var/db/vespa/filedistribution"));
        forceDownload = Boolean.parseBoolean(System.getenv("VESPA_FORCE_DOWNLOAD_OF_FILE_REFERENCES"));
        String backOff = System.getenv("VESPA_FILE_DOWNLOAD_BACKOFF_INITIAL_TIME_MS");
        backoffInitialTime = Duration.ofMillis(backOff == null ? 1000L : Long.parseLong(backOff));
    }

    private static class EmptyConnectionPool
    implements ConnectionPool {
        private EmptyConnectionPool() {
        }

        public void close() {
        }

        public Connection getCurrent() {
            return null;
        }

        public Connection switchConnection(Connection connection) {
            return null;
        }

        public int getSize() {
            return 0;
        }
    }
}

