/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.filedistribution;

import com.yahoo.concurrent.DaemonThreadFactory;
import com.yahoo.config.FileReference;
import com.yahoo.jrt.Int32Value;
import com.yahoo.jrt.Request;
import com.yahoo.jrt.StringArray;
import com.yahoo.jrt.StringValue;
import com.yahoo.jrt.Value;
import com.yahoo.vespa.config.Connection;
import com.yahoo.vespa.config.ConnectionPool;
import com.yahoo.vespa.filedistribution.Downloads;
import com.yahoo.vespa.filedistribution.FileDownloader;
import com.yahoo.vespa.filedistribution.FileReferenceData;
import com.yahoo.vespa.filedistribution.FileReferenceDownload;
import java.io.File;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

public class FileReferenceDownloader {
    private static final Logger log = Logger.getLogger(FileReferenceDownloader.class.getName());
    private final ExecutorService downloadExecutor = Executors.newFixedThreadPool(Math.max(8, Runtime.getRuntime().availableProcessors()), (ThreadFactory)new DaemonThreadFactory("filereference downloader"));
    private final ConnectionPool connectionPool;
    private final Downloads downloads;
    private final Duration downloadTimeout;
    private final Duration sleepBetweenRetries;
    private final Duration rpcTimeout;
    private final File downloadDirectory;
    private final Set<FileReferenceData.CompressionType> acceptedCompressionTypes;

    FileReferenceDownloader(ConnectionPool connectionPool, Downloads downloads, Duration timeout, Duration sleepBetweenRetries, File downloadDirectory, Set<FileReferenceData.CompressionType> acceptedCompressionTypes) {
        this.connectionPool = connectionPool;
        this.downloads = downloads;
        this.downloadTimeout = timeout;
        this.sleepBetweenRetries = sleepBetweenRetries;
        this.downloadDirectory = downloadDirectory;
        String timeoutString = System.getenv("VESPA_CONFIGPROXY_FILEDOWNLOAD_RPC_TIMEOUT");
        this.rpcTimeout = Duration.ofSeconds(timeoutString == null ? 30L : (long)Integer.parseInt(timeoutString));
        this.acceptedCompressionTypes = FileReferenceDownloader.requireNonEmpty(acceptedCompressionTypes);
    }

    private void waitUntilDownloadStarted(FileReferenceDownload fileReferenceDownload) {
        Instant end = Instant.now().plus(this.downloadTimeout);
        FileReference fileReference = fileReferenceDownload.fileReference();
        int retryCount = 0;
        Connection connection = this.connectionPool.getCurrent();
        do {
            this.backoff(retryCount);
            if (FileDownloader.fileReferenceExists(fileReference, this.downloadDirectory)) {
                return;
            }
            if (this.startDownloadRpc(fileReferenceDownload, retryCount, connection)) {
                return;
            }
            connection = this.connectionPool.switchConnection(connection);
        } while (++retryCount < 5 || Instant.now().isAfter(end));
        fileReferenceDownload.future().completeExceptionally(new RuntimeException("Failed getting " + fileReference));
        this.downloads.remove(fileReference);
    }

    private void backoff(int retryCount) {
        if (retryCount > 0) {
            try {
                long sleepTime = Math.min(120000L, (long)Math.pow(2.0, retryCount) * this.sleepBetweenRetries.toMillis());
                Thread.sleep(sleepTime);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
    }

    Future<Optional<File>> startDownload(FileReferenceDownload fileReferenceDownload) {
        FileReference fileReference = fileReferenceDownload.fileReference();
        Optional<FileReferenceDownload> inProgress = this.downloads.get(fileReference);
        if (inProgress.isPresent()) {
            return inProgress.get().future();
        }
        log.log(Level.FINE, () -> "Will download " + fileReference + " with timeout " + this.downloadTimeout);
        this.downloads.add(fileReferenceDownload);
        this.downloadExecutor.submit(() -> this.waitUntilDownloadStarted(fileReferenceDownload));
        return fileReferenceDownload.future();
    }

    void failedDownloading(FileReference fileReference) {
        this.downloads.remove(fileReference);
    }

    private boolean startDownloadRpc(FileReferenceDownload fileReferenceDownload, int retryCount, Connection connection) {
        Request request = this.createRequest(fileReferenceDownload);
        Duration rpcTimeout = this.rpcTimeout(retryCount);
        connection.invokeSync(request, rpcTimeout);
        Level logLevel = retryCount > 3 ? Level.INFO : Level.FINE;
        FileReference fileReference = fileReferenceDownload.fileReference();
        if (this.validateResponse(request)) {
            log.log(Level.FINE, () -> "Request callback, OK. Req: " + request + "\nSpec: " + connection);
            int errorCode = request.returnValues().get(0).asInt32();
            if (errorCode == 0) {
                log.log(Level.FINE, () -> "Found " + fileReference + " available at " + connection.getAddress());
                return true;
            }
            log.log(logLevel, fileReference + " not found or timed out (error code " + errorCode + ") at " + connection.getAddress());
            return false;
        }
        log.log(logLevel, "Downloading " + fileReference + " from " + connection.getAddress() + " failed: error code " + request.errorCode() + " (" + request.errorMessage() + "). (retry " + retryCount + ", rpc timeout " + rpcTimeout + ")");
        return false;
    }

    private Request createRequest(FileReferenceDownload fileReferenceDownload) {
        Request request = new Request("filedistribution.serveFile");
        request.parameters().add((Value)new StringValue(fileReferenceDownload.fileReference().value()));
        request.parameters().add((Value)new Int32Value(fileReferenceDownload.downloadFromOtherSourceIfNotFound() ? 0 : 1));
        String[] temp = new String[this.acceptedCompressionTypes.size()];
        this.acceptedCompressionTypes.stream().map(Enum::name).toList().toArray(temp);
        request.parameters().add((Value)new StringArray(temp));
        return request;
    }

    private Duration rpcTimeout(int retryCount) {
        return Duration.ofSeconds(this.rpcTimeout.getSeconds()).plus(Duration.ofSeconds((long)retryCount * 5L));
    }

    private boolean validateResponse(Request request) {
        if (request.isError()) {
            return false;
        }
        if (request.returnValues().size() == 0) {
            return false;
        }
        if (!request.checkReturnTypes("is")) {
            log.log(Level.WARNING, "Invalid return types for response: " + request.errorMessage());
            return false;
        }
        return true;
    }

    public void close() {
        this.downloadExecutor.shutdown();
        try {
            this.downloadExecutor.awaitTermination(30L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.interrupted();
        }
    }

    private static Set<FileReferenceData.CompressionType> requireNonEmpty(Set<FileReferenceData.CompressionType> s) {
        if (Objects.requireNonNull(s).isEmpty()) {
            throw new IllegalArgumentException("set must be non-empty");
        }
        return s;
    }
}

