/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.filedistribution;

import com.yahoo.config.FileReference;
import com.yahoo.jrt.Int32Value;
import com.yahoo.jrt.Method;
import com.yahoo.jrt.Request;
import com.yahoo.jrt.Supervisor;
import com.yahoo.jrt.Value;
import com.yahoo.log.LogLevel;
import com.yahoo.vespa.filedistribution.CompressedFileReference;
import com.yahoo.vespa.filedistribution.FileReferenceData;
import com.yahoo.vespa.filedistribution.FileReferenceDataBlob;
import com.yahoo.vespa.filedistribution.FileReferenceDownloader;
import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.jpountz.xxhash.StreamingXXHash64;
import net.jpountz.xxhash.XXHashFactory;

public class FileReceiver {
    private static final Logger log = Logger.getLogger(FileReceiver.class.getName());
    public static final String RECEIVE_METHOD = "filedistribution.receiveFile";
    public static final String RECEIVE_META_METHOD = "filedistribution.receiveFileMeta";
    public static final String RECEIVE_PART_METHOD = "filedistribution.receiveFilePart";
    public static final String RECEIVE_EOF_METHOD = "filedistribution.receiveFileEof";
    private final Supervisor supervisor;
    private final FileReferenceDownloader downloader;
    private final File downloadDirectory;
    private final File tmpDirectory;
    private final AtomicInteger nextSessionId = new AtomicInteger(1);
    private final Map<Integer, Session> sessions = new HashMap<Integer, Session>();

    FileReceiver(Supervisor supervisor, FileReferenceDownloader downloader, File downloadDirectory, File tmpDirectory) {
        this.supervisor = supervisor;
        this.downloader = downloader;
        this.downloadDirectory = downloadDirectory;
        this.tmpDirectory = tmpDirectory;
        this.registerMethods();
    }

    private void registerMethods() {
        this.receiveFileMethod(this).forEach(arg_0 -> ((Supervisor)this.supervisor).addMethod(arg_0));
    }

    private List<Method> receiveFileMethod(Object handler) {
        ArrayList<Method> methods = new ArrayList<Method>();
        methods.add(new Method(RECEIVE_META_METHOD, "sssl", "ii", handler, "receiveFileMeta").paramDesc(0, "filereference", "file reference to download").paramDesc(1, "filename", "filename").paramDesc(2, "type", "'file' or 'compressed'").paramDesc(3, "filelength", "length in bytes of file").returnDesc(0, "ret", "0 if success, 1 otherwise").returnDesc(1, "session-id", "Session id to be used for this transfer"));
        methods.add(new Method(RECEIVE_PART_METHOD, "siix", "i", handler, "receiveFilePart").paramDesc(0, "filereference", "file reference to download").paramDesc(1, "session-id", "Session id to be used for this transfer").paramDesc(2, "partid", "relative part number starting at zero").paramDesc(3, "data", "bytes in this part").returnDesc(0, "ret", "0 if success, 1 otherwise"));
        methods.add(new Method(RECEIVE_EOF_METHOD, "silis", "i", handler, "receiveFileEof").paramDesc(0, "filereference", "file reference to download").paramDesc(1, "session-id", "Session id to be used for this transfer").paramDesc(2, "crc-code", "crc code (xxhash64)").paramDesc(3, "error-code", "Error code. 0 if none").paramDesc(4, "error-description", "Error description.").returnDesc(0, "ret", "0 if success, 1 if crc mismatch, 2 otherwise"));
        methods.add(new Method(RECEIVE_METHOD, "sssxlis", "i", handler, "receiveFile").methodDesc("receive file reference content").paramDesc(0, "file reference", "file reference to download").paramDesc(1, "filename", "filename").paramDesc(2, "type", "'file' or 'compressed'").paramDesc(3, "content", "array of bytes").paramDesc(4, "hash", "xx64hash of the file content").paramDesc(5, "errorcode", "Error code. 0 if none").paramDesc(6, "error-description", "Error description.").returnDesc(0, "ret", "0 if success, 1 otherwise"));
        return methods;
    }

    public final void receiveFile(Request req) {
        FileReference fileReference = new FileReference(req.parameters().get(0).asString());
        String filename = req.parameters().get(1).asString();
        String type = req.parameters().get(2).asString();
        byte[] content = req.parameters().get(3).asData();
        long xxhash = req.parameters().get(4).asInt64();
        int errorCode = req.parameters().get(5).asInt32();
        String errorDescription = req.parameters().get(6).asString();
        if (errorCode == 0) {
            log.log((Level)LogLevel.DEBUG, "Receiving file reference '" + fileReference.value() + "'");
            this.receiveFile(new FileReferenceDataBlob(fileReference, filename, FileReferenceData.Type.valueOf(type), content, xxhash));
            req.returnValues().add((Value)new Int32Value(0));
        } else {
            log.log(LogLevel.WARNING, "Receiving file reference '" + fileReference.value() + "' failed: " + errorDescription);
            req.returnValues().add((Value)new Int32Value(1));
        }
    }

    void receiveFile(FileReferenceData fileReferenceData) {
        long xxHashFromContent = fileReferenceData.xxhash();
        if (xxHashFromContent != fileReferenceData.xxhash()) {
            throw new RuntimeException("xxhash from content (" + xxHashFromContent + ") is not equal to xxhash in request (" + fileReferenceData.xxhash() + ")");
        }
        File fileReferenceDir = new File(this.downloadDirectory, fileReferenceData.fileReference().value());
        File file = new File(fileReferenceDir, fileReferenceData.filename());
        try {
            File tempDownloadedDir = Files.createTempDirectory(this.tmpDirectory.toPath(), "downloaded", new FileAttribute[0]).toFile();
            File tempFile = new File(tempDownloadedDir, fileReferenceData.filename());
            Files.write(tempFile.toPath(), fileReferenceData.content().array(), new OpenOption[0]);
            if (fileReferenceData.type() == FileReferenceData.Type.compressed) {
                File decompressedDir = Files.createTempDirectory(tempDownloadedDir.toPath(), "decompressed", new FileAttribute[0]).toFile();
                log.log((Level)LogLevel.DEBUG, "Compressed file, unpacking " + tempFile + " to " + decompressedDir);
                CompressedFileReference.decompress(tempFile, decompressedDir);
                FileReceiver.moveFileToDestination(decompressedDir, fileReferenceDir);
            } else {
                log.log((Level)LogLevel.DEBUG, "Uncompressed file, moving to " + file.getAbsolutePath());
                Files.createDirectories(fileReferenceDir.toPath(), new FileAttribute[0]);
                FileReceiver.moveFileToDestination(tempFile, file);
            }
            this.downloader.completedDownloading(fileReferenceData.fileReference(), file);
        }
        catch (IOException e) {
            log.log((Level)LogLevel.ERROR, "Failed writing file: " + e.getMessage(), e);
            throw new RuntimeException("Failed writing file: ", e);
        }
    }

    private static void moveFileToDestination(File tempFile, File destination) {
        try {
            Files.move(tempFile.toPath(), destination.toPath(), new CopyOption[0]);
            log.log((Level)LogLevel.DEBUG, "File moved from " + tempFile.getAbsolutePath() + " to " + destination.getAbsolutePath());
        }
        catch (FileAlreadyExistsException e) {
            log.log((Level)LogLevel.DEBUG, "File '" + destination.getAbsolutePath() + "' already exists, continuing: " + e.getMessage());
        }
        catch (IOException e) {
            String message = "Failed moving file '" + tempFile.getAbsolutePath() + "' to '" + destination.getAbsolutePath() + "'";
            log.log((Level)LogLevel.ERROR, message, e);
            throw new RuntimeException(message, e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void receiveFileMeta(Request req) {
        log.log((Level)LogLevel.DEBUG, "Received method call '" + req.methodName() + "' with parameters : " + req.parameters());
        FileReference reference = new FileReference(req.parameters().get(0).asString());
        String fileName = req.parameters().get(1).asString();
        String type = req.parameters().get(2).asString();
        long fileSize = req.parameters().get(3).asInt64();
        int sessionId = this.nextSessionId.getAndIncrement();
        int retval = 0;
        Map<Integer, Session> map = this.sessions;
        synchronized (map) {
            if (this.sessions.containsKey(sessionId)) {
                retval = 1;
                log.severe("Session id " + sessionId + " already exist, impossible. Request from(" + req.target() + ")");
            } else {
                try {
                    this.sessions.put(sessionId, new Session(this.downloadDirectory, this.tmpDirectory, sessionId, reference, FileReferenceData.Type.valueOf(type), fileName, fileSize));
                }
                catch (Exception e) {
                    retval = 1;
                }
            }
        }
        req.returnValues().add((Value)new Int32Value(retval));
        req.returnValues().add((Value)new Int32Value(sessionId));
    }

    public final void receiveFilePart(Request req) {
        log.log((Level)LogLevel.DEBUG, "Received method call '" + req.methodName() + "' with parameters : " + req.parameters());
        FileReference reference = new FileReference(req.parameters().get(0).asString());
        int sessionId = req.parameters().get(1).asInt32();
        int partId = req.parameters().get(2).asInt32();
        byte[] part = req.parameters().get(3).asData();
        Session session = this.getSession(sessionId);
        int retval = FileReceiver.verifySession(session, sessionId, reference);
        try {
            session.addPart(partId, part);
        }
        catch (Exception e) {
            log.severe("Got exception + " + e);
            retval = 1;
        }
        double completeness = (double)session.currentFileSize / (double)session.fileSize;
        log.log((Level)LogLevel.DEBUG, String.format("%.1f percent of '%s' downloaded", completeness * 100.0, reference.value()));
        this.downloader.setDownloadStatus(reference, completeness);
        req.returnValues().add((Value)new Int32Value(retval));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void receiveFileEof(Request req) {
        log.log((Level)LogLevel.DEBUG, "Received method call '" + req.methodName() + "' with parameters : " + req.parameters());
        FileReference reference = new FileReference(req.parameters().get(0).asString());
        int sessionId = req.parameters().get(1).asInt32();
        long xxhash = req.parameters().get(2).asInt64();
        Session session = this.getSession(sessionId);
        int retval = FileReceiver.verifySession(session, sessionId, reference);
        File file = session.close(xxhash);
        this.downloader.completedDownloading(reference, file);
        Map<Integer, Session> map = this.sessions;
        synchronized (map) {
            this.sessions.remove(sessionId);
        }
        req.returnValues().add((Value)new Int32Value(retval));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Session getSession(Integer sessionId) {
        Map<Integer, Session> map = this.sessions;
        synchronized (map) {
            return this.sessions.get(sessionId);
        }
    }

    private static int verifySession(Session session, int sessionId, FileReference reference) {
        if (session == null) {
            log.severe("session-id " + sessionId + " does not exist.");
            return 1;
        }
        if (!session.reference.equals((Object)reference)) {
            log.severe("Session " + session.sessionId + " expects reference " + reference.value() + ", but was " + session.reference.value());
            return 1;
        }
        return 0;
    }

    static final class Session {
        private final StreamingXXHash64 hasher = XXHashFactory.fastestInstance().newStreamingHash64(0L);
        private final int sessionId;
        private final FileReference reference;
        private final FileReferenceData.Type fileType;
        private final String fileName;
        private final long fileSize;
        private long currentFileSize;
        private long currentPartId;
        private long currentHash;
        private final File fileReferenceDir;
        private final File tmpDir;
        private final File inprogressFile;

        Session(File downloadDirectory, File tmpDirectory, int sessionId, FileReference reference, FileReferenceData.Type fileType, String fileName, long fileSize) {
            this.sessionId = sessionId;
            this.reference = reference;
            this.fileType = fileType;
            this.fileName = fileName;
            this.fileSize = fileSize;
            this.currentFileSize = 0L;
            this.currentPartId = 0L;
            this.currentHash = 0L;
            this.fileReferenceDir = new File(downloadDirectory, reference.value());
            this.tmpDir = tmpDirectory;
            try {
                Files.createDirectories(this.fileReferenceDir.toPath(), new FileAttribute[0]);
            }
            catch (IOException e) {
                log.log((Level)LogLevel.ERROR, "Failed creating directory(" + this.fileReferenceDir.toPath() + "): " + e.getMessage(), e);
                throw new RuntimeException("Failed creating directory(" + this.fileReferenceDir.toPath() + "): ", e);
            }
            try {
                this.inprogressFile = Files.createTempFile(tmpDirectory.toPath(), fileName, ".inprogress", new FileAttribute[0]).toFile();
            }
            catch (IOException e) {
                String msg = "Failed creating tempfile for inprogress file for(" + fileName + ") in '" + this.fileReferenceDir.toPath() + "': ";
                log.log((Level)LogLevel.ERROR, msg + e.getMessage(), e);
                throw new RuntimeException(msg, e);
            }
        }

        void addPart(int partId, byte[] part) {
            if ((long)partId != this.currentPartId) {
                throw new IllegalStateException("Received partid " + partId + " while expecting " + this.currentPartId);
            }
            if (this.fileSize < this.currentFileSize + (long)part.length) {
                throw new IllegalStateException("Received part would extend the file from " + this.currentFileSize + " to " + (this.currentFileSize + (long)part.length) + ", but " + this.fileSize + " is max.");
            }
            try {
                Files.write(this.inprogressFile.toPath(), part, StandardOpenOption.WRITE, StandardOpenOption.APPEND);
            }
            catch (IOException e) {
                log.log((Level)LogLevel.ERROR, "Failed writing to file(" + this.inprogressFile.toPath() + "): " + e.getMessage(), e);
                throw new RuntimeException("Failed writing to file(" + this.inprogressFile.toPath() + "): ", e);
            }
            this.currentFileSize += (long)part.length;
            ++this.currentPartId;
            this.hasher.update(part, 0, part.length);
        }

        File close(long hash) {
            if (this.hasher.getValue() != hash) {
                throw new RuntimeException("xxhash from content (" + this.currentHash + ") is not equal to xxhash in request (" + hash + ")");
            }
            File file = new File(this.fileReferenceDir, this.fileName);
            try {
                if (this.fileType == FileReferenceData.Type.compressed) {
                    File decompressedDir = Files.createTempDirectory(this.tmpDir.toPath(), "archive", new FileAttribute[0]).toFile();
                    log.log((Level)LogLevel.DEBUG, "Archived file, unpacking " + this.inprogressFile + " to " + decompressedDir);
                    CompressedFileReference.decompress(this.inprogressFile, decompressedDir);
                    FileReceiver.moveFileToDestination(decompressedDir, this.fileReferenceDir);
                } else {
                    log.log((Level)LogLevel.DEBUG, "Uncompressed file, moving to " + file.getAbsolutePath());
                    FileReceiver.moveFileToDestination(this.inprogressFile, file);
                }
            }
            catch (IOException e) {
                log.log((Level)LogLevel.ERROR, "Failed writing file: " + e.getMessage(), e);
                throw new RuntimeException("Failed writing file: ", e);
            }
            return file;
        }

        double percentageReceived() {
            return (double)this.currentFileSize / (double)this.fileSize;
        }
    }
}

