/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.search.searchers;

import com.google.inject.Inject;
import com.yahoo.cloud.config.ClusterInfoConfig;
import com.yahoo.metrics.simple.Counter;
import com.yahoo.metrics.simple.MetricReceiver;
import com.yahoo.metrics.simple.Point;
import com.yahoo.processing.request.CompoundName;
import com.yahoo.search.Query;
import com.yahoo.search.Result;
import com.yahoo.search.Searcher;
import com.yahoo.search.config.RateLimitingConfig;
import com.yahoo.search.result.ErrorMessage;
import com.yahoo.search.searchchain.Execution;
import com.yahoo.yolean.chain.Provides;
import java.time.Clock;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;

@Provides(value={"rateLimiting"})
public class RateLimitingSearcher
extends Searcher {
    public static final String RATE_LIMITING = "rateLimiting";
    public static final CompoundName idKey = new CompoundName("rate.id");
    public static final CompoundName costKey = new CompoundName("rate.cost");
    public static final CompoundName quotaKey = new CompoundName("rate.quota");
    public static final CompoundName idDimensionKey = new CompoundName("rate.idDimension");
    public static final CompoundName dryRunKey = new CompoundName("rate.dryRun");
    private static final String requestsOverQuotaMetricName = "requestsOverQuota";
    private final int nodeCount;
    private final AvailableCapacity availableCapacity;
    private final ThreadLocal<Map<String, Double>> allocatedCapacity = new ThreadLocal();
    private final Counter overQuotaCounter;
    private final double capacityIncrement;
    private final double recheckForCapacityProbability;

    @Inject
    public RateLimitingSearcher(RateLimitingConfig rateLimitingConfig, ClusterInfoConfig clusterInfoConfig, MetricReceiver metric) {
        this(rateLimitingConfig, clusterInfoConfig, metric, Clock.systemUTC());
    }

    public RateLimitingSearcher(RateLimitingConfig rateLimitingConfig, ClusterInfoConfig clusterInfoConfig, MetricReceiver metric, Clock clock) {
        this.capacityIncrement = rateLimitingConfig.capacityIncrement();
        this.recheckForCapacityProbability = rateLimitingConfig.recheckForCapacityProbability();
        this.availableCapacity = new AvailableCapacity(rateLimitingConfig.maxAvailableCapacity(), clock);
        this.nodeCount = clusterInfoConfig.nodeCount();
        this.overQuotaCounter = metric.declareCounter(requestsOverQuotaMetricName);
    }

    @Override
    public Result search(Query query, Execution execution) {
        String id = query.properties().getString(idKey);
        Double rate = query.properties().getDouble(quotaKey);
        if (id == null || rate == null) {
            query.trace(false, 6, "Skipping rate limiting check. Need both " + idKey + " and " + quotaKey + " set");
            return execution.search(query);
        }
        rate = rate / (double)this.nodeCount;
        if (this.allocatedCapacity.get() == null) {
            this.allocatedCapacity.set(new HashMap());
        }
        if (this.allocatedCapacity.get().get(id) == null) {
            this.requestCapacity(id, rate);
        }
        if (this.getAllocatedCapacity(id) <= 0.0 && ThreadLocalRandom.current().nextDouble() < this.recheckForCapacityProbability) {
            this.requestCapacity(id, rate);
        }
        if (rate == 0.0 || this.getAllocatedCapacity(id) <= 0.0) {
            String idDim = query.properties().getString(idDimensionKey, null);
            if (idDim == null) {
                this.overQuotaCounter.add(1L);
            } else {
                this.overQuotaCounter.add(1L, this.createContext(idDim, id));
            }
            if (!query.properties().getBoolean(dryRunKey, false)) {
                return new Result(query, new ErrorMessage(429, "Too many requests", "Allowed rate: " + rate + "/s"));
            }
        }
        Result result = execution.search(query);
        this.addAllocatedCapacity(id, -query.properties().getDouble(costKey, 1.0).doubleValue());
        if (this.getAllocatedCapacity(id) <= 0.0) {
            this.requestCapacity(id, rate);
        }
        return result;
    }

    private Point createContext(String dimensionName, String dimensionValue) {
        return this.overQuotaCounter.builder().set(dimensionName, dimensionValue).build();
    }

    private double getAllocatedCapacity(String id) {
        Double value = this.allocatedCapacity.get().get(id);
        if (value == null) {
            return 0.0;
        }
        return value;
    }

    private void addAllocatedCapacity(String id, double newCapacity) {
        Double capacity = this.allocatedCapacity.get().get(id);
        if (capacity != null) {
            newCapacity += capacity.doubleValue();
        }
        this.allocatedCapacity.get().put(id, newCapacity);
    }

    private void requestCapacity(String id, double rate) {
        double minimumRequested = Math.max(0.0, -this.getAllocatedCapacity(id));
        double preferredRequested = Math.max(this.capacityIncrement, -this.getAllocatedCapacity(id));
        this.addAllocatedCapacity(id, this.availableCapacity.request(id, minimumRequested, preferredRequested, rate));
    }

    private static class AvailableCapacity {
        private final double maxAvailableCapacity;
        private final Clock clock;
        private final Map<String, CapacityAllocation> available = new HashMap<String, CapacityAllocation>();

        public AvailableCapacity(double maxAvailableCapacity, Clock clock) {
            this.maxAvailableCapacity = maxAvailableCapacity;
            this.clock = clock;
        }

        public synchronized double request(String id, double minimumRequested, double preferredRequested, double rate) {
            CapacityAllocation allocation = this.available.get(id);
            if (allocation == null) {
                allocation = new CapacityAllocation(rate, this.clock);
                this.available.put(id, allocation);
            }
            return allocation.request(minimumRequested, preferredRequested, rate, this.maxAvailableCapacity);
        }
    }

    private static class CapacityAllocation {
        private double capacity;
        private final Clock clock;
        private long lastAllocatedTime;

        public CapacityAllocation(double initialCapacity, Clock clock) {
            this.capacity = initialCapacity;
            this.clock = clock;
            this.lastAllocatedTime = clock.millis();
        }

        public double request(double minimumRequested, double preferredRequested, double rate, double maxAvailableCapacity) {
            double grantedCapacity;
            if (preferredRequested > this.capacity) {
                long currentTime = this.clock.millis();
                this.capacity += Math.min(maxAvailableCapacity, rate / 1000.0 * (double)Math.max(0L, currentTime - this.lastAllocatedTime));
                this.lastAllocatedTime = currentTime;
            }
            if ((grantedCapacity = Math.min(this.capacity / 10.0, preferredRequested)) < minimumRequested) {
                grantedCapacity = Math.min(minimumRequested, this.capacity);
            }
            this.capacity -= grantedCapacity;
            return grantedCapacity;
        }
    }
}

