/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.search.dispatch.searchcluster;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.math.Quantiles;
import com.yahoo.container.handler.VipStatus;
import com.yahoo.net.HostName;
import com.yahoo.prelude.Pong;
import com.yahoo.search.cluster.ClusterMonitor;
import com.yahoo.search.cluster.NodeManager;
import com.yahoo.search.dispatch.TopKEstimator;
import com.yahoo.search.dispatch.searchcluster.Group;
import com.yahoo.search.dispatch.searchcluster.Node;
import com.yahoo.search.dispatch.searchcluster.PingFactory;
import com.yahoo.search.dispatch.searchcluster.Pinger;
import com.yahoo.search.dispatch.searchcluster.PongHandler;
import com.yahoo.vespa.config.search.DispatchConfig;
import java.io.Serializable;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.logging.Logger;
import java.util.stream.Collectors;

public class SearchCluster
implements NodeManager<Node> {
    private static final Logger log = Logger.getLogger(SearchCluster.class.getName());
    private final DispatchConfig dispatchConfig;
    private final int size;
    private final String clusterId;
    private final ImmutableMap<Integer, Group> groups;
    private final ImmutableMultimap<String, Node> nodesByHost;
    private final ImmutableList<Group> orderedGroups;
    private final VipStatus vipStatus;
    private final PingFactory pingFactory;
    private final TopKEstimator hitEstimator;
    private long nextLogTime = 0L;
    private static final double SKEW_FACTOR = 0.05;
    private final Optional<Node> localCorpusDispatchTarget;

    public SearchCluster(String clusterId, DispatchConfig dispatchConfig, VipStatus vipStatus, PingFactory pingFactory) {
        this.clusterId = clusterId;
        this.dispatchConfig = dispatchConfig;
        this.vipStatus = vipStatus;
        this.pingFactory = pingFactory;
        ImmutableList<Node> nodes = SearchCluster.toNodes(dispatchConfig);
        this.size = nodes.size();
        ImmutableMap.Builder groupsBuilder = new ImmutableMap.Builder();
        for (Map.Entry<Integer, List<Node>> group : nodes.stream().collect(Collectors.groupingBy(Node::group)).entrySet()) {
            Group g = new Group(group.getKey(), group.getValue());
            groupsBuilder.put((Object)group.getKey(), (Object)g);
        }
        this.groups = groupsBuilder.build();
        LinkedHashMap groupIntroductionOrder = new LinkedHashMap();
        nodes.forEach(node -> groupIntroductionOrder.put(node.group(), (Group)this.groups.get((Object)node.group())));
        this.orderedGroups = ImmutableList.builder().addAll(groupIntroductionOrder.values()).build();
        ImmutableMultimap.Builder nodesByHostBuilder = new ImmutableMultimap.Builder();
        for (Node node2 : nodes) {
            nodesByHostBuilder.put((Object)node2.hostname(), (Object)node2);
        }
        this.nodesByHost = nodesByHostBuilder.build();
        this.hitEstimator = new TopKEstimator(30.0, dispatchConfig.topKProbability(), 0.05);
        this.localCorpusDispatchTarget = SearchCluster.findLocalCorpusDispatchTarget(HostName.getLocalhost(), this.nodesByHost, this.groups);
    }

    public void addMonitoring(ClusterMonitor<Node> clusterMonitor) {
        for (Group group : this.orderedGroups()) {
            for (Node node : group.nodes()) {
                clusterMonitor.add(node, true);
            }
        }
    }

    private static Optional<Node> findLocalCorpusDispatchTarget(String selfHostname, ImmutableMultimap<String, Node> nodesByHost, ImmutableMap<Integer, Group> groups) {
        ImmutableCollection localSearchNodes = nodesByHost.get((Object)selfHostname);
        if (localSearchNodes.size() != 1) {
            return Optional.empty();
        }
        Node localSearchNode = (Node)localSearchNodes.iterator().next();
        Group localSearchGroup = (Group)groups.get((Object)localSearchNode.group());
        if (localSearchGroup.nodes().size() != 1) {
            return Optional.empty();
        }
        return Optional.of(localSearchNode);
    }

    private static ImmutableList<Node> toNodes(DispatchConfig dispatchConfig) {
        ImmutableList.Builder nodesBuilder = new ImmutableList.Builder();
        for (DispatchConfig.Node node : dispatchConfig.node()) {
            nodesBuilder.add((Object)new Node(node.key(), node.host(), node.group()));
        }
        return nodesBuilder.build();
    }

    public DispatchConfig dispatchConfig() {
        return this.dispatchConfig;
    }

    public int size() {
        return this.size;
    }

    public ImmutableMap<Integer, Group> groups() {
        return this.groups;
    }

    public ImmutableList<Group> orderedGroups() {
        return this.orderedGroups;
    }

    public Optional<Group> group(int n) {
        if (this.orderedGroups().size() > n) {
            return Optional.of((Group)this.orderedGroups().get(n));
        }
        return Optional.empty();
    }

    public int wantedGroupSize() {
        if (this.groups().size() == 0) {
            return this.size();
        }
        return this.size() / this.groups().size();
    }

    public int groupsWithSufficientCoverage() {
        int covered = 0;
        for (Group g : this.orderedGroups()) {
            if (!g.hasSufficientCoverage()) continue;
            ++covered;
        }
        return covered;
    }

    public Optional<Node> localCorpusDispatchTarget() {
        if (this.localCorpusDispatchTarget.isEmpty()) {
            return Optional.empty();
        }
        Group localSearchGroup = (Group)this.groups().get((Object)this.localCorpusDispatchTarget.get().group());
        if (!localSearchGroup.hasSufficientCoverage()) {
            return Optional.empty();
        }
        if (this.localCorpusDispatchTarget.get().isWorking() == Boolean.FALSE) {
            return Optional.empty();
        }
        return this.localCorpusDispatchTarget;
    }

    private void updateWorkingState(Node node, boolean isWorking) {
        node.setWorking(isWorking);
        this.updateVipStatusOnNodeChange(node, isWorking);
    }

    @Override
    public void working(Node node) {
        this.updateWorkingState(node, true);
    }

    @Override
    public void failed(Node node) {
        this.updateWorkingState(node, false);
    }

    private void updateSufficientCoverage(Group group, boolean sufficientCoverage) {
        if (sufficientCoverage == group.hasSufficientCoverage()) {
            return;
        }
        group.setHasSufficientCoverage(sufficientCoverage);
        this.updateVipStatusOnCoverageChange(group, sufficientCoverage);
    }

    private void updateVipStatusOnNodeChange(Node node, boolean nodeIsWorking) {
        if (this.localCorpusDispatchTarget.isEmpty()) {
            if (this.hasInformationAboutAllNodes()) {
                this.setInRotationOnlyIf(this.hasWorkingNodes());
            }
        } else if (this.usesLocalCorpusIn(node) && (nodeIsWorking || this.size() > 1)) {
            this.setInRotationOnlyIf(nodeIsWorking);
        }
    }

    private void updateVipStatusOnCoverageChange(Group group, boolean sufficientCoverage) {
        if (!this.localCorpusDispatchTarget.isEmpty() && this.usesLocalCorpusIn(group)) {
            this.setInRotationOnlyIf(sufficientCoverage);
        }
    }

    private void setInRotationOnlyIf(boolean inRotation) {
        if (inRotation) {
            this.vipStatus.addToRotation(this.clusterId);
        } else {
            this.vipStatus.removeFromRotation(this.clusterId);
        }
    }

    public int estimateHitsToFetch(int wantedHits, int numPartitions) {
        return this.hitEstimator.estimateK(wantedHits, numPartitions);
    }

    public int estimateHitsToFetch(int wantedHits, int numPartitions, double topKProbability) {
        return this.hitEstimator.estimateK(wantedHits, numPartitions, topKProbability);
    }

    public boolean hasInformationAboutAllNodes() {
        return this.nodesByHost.values().stream().allMatch(node -> node.isWorking() != null);
    }

    private boolean hasWorkingNodes() {
        return this.nodesByHost.values().stream().anyMatch(node -> node.isWorking() != Boolean.FALSE);
    }

    private boolean usesLocalCorpusIn(Node node) {
        return this.localCorpusDispatchTarget.isPresent() && this.localCorpusDispatchTarget.get().equals(node);
    }

    private boolean usesLocalCorpusIn(Group group) {
        return this.localCorpusDispatchTarget.isPresent() && this.localCorpusDispatchTarget.get().group() == group.id();
    }

    @Override
    public void ping(ClusterMonitor clusterMonitor, Node node, Executor executor) {
        Pinger pinger = this.pingFactory.createPinger(node, clusterMonitor, new PongCallback(node, clusterMonitor));
        pinger.ping();
    }

    private void pingIterationCompletedSingleGroup() {
        Group group = (Group)this.groups().values().iterator().next();
        group.aggregateNodeValues();
        this.updateSufficientCoverage(group, true);
        boolean sufficientCoverage = this.isGroupCoverageSufficient(group.activeDocuments(), group.activeDocuments());
        this.trackGroupCoverageChanges(group, sufficientCoverage, group.activeDocuments());
    }

    private void pingIterationCompletedMultipleGroups() {
        this.orderedGroups().forEach(Group::aggregateNodeValues);
        long medianDocuments = this.medianDocumentsPerGroup();
        boolean anyGroupsSufficientCoverage = false;
        for (Group group : this.orderedGroups()) {
            boolean sufficientCoverage = this.isGroupCoverageSufficient(group.activeDocuments(), medianDocuments);
            anyGroupsSufficientCoverage = anyGroupsSufficientCoverage || sufficientCoverage;
            this.updateSufficientCoverage(group, sufficientCoverage);
            this.trackGroupCoverageChanges(group, sufficientCoverage, medianDocuments);
        }
    }

    private long medianDocumentsPerGroup() {
        if (this.orderedGroups().isEmpty()) {
            return 0L;
        }
        List activeDocuments = this.orderedGroups().stream().map(Group::activeDocuments).collect(Collectors.toList());
        return (long)Quantiles.median().compute(activeDocuments);
    }

    @Override
    public void pingIterationCompleted() {
        int numGroups = this.orderedGroups().size();
        if (numGroups == 1) {
            this.pingIterationCompletedSingleGroup();
        } else {
            this.pingIterationCompletedMultipleGroups();
        }
    }

    private boolean isGroupCoverageSufficient(long activeDocuments, long medianDocuments) {
        double documentCoverage = 100.0 * (double)activeDocuments / (double)medianDocuments;
        return medianDocuments <= 0L || !(documentCoverage < this.dispatchConfig.minActivedocsPercentage());
    }

    public boolean isPartialGroupCoverageSufficient(List<Node> nodes) {
        if (this.orderedGroups().size() == 1) {
            return true;
        }
        long activeDocuments = nodes.stream().mapToLong(Node::getActiveDocuments).sum();
        return this.isGroupCoverageSufficient(activeDocuments, this.medianDocumentsPerGroup());
    }

    private void trackGroupCoverageChanges(Group group, boolean fullCoverage, long medianDocuments) {
        if (!this.hasInformationAboutAllNodes()) {
            return;
        }
        boolean changed = group.fullCoverageStatusChanged(fullCoverage);
        if (changed || !fullCoverage && System.currentTimeMillis() > this.nextLogTime) {
            this.nextLogTime = System.currentTimeMillis() + 30000L;
            if (fullCoverage) {
                log.info("Cluster " + this.clusterId + ": " + group + " has full coverage. Active documents: " + group.activeDocuments() + "/" + medianDocuments + ", working nodes: " + group.workingNodes() + "/" + group.nodes().size());
            } else {
                StringBuilder unresponsive = new StringBuilder();
                for (Node node : group.nodes()) {
                    if (node.isWorking() == Boolean.TRUE) continue;
                    unresponsive.append('\n').append(node);
                }
                log.warning("Cluster " + this.clusterId + ": " + group + " has reduced coverage: Active documents: " + group.activeDocuments() + "/" + medianDocuments + ", working nodes: " + group.workingNodes() + "/" + group.nodes().size() + ", unresponsive nodes: " + (Serializable)((Object)(unresponsive.toString().isEmpty() ? " none" : unresponsive)));
            }
        }
    }

    private static class PongCallback
    implements PongHandler {
        private final ClusterMonitor<Node> clusterMonitor;
        private final Node node;

        PongCallback(Node node, ClusterMonitor<Node> clusterMonitor) {
            this.node = node;
            this.clusterMonitor = clusterMonitor;
        }

        @Override
        public void handle(Pong pong) {
            if (pong.badResponse()) {
                this.clusterMonitor.failed(this.node, pong.error().get());
            } else {
                if (pong.activeDocuments().isPresent()) {
                    this.node.setActiveDocuments(pong.activeDocuments().get());
                    this.node.setBlockingWrites(pong.isBlockingWrites());
                }
                this.clusterMonitor.responded(this.node);
            }
        }
    }
}

