/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.jdisc.http.server.jetty;

import com.yahoo.jdisc.http.ConnectorConfig;
import java.nio.channels.SelectableChannel;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.eclipse.jetty.io.Connection;
import org.eclipse.jetty.io.SelectorManager;
import org.eclipse.jetty.server.AbstractConnector;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.eclipse.jetty.util.component.AbstractLifeCycle;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.LifeCycle;
import org.eclipse.jetty.util.statistic.RateStatistic;
import org.eclipse.jetty.util.thread.Scheduler;

@ManagedObject(value="Monitor various resource constraints and throttles new connections once a threshold is exceeded")
class ConnectionThrottler
extends ContainerLifeCycle
implements SelectorManager.AcceptListener {
    private static final Logger log = Logger.getLogger(ConnectionThrottler.class.getName());
    private final Object monitor = new Object();
    private final Collection<ResourceLimit> resourceLimits = new ArrayList<ResourceLimit>();
    private final AbstractConnector connector;
    private final Duration idleTimeout;
    private final Scheduler scheduler;
    private boolean isRegistered = false;
    private boolean isThrottling = false;

    ConnectionThrottler(AbstractConnector connector, ConnectorConfig.Throttling config) {
        this(Runtime.getRuntime(), new RateStatistic(1L, TimeUnit.SECONDS), connector.getScheduler(), connector, config);
    }

    ConnectionThrottler(Runtime runtime, RateStatistic rateStatistic, Scheduler scheduler, AbstractConnector connector, ConnectorConfig.Throttling config) {
        this.connector = connector;
        if (config.maxHeapUtilization() != -1.0) {
            this.resourceLimits.add(new HeapResourceLimit(runtime, config.maxHeapUtilization()));
        }
        if (config.maxConnections() != -1) {
            this.resourceLimits.add(new ConnectionLimitThreshold(config.maxConnections()));
        }
        if (config.maxAcceptRate() != -1) {
            this.resourceLimits.add(new AcceptRateLimit(rateStatistic, config.maxAcceptRate()));
        }
        this.idleTimeout = config.idleTimeout() != -1.0 ? Duration.ofMillis((long)(config.idleTimeout() * 1000.0)) : null;
        this.scheduler = scheduler;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void registerWithConnector() {
        Object object = this.monitor;
        synchronized (object) {
            if (this.isRegistered) {
                return;
            }
            this.isRegistered = true;
            this.resourceLimits.forEach(arg_0 -> ((AbstractConnector)this.connector).addBean(arg_0));
            this.connector.addBean((Object)this);
        }
    }

    public void onAccepting(SelectableChannel channel) {
        this.throttleIfAnyThresholdIsExceeded();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void throttleIfAnyThresholdIsExceeded() {
        Object object = this.monitor;
        synchronized (object) {
            if (this.isThrottling) {
                return;
            }
            List<String> reasons = this.getThrottlingReasons();
            if (reasons.isEmpty()) {
                return;
            }
            log.warning(String.format("Throttling new connection. Reasons: %s", reasons));
            this.isThrottling = true;
            if (this.connector.isAccepting()) {
                this.connector.setAccepting(false);
            }
            if (this.idleTimeout != null) {
                log.warning(String.format("Applying idle timeout to existing connections: timeout=%sms", this.idleTimeout));
                this.connector.getConnectedEndPoints().forEach(endPoint -> endPoint.setIdleTimeout(this.idleTimeout.toMillis()));
            }
            this.scheduler.schedule(this::unthrottleIfBelowThresholds, 1L, TimeUnit.SECONDS);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void unthrottleIfBelowThresholds() {
        Object object = this.monitor;
        synchronized (object) {
            if (!this.isThrottling) {
                return;
            }
            List<String> reasons = this.getThrottlingReasons();
            if (!reasons.isEmpty()) {
                log.warning(String.format("Throttling continued. Reasons: %s", reasons));
                this.scheduler.schedule(this::unthrottleIfBelowThresholds, 1L, TimeUnit.SECONDS);
                return;
            }
            if (this.idleTimeout != null) {
                long originalTimeout = this.connector.getIdleTimeout();
                log.info(String.format("Reverting idle timeout for existing connections: timeout=%sms", originalTimeout));
                this.connector.getConnectedEndPoints().forEach(endPoint -> endPoint.setIdleTimeout(originalTimeout));
            }
            log.info("Throttling disabled - resource thresholds no longer exceeded");
            if (!this.connector.isAccepting()) {
                this.connector.setAccepting(true);
            }
            this.isThrottling = false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<String> getThrottlingReasons() {
        Object object = this.monitor;
        synchronized (object) {
            return this.resourceLimits.stream().map(ResourceLimit::isThresholdExceeded).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
        }
    }

    private static class HeapResourceLimit
    extends AbstractLifeCycle
    implements ResourceLimit {
        private final Runtime runtime;
        private final double maxHeapUtilization;

        HeapResourceLimit(Runtime runtime, double maxHeapUtilization) {
            this.runtime = runtime;
            this.maxHeapUtilization = maxHeapUtilization;
        }

        @Override
        public Optional<String> isThresholdExceeded() {
            double heapUtilization = (double)(this.runtime.maxMemory() - this.runtime.freeMemory()) / (double)this.runtime.maxMemory();
            if (heapUtilization > this.maxHeapUtilization) {
                return Optional.of(String.format("Max heap utilization exceeded: %f%%>%f%%", heapUtilization * 100.0, this.maxHeapUtilization * 100.0));
            }
            return Optional.empty();
        }
    }

    private static class ConnectionLimitThreshold
    extends AbstractLifeCycle
    implements ResourceLimit {
        private final Object monitor = new Object();
        private final int maxConnections;
        private final Set<SelectableChannel> connectionsAccepting = new HashSet<SelectableChannel>();
        private int connectionOpened;

        ConnectionLimitThreshold(int maxConnections) {
            this.maxConnections = maxConnections;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Optional<String> isThresholdExceeded() {
            Object object = this.monitor;
            synchronized (object) {
                int totalConnections = this.connectionOpened + this.connectionsAccepting.size();
                if (totalConnections > this.maxConnections) {
                    return Optional.of(String.format("Max connection exceeded: %d>%d", totalConnections, this.maxConnections));
                }
                return Optional.empty();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void onOpened(Connection connection) {
            Object object = this.monitor;
            synchronized (object) {
                this.connectionsAccepting.remove(connection.getEndPoint().getTransport());
                ++this.connectionOpened;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void onClosed(Connection connection) {
            Object object = this.monitor;
            synchronized (object) {
                --this.connectionOpened;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void onAccepting(SelectableChannel channel) {
            Object object = this.monitor;
            synchronized (object) {
                this.connectionsAccepting.add(channel);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void onAcceptFailed(SelectableChannel channel, Throwable cause) {
            Object object = this.monitor;
            synchronized (object) {
                this.connectionsAccepting.remove(channel);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void doStop() {
            Object object = this.monitor;
            synchronized (object) {
                this.connectionsAccepting.clear();
                this.connectionOpened = 0;
            }
        }
    }

    private static class AcceptRateLimit
    extends AbstractLifeCycle
    implements ResourceLimit {
        private final Object monitor = new Object();
        private final RateStatistic rateStatistic;
        private final int maxAcceptRate;

        AcceptRateLimit(RateStatistic rateStatistic, int maxAcceptRate) {
            this.rateStatistic = rateStatistic;
            this.maxAcceptRate = maxAcceptRate;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Optional<String> isThresholdExceeded() {
            Object object = this.monitor;
            synchronized (object) {
                int acceptRate = this.rateStatistic.getRate();
                if (acceptRate > this.maxAcceptRate) {
                    return Optional.of(String.format("Max accept rate exceeded: %d>%d", acceptRate, this.maxAcceptRate));
                }
                return Optional.empty();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void onAccepting(SelectableChannel channel) {
            Object object = this.monitor;
            synchronized (object) {
                this.rateStatistic.record();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void doStop() {
            Object object = this.monitor;
            synchronized (object) {
                this.rateStatistic.reset();
            }
        }
    }

    private static interface ResourceLimit
    extends LifeCycle,
    SelectorManager.AcceptListener,
    Connection.Listener {
        public Optional<String> isThresholdExceeded();

        default public void onOpened(Connection connection) {
        }

        default public void onClosed(Connection connection) {
        }
    }
}

