/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.container.handler;

import com.google.common.collect.Iterators;
import com.google.common.collect.UnmodifiableIterator;
import com.yahoo.vespa.defaults.Defaults;
import com.yahoo.yolean.Exceptions;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.PriorityQueue;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.GZIPInputStream;

class LogReader {
    static final Pattern logArchivePathPattern = Pattern.compile("(\\d{4})/(\\d{2})/(\\d{2})/(\\d{2})-\\d+(\\.gz|\\.zst)?");
    static final Pattern vespaLogPathPattern = Pattern.compile("vespa\\.log(?:-(\\d{4})-(\\d{2})-(\\d{2})\\.(\\d{2})-(\\d{2})-(\\d{2})(?:\\.gz|\\.zst)?)?");
    private final Path logDirectory;
    private final Pattern logFilePattern;

    LogReader(String logDirectory, String logFilePattern) {
        this(Paths.get(Defaults.getDefaults().underVespaHome(logDirectory), new String[0]), Pattern.compile(logFilePattern));
    }

    LogReader(Path logDirectory, Pattern logFilePattern) {
        this.logDirectory = logDirectory;
        this.logFilePattern = logFilePattern;
    }

    void writeLogs(OutputStream out, Instant from, Instant to, long maxLines, Optional<String> hostname) {
        double fromSeconds = (double)from.getEpochSecond() + (double)from.getNano() / 1.0E9;
        double toSeconds = (double)to.getEpochSecond() + (double)to.getNano() / 1.0E9;
        long linesWritten = 0L;
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(out));
        for (List<Path> logs : this.getMatchingFiles(from, to)) {
            ArrayList<LogLineIterator> logLineIterators = new ArrayList<LogLineIterator>();
            try {
                for (Path log : logs) {
                    logLineIterators.add(new LogLineIterator(log, fromSeconds, toSeconds, hostname));
                }
                UnmodifiableIterator lines = Iterators.mergeSorted(logLineIterators, Comparator.comparingDouble(LineWithTimestamp::timestamp));
                PriorityQueue<LineWithTimestamp> heap = new PriorityQueue<LineWithTimestamp>(Comparator.comparingDouble(LineWithTimestamp::timestamp));
                while (lines.hasNext()) {
                    heap.offer((LineWithTimestamp)lines.next());
                    if (heap.size() <= 1000) continue;
                    if (linesWritten++ >= maxLines) {
                        return;
                    }
                    writer.write(heap.poll().line);
                    writer.newLine();
                }
                while (!heap.isEmpty()) {
                    if (linesWritten++ >= maxLines) {
                        return;
                    }
                    writer.write(heap.poll().line);
                    writer.newLine();
                }
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            finally {
                for (LogLineIterator ll : logLineIterators) {
                    try {
                        ll.close();
                    }
                    catch (IOException iOException) {}
                }
                Exceptions.uncheck(writer::flush);
            }
        }
    }

    private List<List<Path>> getMatchingFiles(final Instant from, Instant to) {
        final ArrayList paths = new ArrayList();
        try {
            Files.walkFileTree(this.logDirectory, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    if (LogReader.this.logFilePattern.matcher(file.getFileName().toString()).matches() && !attrs.lastModifiedTime().toInstant().isBefore(from)) {
                        paths.add(file);
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        TreeMap logsByTimestamp = paths.stream().collect(Collectors.groupingBy(this::extractTimestamp, TreeMap::new, Collectors.toList()));
        ArrayList<List<Path>> sorted = new ArrayList<List<Path>>();
        for (Map.Entry entry : logsByTimestamp.entrySet()) {
            if (((Instant)entry.getKey()).isAfter(from)) {
                sorted.add((List)entry.getValue());
            }
            if (!((Instant)entry.getKey()).isAfter(to)) continue;
            break;
        }
        return sorted;
    }

    Instant extractTimestamp(Path path) {
        String relativePath = this.logDirectory.relativize(path).toString();
        Matcher matcher = logArchivePathPattern.matcher(relativePath);
        if (matcher.matches()) {
            return ZonedDateTime.of(Integer.parseInt(matcher.group(1)), Integer.parseInt(matcher.group(2)), Integer.parseInt(matcher.group(3)), Integer.parseInt(matcher.group(4)), 0, 0, 0, ZoneId.of("UTC")).toInstant().plus(Duration.ofHours(1L));
        }
        matcher = vespaLogPathPattern.matcher(relativePath);
        if (matcher.matches()) {
            if (matcher.group(1) == null) {
                return Instant.MAX;
            }
            return ZonedDateTime.of(Integer.parseInt(matcher.group(1)), Integer.parseInt(matcher.group(2)), Integer.parseInt(matcher.group(3)), Integer.parseInt(matcher.group(4)), Integer.parseInt(matcher.group(5)), Integer.parseInt(matcher.group(6)), 0, ZoneId.of("UTC")).toInstant().plus(Duration.ofSeconds(1L));
        }
        throw new IllegalArgumentException("Unrecognized file pattern for file at '" + path + "'");
    }

    private static class LogLineIterator
    implements Iterator<LineWithTimestamp>,
    AutoCloseable {
        private final BufferedReader reader;
        private final double from;
        private final double to;
        private final Optional<String> hostname;
        private LineWithTimestamp next;
        private Process zcat = null;

        private InputStream openFile(Path log) {
            boolean gzipped = log.toString().endsWith(".gz");
            boolean is_zstd = log.toString().endsWith(".zst");
            try {
                if (gzipped) {
                    InputStream in_gz = Files.newInputStream(log, new OpenOption[0]);
                    return new GZIPInputStream(in_gz);
                }
                if (is_zstd) {
                    ProcessBuilder pb = new ProcessBuilder("zstdcat", log.toString());
                    pb.redirectError(ProcessBuilder.Redirect.DISCARD);
                    this.zcat = pb.start();
                    this.zcat.getOutputStream().close();
                    return this.zcat.getInputStream();
                }
                try {
                    return Files.newInputStream(log, new OpenOption[0]);
                }
                catch (NoSuchFileException e) {
                    Path p = Paths.get(log + ".gz", new String[0]);
                    if (Files.exists(p, new LinkOption[0])) {
                        return this.openFile(p);
                    }
                    p = Paths.get(log + ".zst", new String[0]);
                    if (Files.exists(p, new LinkOption[0])) {
                        return this.openFile(p);
                    }
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
            return InputStream.nullInputStream();
        }

        private LogLineIterator(Path log, double from, double to, Optional<String> hostname) throws IOException {
            InputStream in = this.openFile(log);
            this.reader = new BufferedReader(new InputStreamReader(in, StandardCharsets.UTF_8));
            this.from = from;
            this.to = to;
            this.hostname = hostname;
            this.next = this.readNext();
        }

        @Override
        public boolean hasNext() {
            return this.next != null;
        }

        @Override
        public LineWithTimestamp next() {
            LineWithTimestamp current = this.next;
            this.next = this.readNext();
            return current;
        }

        @Override
        public void close() throws IOException {
            this.reader.close();
            if (this.zcat != null) {
                this.zcat.destroy();
            }
        }

        private LineWithTimestamp readNext() {
            try {
                String line;
                while ((line = this.reader.readLine()) != null) {
                    String[] parts = line.split("\t");
                    if (parts.length != 7 || this.hostname.map(host -> !host.equals(parts[1])).orElse(false).booleanValue()) continue;
                    double timestamp = Double.parseDouble(parts[0]);
                    if (timestamp > this.to) {
                        return null;
                    }
                    if (!(timestamp >= this.from)) continue;
                    return new LineWithTimestamp(line, timestamp);
                }
                return null;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
    }

    private static class LineWithTimestamp {
        final String line;
        final double timestamp;

        LineWithTimestamp(String line, double timestamp) {
            this.line = line;
            this.timestamp = timestamp;
        }

        String line() {
            return this.line;
        }

        double timestamp() {
            return this.timestamp;
        }
    }
}

