/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.filedistribution;

import com.yahoo.cloud.config.ConfigserverConfig;
import com.yahoo.component.annotation.Inject;
import com.yahoo.concurrent.DaemonThreadFactory;
import com.yahoo.config.FileReference;
import com.yahoo.config.subscription.ConfigSourceSet;
import com.yahoo.jrt.Int32Value;
import com.yahoo.jrt.Request;
import com.yahoo.jrt.StringValue;
import com.yahoo.jrt.Supervisor;
import com.yahoo.jrt.Transport;
import com.yahoo.jrt.Value;
import com.yahoo.vespa.config.ConnectionPool;
import com.yahoo.vespa.config.server.filedistribution.FileDirectory;
import com.yahoo.vespa.config.server.filedistribution.FileDistributionUtil;
import com.yahoo.vespa.filedistribution.EmptyFileReferenceData;
import com.yahoo.vespa.filedistribution.FileDistributionConnectionPool;
import com.yahoo.vespa.filedistribution.FileDownloader;
import com.yahoo.vespa.filedistribution.FileReferenceCompressor;
import com.yahoo.vespa.filedistribution.FileReferenceData;
import com.yahoo.vespa.filedistribution.FileReferenceDownload;
import com.yahoo.vespa.filedistribution.LazyFileReferenceData;
import com.yahoo.vespa.filedistribution.LazyTemporaryStorageFileReferenceData;
import com.yahoo.vespa.flags.FlagSource;
import com.yahoo.vespa.flags.Flags;
import com.yahoo.vespa.flags.ListFlag;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.Instant;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

public class FileServer {
    private static final Logger log = Logger.getLogger(FileServer.class.getName());
    private static final Duration timeout = Duration.ofSeconds(10L);
    private final FileDirectory fileDirectory;
    private final ExecutorService executor;
    private final FileDownloader downloader;
    private final List<FileReferenceData.CompressionType> compressionTypes;

    @Inject
    public FileServer(ConfigserverConfig configserverConfig, FlagSource flagSource, FileDirectory fileDirectory) {
        this(FileServer.createFileDownloader(FileDistributionUtil.getOtherConfigServersInCluster(configserverConfig), FileServer.compressionTypes(((ListFlag)Flags.FILE_DISTRIBUTION_ACCEPTED_COMPRESSION_TYPES.bindTo(flagSource)).value())), FileServer.compressionTypesAsList(((ListFlag)Flags.FILE_DISTRIBUTION_COMPRESSION_TYPES_TO_SERVE.bindTo(flagSource)).value()), fileDirectory);
    }

    public FileServer(FileDirectory fileDirectory) {
        this(FileServer.createFileDownloader(List.of(), Set.of(FileReferenceData.CompressionType.gzip)), List.of(FileReferenceData.CompressionType.gzip), fileDirectory);
    }

    FileServer(FileDownloader fileDownloader, List<FileReferenceData.CompressionType> compressionTypes, FileDirectory fileDirectory) {
        this.downloader = fileDownloader;
        this.fileDirectory = fileDirectory;
        this.executor = Executors.newFixedThreadPool(Math.max(8, Runtime.getRuntime().availableProcessors()), (ThreadFactory)new DaemonThreadFactory("file-server-"));
        this.compressionTypes = compressionTypes;
    }

    boolean hasFile(String fileReference) {
        return this.hasFile(new FileReference(fileReference));
    }

    private boolean hasFile(FileReference reference) {
        try {
            return this.fileDirectory.getFile(reference).exists();
        }
        catch (IllegalArgumentException e) {
            log.log(Level.FINE, () -> "Failed locating " + reference + ": " + e.getMessage());
            return false;
        }
    }

    FileDirectory getRootDir() {
        return this.fileDirectory;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void startFileServing(FileReference reference, Receiver target, Set<FileReferenceData.CompressionType> acceptedCompressionTypes) {
        if (!this.fileDirectory.getFile(reference).exists()) {
            return;
        }
        File file = this.fileDirectory.getFile(reference);
        log.log(Level.FINE, () -> "Start serving " + reference + " with file '" + file.getAbsolutePath() + "'");
        try (FileReferenceData fileData = EmptyFileReferenceData.empty((FileReference)reference, (String)file.getName());){
            fileData = this.readFileReferenceData(reference, acceptedCompressionTypes);
            target.receive(fileData, new ReplayStatus(0, "OK"));
            log.log(Level.FINE, () -> "Done serving " + reference.value() + " with file '" + file.getAbsolutePath() + "'");
        }
    }

    private FileReferenceData readFileReferenceData(FileReference reference, Set<FileReferenceData.CompressionType> acceptedCompressionTypes) throws IOException {
        File file = this.fileDirectory.getFile(reference);
        if (file.isDirectory()) {
            Path tempFile = Files.createTempFile("filereferencedata", reference.value(), new FileAttribute[0]);
            FileReferenceData.CompressionType compressionType = this.chooseCompressionType(acceptedCompressionTypes);
            log.log(Level.FINE, () -> "accepted compression types=" + acceptedCompressionTypes + ", compression type to use=" + compressionType);
            File compressedFile = new FileReferenceCompressor(FileReferenceData.Type.compressed, compressionType).compress(file.getParentFile(), tempFile.toFile());
            return new LazyTemporaryStorageFileReferenceData(reference, file.getName(), FileReferenceData.Type.compressed, compressedFile, compressionType);
        }
        return new LazyFileReferenceData(reference, file.getName(), FileReferenceData.Type.file, file, FileReferenceData.CompressionType.gzip);
    }

    public void serveFile(FileReference fileReference, boolean downloadFromOtherSourceIfNotFound, Set<FileReferenceData.CompressionType> acceptedCompressionTypes, Request request, Receiver receiver) {
        if (this.executor instanceof ThreadPoolExecutor) {
            log.log(Level.FINE, () -> "Active threads: " + ((ThreadPoolExecutor)this.executor).getActiveCount());
        }
        log.log(Level.FINE, () -> "Received request for file reference '" + fileReference + "' from " + request.target());
        Instant deadline = Instant.now().plus(timeout);
        String client = request.target().toString();
        this.executor.execute(() -> {
            FileApiErrorCodes result = this.serveFileInternal(fileReference, downloadFromOtherSourceIfNotFound, client, receiver, deadline, acceptedCompressionTypes);
            request.returnValues().add((Value)new Int32Value(result.getCode())).add((Value)new StringValue(result.getDescription()));
            request.returnRequest();
        });
    }

    private FileApiErrorCodes serveFileInternal(FileReference fileReference, boolean downloadFromOtherSourceIfNotFound, String client, Receiver receiver, Instant deadline, Set<FileReferenceData.CompressionType> acceptedCompressionTypes) {
        boolean fileExists;
        if (Instant.now().isAfter(deadline)) {
            log.log(Level.INFO, () -> "Deadline exceeded for request for file reference '" + fileReference + "' from " + client);
            return FileApiErrorCodes.TIMEOUT;
        }
        try {
            FileReferenceDownload fileReferenceDownload = new FileReferenceDownload(fileReference, client, downloadFromOtherSourceIfNotFound);
            fileExists = this.hasFileDownloadIfNeeded(fileReferenceDownload);
            if (fileExists) {
                this.startFileServing(fileReference, receiver, acceptedCompressionTypes);
            }
        }
        catch (IllegalArgumentException e) {
            fileExists = false;
            log.warning("Failed serving file reference '" + fileReference + "', request from " + client + " failed with: " + e.getMessage());
        }
        return fileExists ? FileApiErrorCodes.OK : FileApiErrorCodes.NOT_FOUND;
    }

    private FileReferenceData.CompressionType chooseCompressionType(Set<FileReferenceData.CompressionType> acceptedCompressionTypes) {
        for (FileReferenceData.CompressionType compressionType : this.compressionTypes) {
            if (!acceptedCompressionTypes.contains(compressionType)) continue;
            return compressionType;
        }
        throw new RuntimeException("Could not find a compression type that can be used. Accepted compression types: " + acceptedCompressionTypes + ", compression types server can use: " + this.compressionTypes);
    }

    boolean hasFileDownloadIfNeeded(FileReferenceDownload fileReferenceDownload) {
        FileReference fileReference = fileReferenceDownload.fileReference();
        if (this.hasFile(fileReference)) {
            return true;
        }
        if (fileReferenceDownload.downloadFromOtherSourceIfNotFound()) {
            log.log(Level.FINE, "File not found, downloading from another source");
            FileReferenceDownload newDownload = new FileReferenceDownload(fileReference, fileReferenceDownload.client(), false);
            boolean fileExists = this.downloader.getFile(newDownload).isPresent();
            if (!fileExists) {
                log.log(Level.INFO, "Failed downloading '" + fileReferenceDownload + "'");
            }
            return fileExists;
        }
        log.log(Level.FINE, "File not found, will not download from another source");
        return false;
    }

    public FileDownloader downloader() {
        return this.downloader;
    }

    public void close() {
        this.downloader.close();
        this.executor.shutdown();
    }

    private static FileDownloader createFileDownloader(List<String> configServers, Set<FileReferenceData.CompressionType> acceptedCompressionTypes) {
        Supervisor supervisor = new Supervisor(new Transport("filedistribution-pool")).setDropEmptyBuffers(true);
        return new FileDownloader(configServers.isEmpty() ? FileDownloader.emptyConnectionPool() : FileServer.createConnectionPool(configServers, supervisor), supervisor, timeout, acceptedCompressionTypes);
    }

    private static LinkedHashSet<FileReferenceData.CompressionType> compressionTypes(List<String> compressionTypes) {
        return compressionTypes.stream().map(FileReferenceData.CompressionType::valueOf).collect(Collectors.toCollection(LinkedHashSet::new));
    }

    private static List<FileReferenceData.CompressionType> compressionTypesAsList(List<String> compressionTypes) {
        return compressionTypes.stream().map(FileReferenceData.CompressionType::valueOf).collect(Collectors.toList());
    }

    private static ConnectionPool createConnectionPool(List<String> configServers, Supervisor supervisor) {
        ConfigSourceSet configSourceSet = new ConfigSourceSet(configServers);
        if (configServers.size() == 0) {
            return FileDownloader.emptyConnectionPool();
        }
        return new FileDistributionConnectionPool(configSourceSet, supervisor);
    }

    public static class ReplayStatus {
        private final int code;
        private final String description;

        ReplayStatus(int code, String description) {
            this.code = code;
            this.description = description;
        }

        public boolean ok() {
            return this.code == 0;
        }

        public int getCode() {
            return this.code;
        }

        public String getDescription() {
            return this.description;
        }
    }

    public static interface Receiver {
        public void receive(FileReferenceData var1, ReplayStatus var2);
    }

    private static enum FileApiErrorCodes {
        OK(0, "OK"),
        NOT_FOUND(1, "File reference not found"),
        TIMEOUT(2, "Timeout");

        private final int code;
        private final String description;

        private FileApiErrorCodes(int code, String description) {
            this.code = code;
            this.description = description;
        }

        int getCode() {
            return this.code;
        }

        String getDescription() {
            return this.description;
        }
    }
}

