/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.filedistribution;

import com.yahoo.config.FileReference;
import com.yahoo.io.IOUtils;
import com.yahoo.text.Utf8;
import com.yahoo.yolean.Exceptions;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.jpountz.xxhash.XXHash64;
import net.jpountz.xxhash.XXHashFactory;

public class FileDirectory {
    private static final Logger log = Logger.getLogger(FileDirectory.class.getName());
    private final File root;

    public FileDirectory(File rootDir) {
        this.root = rootDir;
        try {
            this.ensureRootExist();
        }
        catch (IllegalArgumentException e) {
            log.log(Level.WARNING, "Failed creating directory in constructor, will retry on demand : " + e.getMessage());
        }
    }

    private void ensureRootExist() {
        if (!this.root.exists()) {
            if (!this.root.mkdir()) {
                throw new IllegalArgumentException("Failed creating root dir '" + this.root.getAbsolutePath() + "'.");
            }
        } else if (!this.root.isDirectory()) {
            throw new IllegalArgumentException("'" + this.root.getAbsolutePath() + "' is not a directory");
        }
    }

    String getPath(FileReference ref) {
        return this.root.getAbsolutePath() + "/" + ref.value();
    }

    public File getFile(FileReference reference) {
        this.ensureRootExist();
        File dir = new File(this.getPath(reference));
        if (!dir.exists()) {
            throw new IllegalArgumentException("File reference '" + reference.value() + "' with absolute path '" + dir.getAbsolutePath() + "' does not exist.");
        }
        if (!dir.isDirectory()) {
            throw new IllegalArgumentException("File reference '" + reference.value() + "' with absolute path '" + dir.getAbsolutePath() + "' is not a directory.");
        }
        File[] files = dir.listFiles(new Filter());
        if (files == null || files.length == 0) {
            throw new IllegalArgumentException("File reference '" + reference.value() + "' with absolute path '" + dir.getAbsolutePath() + " does not contain any files");
        }
        return files[0];
    }

    File getRoot() {
        return this.root;
    }

    private Long computeHash(File file) throws IOException {
        XXHash64 hasher = XXHashFactory.fastestInstance().hash64();
        if (file.isDirectory()) {
            return Files.walk(file.toPath(), 100, new FileVisitOption[0]).map(path -> {
                try {
                    log.log(Level.FINEST, () -> "Calculating hash for '" + path + "'");
                    return this.hash(path.toFile(), hasher);
                }
                catch (IOException e) {
                    log.log(Level.WARNING, "Failed getting hash from '" + path + "'");
                    return 0;
                }
            }).mapToLong(rec$ -> ((Number)rec$).longValue()).sum();
        }
        return this.hash(file, hasher);
    }

    private long hash(File file, XXHash64 hasher) throws IOException {
        byte[] wholeFile = file.isDirectory() ? new byte[]{} : IOUtils.readFileBytes((File)file);
        return hasher.hash(ByteBuffer.wrap(wholeFile), hasher.hash(ByteBuffer.wrap(Utf8.toBytes((String)file.getName())), 0L));
    }

    public FileReference addFile(File source) throws IOException {
        Long hash = this.computeHash(source);
        FileReference fileReference = this.fileReferenceFromHash(hash);
        if (this.shouldAddFile(source, hash)) {
            return this.addFile(source, fileReference);
        }
        return fileReference;
    }

    private boolean shouldAddFile(File source, Long hashOfFileToBeAdded) throws IOException {
        FileReference fileReference = this.fileReferenceFromHash(hashOfFileToBeAdded);
        File destinationDir = this.destinationDir(fileReference);
        if (!destinationDir.exists()) {
            return true;
        }
        File existingFile = destinationDir.toPath().resolve(source.getName()).toFile();
        if (!existingFile.exists() || !this.computeHash(existingFile).equals(hashOfFileToBeAdded)) {
            log.log(Level.WARNING, "Directory for file reference '" + fileReference.value() + "' has content that does not match its hash, deleting everything in " + destinationDir.getAbsolutePath());
            IOUtils.recursiveDeleteDir((File)destinationDir);
            return true;
        }
        log.log(Level.FINE, "Directory for file reference '" + fileReference.value() + "' already exists and has all content");
        return false;
    }

    private File destinationDir(FileReference fileReference) {
        return new File(this.root, fileReference.value());
    }

    private FileReference fileReferenceFromHash(Long hash) {
        return new FileReference(Long.toHexString(hash));
    }

    private FileReference addFile(File source, FileReference reference) {
        this.ensureRootExist();
        Path tempDestinationDir = (Path)Exceptions.uncheck(() -> Files.createTempDirectory(this.root.toPath(), "writing", new FileAttribute[0]));
        try {
            this.logfileInfo(source);
            File destinationDir = this.destinationDir(reference);
            File tempDestination = new File(tempDestinationDir.toFile(), source.getName());
            if (!destinationDir.mkdir()) {
                log.log(Level.WARNING, () -> "destination dir " + destinationDir + " already exists");
            }
            log.log(Level.FINE, () -> "Copying " + source.getAbsolutePath() + " to " + tempDestination.getAbsolutePath());
            if (source.isDirectory()) {
                IOUtils.copyDirectory((File)source, (File)tempDestination, (int)-1);
            } else {
                FileDirectory.copyFile(source, tempDestination);
            }
            log.log(Level.FINE, () -> "Moving " + tempDestinationDir + " to " + destinationDir.getAbsolutePath());
            if (!tempDestinationDir.toFile().renameTo(destinationDir)) {
                log.log(Level.WARNING, "Failed moving '" + tempDestinationDir.toFile().getAbsolutePath() + "' to '" + tempDestination.getAbsolutePath() + "'.");
            }
            FileReference fileReference = reference;
            return fileReference;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        finally {
            IOUtils.recursiveDeleteDir((File)tempDestinationDir.toFile());
        }
    }

    private void logfileInfo(File file) throws IOException {
        BasicFileAttributes basicFileAttributes = Files.readAttributes(file.toPath(), BasicFileAttributes.class, new LinkOption[0]);
        log.log(Level.FINE, () -> "Adding file " + file.getAbsolutePath() + " (created " + basicFileAttributes.creationTime() + ", modified " + basicFileAttributes.lastModifiedTime() + ", size " + basicFileAttributes.size() + ")");
    }

    private static void copyFile(File source, File dest) throws IOException {
        try (FileChannel sourceChannel = new FileInputStream(source).getChannel();
             FileChannel destChannel = new FileOutputStream(dest).getChannel();){
            destChannel.transferFrom(sourceChannel, 0L, sourceChannel.size());
        }
    }

    public String toString() {
        return "root dir: " + this.root.getAbsolutePath();
    }

    private static class Filter
    implements FilenameFilter {
        private Filter() {
        }

        @Override
        public boolean accept(File dir, String name) {
            return !".".equals(name) && !"..".equals(name);
        }
    }
}

