/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.filedistribution;

import com.yahoo.cloud.config.ConfigserverConfig;
import com.yahoo.component.annotation.Inject;
import com.yahoo.concurrent.DaemonThreadFactory;
import com.yahoo.config.FileReference;
import com.yahoo.config.subscription.ConfigSourceSet;
import com.yahoo.jrt.Int32Value;
import com.yahoo.jrt.Request;
import com.yahoo.jrt.StringValue;
import com.yahoo.jrt.Supervisor;
import com.yahoo.jrt.Transport;
import com.yahoo.jrt.Value;
import com.yahoo.vespa.config.ConnectionPool;
import com.yahoo.vespa.config.server.filedistribution.FileDirectory;
import com.yahoo.vespa.config.server.filedistribution.FileDistributionUtil;
import com.yahoo.vespa.filedistribution.FileApiErrorCodes;
import com.yahoo.vespa.filedistribution.FileDistributionConnectionPool;
import com.yahoo.vespa.filedistribution.FileDownloader;
import com.yahoo.vespa.filedistribution.FileReferenceCompressor;
import com.yahoo.vespa.filedistribution.FileReferenceData;
import com.yahoo.vespa.filedistribution.FileReferenceDownload;
import com.yahoo.vespa.filedistribution.LazyFileReferenceData;
import com.yahoo.vespa.filedistribution.LazyTemporaryStorageFileReferenceData;
import com.yahoo.vespa.flags.FlagSource;
import com.yahoo.yolean.Exceptions;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;

public class FileServer {
    private static final Logger log = Logger.getLogger(FileServer.class.getName());
    private static final Duration timeout = Duration.ofSeconds(60L);
    private static final List<FileReferenceData.CompressionType> compressionTypesToServe = List.of(FileReferenceData.CompressionType.zstd, FileReferenceData.CompressionType.lz4, FileReferenceData.CompressionType.gzip, FileReferenceData.CompressionType.none);
    private static final String tempFilereferencedataPrefix = "filereferencedata";
    private static final Path tempFilereferencedataDir = Paths.get(System.getProperty("java.io.tmpdir"), new String[0]);
    private final FileDirectory fileDirectory;
    private final ThreadPoolExecutor executor;
    private final FileDownloader downloader;
    private final List<FileReferenceData.CompressionType> compressionTypes;

    @Inject
    public FileServer(ConfigserverConfig configserverConfig, FlagSource flagSource, FileDirectory fileDirectory) {
        this(FileServer.createFileDownloader(FileDistributionUtil.getOtherConfigServersInCluster(configserverConfig)), compressionTypesToServe, fileDirectory);
        try (Stream files = (Stream)Exceptions.uncheck(() -> Files.list(tempFilereferencedataDir));){
            files.filter(path -> path.toFile().isFile()).filter(path -> path.toFile().getName().startsWith(tempFilereferencedataPrefix)).forEach(path -> Exceptions.uncheck(() -> Files.delete(path)));
        }
    }

    FileServer(FileDownloader fileDownloader, List<FileReferenceData.CompressionType> compressionTypes, FileDirectory fileDirectory) {
        this.downloader = fileDownloader;
        this.fileDirectory = fileDirectory;
        this.executor = (ThreadPoolExecutor)Executors.newFixedThreadPool(Math.max(8, Runtime.getRuntime().availableProcessors()), (ThreadFactory)new DaemonThreadFactory("file-server-"));
        this.compressionTypes = compressionTypes;
    }

    boolean hasFile(String fileReference) {
        return this.hasFile(new FileReference(fileReference));
    }

    private boolean hasFile(FileReference reference) {
        Optional<File> file = this.fileDirectory.getFile(reference);
        if (file.isPresent()) {
            return file.get().exists();
        }
        log.log(Level.FINE, () -> "Failed locating " + String.valueOf(reference));
        return false;
    }

    FileDirectory getRootDir() {
        return this.fileDirectory;
    }

    void startFileServing(FileReference reference, File file, Receiver target, Set<FileReferenceData.CompressionType> acceptedCompressionTypes) {
        String absolutePath = file.getAbsolutePath();
        try (FileReferenceData fileData = this.fileReferenceData(reference, acceptedCompressionTypes, file);){
            log.log(Level.FINE, () -> "Start serving " + reference.value() + " with file '" + absolutePath + "'");
            target.receive(fileData, new ReplayStatus(0, "OK"));
            log.log(Level.FINE, () -> "Done serving " + reference.value() + " with file '" + absolutePath + "'");
        }
        catch (IOException ioe) {
            throw new UncheckedIOException("For " + reference.value() + ": failed reading file '" + absolutePath + "' for sending to '" + target.toString() + "'. ", ioe);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed serving " + reference.value() + " to '" + String.valueOf(target) + "': ", e);
        }
    }

    private FileReferenceData fileReferenceData(FileReference reference, Set<FileReferenceData.CompressionType> acceptedCompressionTypes, File file) throws IOException {
        FileReferenceData.CompressionType compressionType = this.chooseCompressionType(acceptedCompressionTypes);
        log.log(Level.FINE, () -> "accepted compression types: " + String.valueOf(acceptedCompressionTypes) + ", will use " + String.valueOf(compressionType));
        if (file.isDirectory()) {
            Path tempFile = Files.createTempFile(tempFilereferencedataDir, tempFilereferencedataPrefix, reference.value(), new FileAttribute[0]);
            Instant start = Instant.now();
            File compressedFile = new FileReferenceCompressor(FileReferenceData.Type.compressed, compressionType).compress(file.getParentFile(), tempFile.toFile());
            Duration duration = Duration.between(start, Instant.now());
            log.log(duration.compareTo(Duration.ofSeconds(10L)) > 0 ? Level.INFO : Level.FINE, () -> "compressed " + String.valueOf(reference) + " with " + String.valueOf(compressionType) + " in " + String.valueOf(Duration.between(start, Instant.now())));
            return new LazyTemporaryStorageFileReferenceData(reference, file.getName(), FileReferenceData.Type.compressed, compressedFile, compressionType);
        }
        return new LazyFileReferenceData(reference, file.getName(), FileReferenceData.Type.file, file, compressionType);
    }

    public void serveFile(FileReference fileReference, boolean downloadFromOtherSourceIfNotFound, Set<FileReferenceData.CompressionType> acceptedCompressionTypes, Request request, Receiver receiver) {
        log.log(Level.FINE, () -> "Received request for " + String.valueOf(fileReference) + " from " + request.target().peerSpec().host() + ", download from other source: " + downloadFromOtherSourceIfNotFound);
        String client = request.target().toString();
        log.log(Level.FINE, this.executor.getActiveCount() + " out of " + this.executor.getMaximumPoolSize() + " threads are active");
        this.executor.execute(() -> {
            FileApiErrorCodes result = this.serveFileInternal(fileReference, downloadFromOtherSourceIfNotFound, client, receiver, acceptedCompressionTypes);
            request.returnValues().add((Value)new Int32Value(result.code())).add((Value)new StringValue(result.description()));
            log.log(Level.FINE, () -> "Returning request for " + String.valueOf(fileReference) + " from " + String.valueOf(request.target()));
            request.returnRequest();
        });
    }

    private FileApiErrorCodes serveFileInternal(FileReference fileReference, boolean downloadFromOtherSourceIfNotFound, String client, Receiver receiver, Set<FileReferenceData.CompressionType> acceptedCompressionTypes) {
        try {
            FileReferenceDownload fileReferenceDownload = new FileReferenceDownload(fileReference, client, downloadFromOtherSourceIfNotFound);
            Optional<File> file = this.getFileDownloadIfNeeded(fileReferenceDownload);
            if (file.isEmpty()) {
                return FileApiErrorCodes.NOT_FOUND;
            }
            this.startFileServing(fileReference, file.get(), receiver, acceptedCompressionTypes);
        }
        catch (Exception e) {
            log.warning("Failed serving " + String.valueOf(fileReference) + ", request from " + client + " failed with: " + e.getMessage());
            return FileApiErrorCodes.TRANSFER_FAILED;
        }
        return FileApiErrorCodes.OK;
    }

    private FileReferenceData.CompressionType chooseCompressionType(Set<FileReferenceData.CompressionType> acceptedCompressionTypes) {
        for (FileReferenceData.CompressionType compressionType : this.compressionTypes) {
            if (!acceptedCompressionTypes.contains(compressionType)) continue;
            return compressionType;
        }
        throw new RuntimeException("Could not find a compression type that can be used. Accepted compression types: " + String.valueOf(acceptedCompressionTypes) + ", compression types server can use: " + String.valueOf(this.compressionTypes));
    }

    public Optional<File> getFileDownloadIfNeeded(FileReferenceDownload fileReferenceDownload) {
        FileReference fileReference = fileReferenceDownload.fileReference();
        Optional file = this.fileDirectory.getFile(fileReference);
        if (file.isPresent()) {
            return file;
        }
        if (fileReferenceDownload.downloadFromOtherSourceIfNotFound()) {
            log.log(Level.FINE, String.valueOf(fileReferenceDownload) + " not found, downloading from another source");
            FileReferenceDownload newDownload = new FileReferenceDownload(fileReference, fileReferenceDownload.client(), false);
            file = this.downloader.getFile(newDownload);
            if (file.isEmpty()) {
                log.log(Level.INFO, "Failed downloading '" + String.valueOf(fileReferenceDownload) + "'");
            }
            return file;
        }
        log.log(Level.FINE, "File not found, will not download from another source");
        return Optional.empty();
    }

    public FileDownloader downloader() {
        return this.downloader;
    }

    public void close() {
        this.downloader.close();
        this.executor.shutdown();
    }

    private static FileDownloader createFileDownloader(List<String> configServers) {
        Supervisor supervisor = new Supervisor(new Transport("filedistribution-pool")).setDropEmptyBuffers(true);
        return new FileDownloader(FileServer.createConnectionPool(configServers, supervisor), supervisor, timeout);
    }

    private static ConnectionPool createConnectionPool(List<String> configServers, Supervisor supervisor) {
        if (configServers.isEmpty()) {
            return FileDownloader.emptyConnectionPool();
        }
        return new FileDistributionConnectionPool(new ConfigSourceSet(configServers), supervisor);
    }

    public static class ReplayStatus {
        private final int code;
        private final String description;

        ReplayStatus(int code, String description) {
            this.code = code;
            this.description = description;
        }

        public boolean ok() {
            return this.code == 0;
        }

        public int getCode() {
            return this.code;
        }

        public String getDescription() {
            return this.description;
        }
    }

    public static interface Receiver {
        public void receive(FileReferenceData var1, ReplayStatus var2);
    }
}

