/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server;

import com.yahoo.cloud.config.ConfigserverConfig;
import com.yahoo.component.AbstractComponent;
import com.yahoo.component.annotation.Inject;
import com.yahoo.concurrent.DaemonThreadFactory;
import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.Deployment;
import com.yahoo.config.provision.TransientException;
import com.yahoo.container.handler.VipStatus;
import com.yahoo.container.jdisc.state.StateMonitor;
import com.yahoo.vespa.config.server.ApplicationRepository;
import com.yahoo.vespa.config.server.filedistribution.FileDirectory;
import com.yahoo.vespa.config.server.maintenance.ConfigServerMaintenance;
import com.yahoo.vespa.config.server.rpc.RpcServer;
import com.yahoo.vespa.config.server.version.VersionState;
import com.yahoo.yolean.Exceptions;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;

public class ConfigServerBootstrap
extends AbstractComponent
implements Runnable {
    private static final Logger log = Logger.getLogger(ConfigServerBootstrap.class.getName());
    private final ApplicationRepository applicationRepository;
    private final RpcServer server;
    private final VersionState versionState;
    private final StateMonitor stateMonitor;
    private final VipStatus vipStatus;
    private final ConfigserverConfig configserverConfig;
    private final Duration maxDurationOfRedeployment;
    private final Duration sleepTimeWhenRedeployingFails;
    private final RedeployingApplicationsFails exitIfRedeployingApplicationsFails;
    private final ExecutorService rpcServerExecutor;
    private final ConfigServerMaintenance configServerMaintenance;
    private final Clock clock;

    @Inject
    public ConfigServerBootstrap(ApplicationRepository applicationRepository, RpcServer server, VersionState versionState, StateMonitor stateMonitor, VipStatus vipStatus, FileDirectory fileDirectory) {
        this(applicationRepository, server, versionState, stateMonitor, vipStatus, RedeployingApplicationsFails.EXIT_JVM, ConfigServerBootstrap.vipStatusMode(applicationRepository), fileDirectory);
    }

    protected ConfigServerBootstrap(ApplicationRepository applicationRepository, RpcServer server, VersionState versionState, StateMonitor stateMonitor, VipStatus vipStatus, RedeployingApplicationsFails exitIfRedeployingApplicationsFails, VipStatusMode vipStatusMode, FileDirectory fileDirectory) {
        this.applicationRepository = applicationRepository;
        this.server = server;
        this.versionState = versionState;
        this.stateMonitor = stateMonitor;
        this.vipStatus = vipStatus;
        this.configserverConfig = applicationRepository.configserverConfig();
        this.maxDurationOfRedeployment = Duration.ofSeconds(this.configserverConfig.maxDurationOfBootstrap());
        this.sleepTimeWhenRedeployingFails = Duration.ofSeconds(this.configserverConfig.sleepTimeWhenRedeployingFails());
        this.exitIfRedeployingApplicationsFails = exitIfRedeployingApplicationsFails;
        this.clock = applicationRepository.clock();
        this.rpcServerExecutor = Executors.newSingleThreadExecutor((ThreadFactory)new DaemonThreadFactory("config server RPC server"));
        this.configServerMaintenance = new ConfigServerMaintenance(applicationRepository, fileDirectory);
        this.configServerMaintenance.startBeforeBootstrap();
        log.log(Level.FINE, () -> "VIP status mode: " + vipStatusMode);
        this.initializing(vipStatusMode);
        this.start();
    }

    public void deconstruct() {
        log.log(Level.INFO, "Stopping config server");
        this.down();
        this.server.stop();
        log.log(Level.FINE, "RPC server stopped");
        this.rpcServerExecutor.shutdown();
        this.configServerMaintenance.shutdown();
    }

    @Override
    public void run() {
        this.start();
        do {
            try {
                Thread.sleep(1000L);
            }
            catch (InterruptedException e) {
                log.log(Level.SEVERE, "Got interrupted", e);
                break;
            }
        } while (this.server.isRunning());
        this.down();
    }

    public void start() {
        this.startRpcServerWithFileDistribution();
        if (this.versionState.isUpgraded()) {
            log.log(Level.INFO, "Config server upgrading from " + this.versionState.storedVersion() + " to " + this.versionState.currentVersion() + ". Redeploying all applications");
            try {
                this.redeployAllApplications();
                this.versionState.saveNewVersion();
                log.log(Level.INFO, "All applications redeployed successfully");
            }
            catch (Exception e) {
                log.log(Level.SEVERE, "Redeployment of applications failed", e);
                this.redeployingApplicationsFailed();
                return;
            }
        }
        this.applicationRepository.bootstrappingDone();
        this.allowConfigRpcRequests(this.server);
        this.up();
        this.configServerMaintenance.startAfterBootstrap();
    }

    StateMonitor.Status status() {
        return this.stateMonitor.status();
    }

    VipStatus vipStatus() {
        return this.vipStatus;
    }

    public ConfigServerMaintenance configServerMaintenance() {
        return this.configServerMaintenance;
    }

    private void up() {
        this.vipStatus.setInRotation(Boolean.valueOf(true));
    }

    private void down() {
        this.vipStatus.setInRotation(Boolean.valueOf(false));
    }

    private void initializing(VipStatusMode vipStatusMode) {
        this.stateMonitor.status(StateMonitor.Status.initializing);
        if (vipStatusMode == VipStatusMode.VIP_STATUS_PROGRAMMATICALLY) {
            this.vipStatus.setInRotation(Boolean.valueOf(false));
        }
    }

    private void startRpcServerWithFileDistribution() {
        this.rpcServerExecutor.execute(this.server);
        Instant end = this.clock.instant().plus(Duration.ofSeconds(10L));
        while (!this.server.isRunning() && this.clock.instant().isBefore(end)) {
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException e) {
                log.log(Level.SEVERE, "Got interrupted", e);
                break;
            }
        }
        if (!this.server.isRunning()) {
            throw new RuntimeException("RPC server not started in 10 seconds");
        }
    }

    private void allowConfigRpcRequests(RpcServer rpcServer) {
        log.log(Level.FINE, "Allowing RPC config requests");
        rpcServer.setUpGetConfigHandlers();
    }

    private void redeployingApplicationsFailed() {
        if (this.exitIfRedeployingApplicationsFails == RedeployingApplicationsFails.EXIT_JVM) {
            System.exit(1);
        }
    }

    private void redeployAllApplications() throws InterruptedException {
        Instant end = this.clock.instant().plus(this.maxDurationOfRedeployment);
        List<ApplicationId> applicationsToRedeploy = new ArrayList<ApplicationId>(this.applicationRepository.listApplications());
        Collections.shuffle(applicationsToRedeploy);
        long failCount = 0L;
        do {
            Duration sleepTime;
            if ((applicationsToRedeploy = this.redeployApplications(applicationsToRedeploy)).isEmpty() || this.sleepTimeWhenRedeployingFails.isZero()) continue;
            if ((sleepTime = this.sleepTimeWhenRedeployingFails.multipliedBy(++failCount)).compareTo(Duration.ofMinutes(10L)) > 0) {
                sleepTime = Duration.ofMinutes(10L);
            }
            log.log(Level.INFO, "Redeployment of " + applicationsToRedeploy + " not finished, will retry in " + sleepTime);
            Thread.sleep(sleepTime.toMillis());
        } while (!applicationsToRedeploy.isEmpty() && this.clock.instant().isBefore(end));
        if (!applicationsToRedeploy.isEmpty()) {
            throw new RuntimeException("Redeploying applications not finished after " + this.maxDurationOfRedeployment + ", exiting, applications that failed redeployment: " + applicationsToRedeploy);
        }
    }

    private List<ApplicationId> redeployApplications(List<ApplicationId> applicationIds) throws InterruptedException {
        ExecutorService executor = Executors.newFixedThreadPool(this.configserverConfig.numRedeploymentThreads(), (ThreadFactory)new DaemonThreadFactory("redeploy-apps-"));
        HashMap deployments = new HashMap();
        log.log(Level.INFO, () -> "Redeploying " + applicationIds.size() + " apps " + applicationIds + " with " + this.configserverConfig.numRedeploymentThreads() + " threads");
        applicationIds.forEach(appId -> deployments.put((ApplicationId)appId, executor.submit(() -> {
            log.log(Level.INFO, () -> "Starting redeployment of " + appId);
            this.applicationRepository.deployFromLocalActive((ApplicationId)appId, true).ifPresent(Deployment::activate);
            log.log(Level.INFO, () -> appId + " redeployed");
        })));
        List<ApplicationId> failedDeployments = this.checkDeployments(deployments);
        executor.shutdown();
        if (!executor.awaitTermination(5L, TimeUnit.HOURS)) {
            log.log(Level.SEVERE, () -> "Unable to shutdown " + executor + ", waited 5 hours. Exiting");
            System.exit(1);
        }
        return failedDeployments;
    }

    private List<ApplicationId> checkDeployments(Map<ApplicationId, Future<?>> deployments) {
        int applicationCount = deployments.size();
        LinkedHashSet failedDeployments = new LinkedHashSet();
        LinkedHashSet finishedDeployments = new LinkedHashSet();
        LogState logState = new LogState(applicationCount);
        do {
            deployments.forEach((applicationId, future) -> {
                if (finishedDeployments.contains(applicationId) || failedDeployments.contains(applicationId)) {
                    return;
                }
                DeploymentStatus status = this.getDeploymentStatus((ApplicationId)applicationId, (Future<?>)future);
                switch (status) {
                    case done: {
                        finishedDeployments.add(applicationId);
                        break;
                    }
                    case inProgress: {
                        break;
                    }
                    case failed: {
                        failedDeployments.add(applicationId);
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("Unknown deployment status " + status);
                    }
                }
            });
            this.logProgress(logState, failedDeployments.size(), finishedDeployments.size());
        } while (failedDeployments.size() + finishedDeployments.size() < applicationCount);
        return new ArrayList<ApplicationId>(failedDeployments);
    }

    private void logProgress(LogState logState, int failedDeployments, int finishedDeployments) {
        if (!(Duration.between(logState.lastLogged, this.clock.instant()).minus(Duration.ofSeconds(10L)).isNegative() || logState.failedDeployments == failedDeployments && logState.finishedDeployments == finishedDeployments)) {
            log.log(Level.INFO, () -> finishedDeployments + " of " + logState.applicationCount + " apps redeployed (" + failedDeployments + " failed)");
            logState.update(this.clock.instant(), failedDeployments, finishedDeployments);
        }
    }

    private DeploymentStatus getDeploymentStatus(ApplicationId applicationId, Future<?> future) {
        try {
            future.get(1L, TimeUnit.MILLISECONDS);
            return DeploymentStatus.done;
        }
        catch (InterruptedException | ExecutionException e) {
            if (e.getCause() instanceof TransientException) {
                log.log(Level.INFO, "Redeploying " + applicationId + " failed with transient error, will retry after bootstrap: " + Exceptions.toMessageString((Throwable)e));
            } else {
                log.log(Level.WARNING, "Redeploying " + applicationId + " failed, will retry", e);
            }
            return DeploymentStatus.failed;
        }
        catch (TimeoutException e) {
            return DeploymentStatus.inProgress;
        }
    }

    private static VipStatusMode vipStatusMode(ApplicationRepository applicationRepository) {
        return applicationRepository.configserverConfig().hostedVespa() ? VipStatusMode.VIP_STATUS_FILE : VipStatusMode.VIP_STATUS_PROGRAMMATICALLY;
    }

    static enum RedeployingApplicationsFails {
        EXIT_JVM,
        CONTINUE;

    }

    static enum VipStatusMode {
        VIP_STATUS_FILE,
        VIP_STATUS_PROGRAMMATICALLY;

    }

    private static class LogState {
        private final int applicationCount;
        private Instant lastLogged = Instant.EPOCH;
        private int failedDeployments = 0;
        private int finishedDeployments = 0;

        public LogState(int applicationCount) {
            this.applicationCount = applicationCount;
        }

        public void update(Instant lastLogged, int failedDeployments, int finishedDeployments) {
            this.lastLogged = lastLogged;
            this.failedDeployments = failedDeployments;
            this.finishedDeployments = finishedDeployments;
        }
    }

    private static enum DeploymentStatus {
        inProgress,
        done,
        failed;

    }
}

