/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.filedistribution;

import com.yahoo.config.FileReference;
import com.yahoo.io.IOUtils;
import com.yahoo.text.Utf8;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.time.Instant;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.jpountz.xxhash.XXHash64;
import net.jpountz.xxhash.XXHashFactory;

public class FileDirectory {
    private static final Logger log = Logger.getLogger(FileDirectory.class.getName());
    private final File root;

    public FileDirectory(File rootDir) {
        this.root = rootDir;
        try {
            this.ensureRootExist();
        }
        catch (IllegalArgumentException e) {
            log.log(Level.WARNING, "Failed creating directory in constructor, will retry on demand : " + e.getMessage());
        }
    }

    private void ensureRootExist() {
        if (!this.root.exists()) {
            if (!this.root.mkdir()) {
                throw new IllegalArgumentException("Failed creating root dir '" + this.root.getAbsolutePath() + "'.");
            }
        } else if (!this.root.isDirectory()) {
            throw new IllegalArgumentException("'" + this.root.getAbsolutePath() + "' is not a directory");
        }
    }

    String getPath(FileReference ref) {
        return this.root.getAbsolutePath() + "/" + ref.value();
    }

    public File getFile(FileReference reference) {
        this.ensureRootExist();
        File dir = new File(this.getPath(reference));
        if (!dir.exists()) {
            throw new IllegalArgumentException("File reference '" + reference.value() + "' with absolute path '" + dir.getAbsolutePath() + "' does not exist.");
        }
        if (!dir.isDirectory()) {
            throw new IllegalArgumentException("File reference '" + reference.value() + "' with absolute path '" + dir.getAbsolutePath() + "' is not a directory.");
        }
        File[] files = dir.listFiles(new Filter());
        if (files == null || files.length == 0) {
            throw new IllegalArgumentException("File reference '" + reference.value() + "' with absolute path '" + dir.getAbsolutePath() + " does not contain any files");
        }
        return files[0];
    }

    File getRoot() {
        return this.root;
    }

    private Long computeHash(File file) throws IOException {
        XXHash64 hasher = XXHashFactory.fastestInstance().hash64();
        if (file.isDirectory()) {
            return Files.walk(file.toPath(), 100, new FileVisitOption[0]).map(path -> {
                try {
                    log.log(Level.FINE, () -> "Calculating hash for '" + path + "'");
                    return this.hash(path.toFile(), hasher);
                }
                catch (IOException e) {
                    log.log(Level.WARNING, "Failed getting hash from '" + path + "'");
                    return 0;
                }
            }).mapToLong(rec$ -> ((Number)rec$).longValue()).sum();
        }
        return this.hash(file, hasher);
    }

    private long hash(File file, XXHash64 hasher) throws IOException {
        byte[] wholeFile = file.isDirectory() ? new byte[]{} : IOUtils.readFileBytes((File)file);
        return hasher.hash(ByteBuffer.wrap(wholeFile), hasher.hash(ByteBuffer.wrap(Utf8.toBytes((String)file.getName())), 0L));
    }

    public FileReference addFile(File source) throws IOException {
        Long hash = this.computeHash(source);
        this.verifyExistingFile(source, hash);
        FileReference fileReference = this.fileReferenceFromHash(hash);
        return this.addFile(source, fileReference);
    }

    private void verifyExistingFile(File source, Long hashOfFileToBeAdded) throws IOException {
        FileReference fileReference = this.fileReferenceFromHash(hashOfFileToBeAdded);
        File destinationDir = this.destinationDir(fileReference);
        if (!destinationDir.exists()) {
            return;
        }
        File existingFile = destinationDir.toPath().resolve(source.getName()).toFile();
        if (!existingFile.exists() || !this.computeHash(existingFile).equals(hashOfFileToBeAdded)) {
            log.log(Level.SEVERE, "Directory for file reference '" + fileReference.value() + "' has content that does not match its hash, deleting everything in " + destinationDir.getAbsolutePath());
            IOUtils.recursiveDeleteDir((File)destinationDir);
        } else {
            FileTime fileTime = FileTime.from(Instant.now());
            Files.setAttribute(destinationDir.toPath(), "basic:lastAccessTime", fileTime, new LinkOption[0]);
        }
    }

    private File destinationDir(FileReference fileReference) {
        return new File(this.root, fileReference.value());
    }

    private FileReference fileReferenceFromHash(Long hash) {
        return new FileReference(Long.toHexString(hash));
    }

    FileReference addFile(File source, FileReference reference) {
        this.ensureRootExist();
        try {
            this.logfileInfo(source);
            File destinationDir = this.destinationDir(reference);
            Path tempDestinationDir = Files.createTempDirectory(this.root.toPath(), "writing", new FileAttribute[0]);
            File destination = new File(tempDestinationDir.toFile(), source.getName());
            if (!destinationDir.exists()) {
                destinationDir.mkdir();
                log.log(Level.FINE, () -> "file reference ' " + reference.value() + "', source: " + source.getAbsolutePath());
                if (source.isDirectory()) {
                    log.log(Level.FINE, () -> "Copying source " + source.getAbsolutePath() + " to " + destination.getAbsolutePath());
                    IOUtils.copyDirectory((File)source, (File)destination, (int)-1);
                } else {
                    FileDirectory.copyFile(source, destination);
                }
                if (!destinationDir.exists()) {
                    log.log(Level.FINE, () -> "Moving from " + tempDestinationDir + " to " + destinationDir.getAbsolutePath());
                    if (!tempDestinationDir.toFile().renameTo(destinationDir)) {
                        log.log(Level.WARNING, "Failed moving '" + tempDestinationDir.toFile().getAbsolutePath() + "' to '" + destination.getAbsolutePath() + "'.");
                    }
                } else {
                    IOUtils.copyDirectory((File)tempDestinationDir.toFile(), (File)destinationDir, (int)-1);
                }
            }
            IOUtils.recursiveDeleteDir((File)tempDestinationDir.toFile());
            return reference;
        }
        catch (IOException e) {
            throw new IllegalArgumentException(e);
        }
    }

    private void logfileInfo(File file) throws IOException {
        BasicFileAttributes basicFileAttributes = Files.readAttributes(file.toPath(), BasicFileAttributes.class, new LinkOption[0]);
        log.log(Level.FINE, () -> "Adding file " + file.getAbsolutePath() + " (created " + basicFileAttributes.creationTime() + ", modified " + basicFileAttributes.lastModifiedTime() + ", size " + basicFileAttributes.size() + ")");
    }

    private static void copyFile(File source, File dest) throws IOException {
        try (FileChannel sourceChannel = new FileInputStream(source).getChannel();
             FileChannel destChannel = new FileOutputStream(dest).getChannel();){
            destChannel.transferFrom(sourceChannel, 0L, sourceChannel.size());
        }
    }

    public String toString() {
        return "root dir: " + this.root.getAbsolutePath();
    }

    private static class Filter
    implements FilenameFilter {
        private Filter() {
        }

        @Override
        public boolean accept(File dir, String name) {
            return !".".equals(name) && !"..".equals(name);
        }
    }
}

