/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.tenant;

import com.google.common.collect.ImmutableSet;
import com.google.inject.Inject;
import com.yahoo.cloud.config.ConfigserverConfig;
import com.yahoo.concurrent.DaemonThreadFactory;
import com.yahoo.concurrent.Lock;
import com.yahoo.concurrent.Locks;
import com.yahoo.concurrent.StripedExecutor;
import com.yahoo.concurrent.ThreadFactoryFactory;
import com.yahoo.config.model.api.ConfigDefinitionRepo;
import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.TenantName;
import com.yahoo.config.provision.Zone;
import com.yahoo.container.jdisc.secretstore.SecretStore;
import com.yahoo.path.Path;
import com.yahoo.text.Utf8;
import com.yahoo.transaction.Transaction;
import com.yahoo.vespa.config.server.ConfigServerDB;
import com.yahoo.vespa.config.server.ReloadListener;
import com.yahoo.vespa.config.server.application.PermanentApplicationPackage;
import com.yahoo.vespa.config.server.application.TenantApplications;
import com.yahoo.vespa.config.server.deploy.TenantFileSystemDirs;
import com.yahoo.vespa.config.server.filedistribution.FileDistributionFactory;
import com.yahoo.vespa.config.server.host.HostRegistry;
import com.yahoo.vespa.config.server.modelfactory.ModelFactoryRegistry;
import com.yahoo.vespa.config.server.monitoring.MetricUpdater;
import com.yahoo.vespa.config.server.monitoring.Metrics;
import com.yahoo.vespa.config.server.provision.HostProvisionerProvider;
import com.yahoo.vespa.config.server.session.SessionPreparer;
import com.yahoo.vespa.config.server.session.SessionRepository;
import com.yahoo.vespa.config.server.tenant.Tenant;
import com.yahoo.vespa.config.server.tenant.TenantListener;
import com.yahoo.vespa.config.server.tenant.TenantMetaData;
import com.yahoo.vespa.config.server.zookeeper.ConfigCurator;
import com.yahoo.vespa.curator.Curator;
import com.yahoo.vespa.curator.transaction.CuratorOperations;
import com.yahoo.vespa.curator.transaction.CuratorTransaction;
import com.yahoo.vespa.flags.FlagSource;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.data.Stat;

public class TenantRepository {
    public static final TenantName HOSTED_VESPA_TENANT = TenantName.from((String)"hosted-vespa");
    private static final TenantName DEFAULT_TENANT = TenantName.defaultName();
    private static final Path tenantsPath = Path.fromString((String)"/config/v2/tenants/");
    private static final Path locksPath = Path.fromString((String)"/config/v2/locks/");
    private static final Path barriersPath = Path.fromString((String)"/config/v2/barriers/");
    private static final Path vespaPath = Path.fromString((String)"/vespa");
    private static final Duration checkForRemovedApplicationsInterval = Duration.ofMinutes(1L);
    private static final Logger log = Logger.getLogger(TenantRepository.class.getName());
    private final Map<TenantName, Tenant> tenants = Collections.synchronizedMap(new LinkedHashMap());
    private final Locks<TenantName> tenantLocks = new Locks(1L, TimeUnit.MINUTES);
    private final HostRegistry hostRegistry;
    private final TenantListener tenantListener;
    private final ConfigCurator configCurator;
    private final Curator curator;
    private final Metrics metrics;
    private final MetricUpdater metricUpdater;
    private final ExecutorService zkCacheExecutor;
    private final StripedExecutor<TenantName> zkWatcherExecutor;
    private final FileDistributionFactory fileDistributionFactory;
    private final FlagSource flagSource;
    private final SecretStore secretStore;
    private final HostProvisionerProvider hostProvisionerProvider;
    private final ConfigserverConfig configserverConfig;
    private final ConfigServerDB configServerDB;
    private final Zone zone;
    private final Clock clock;
    private final ModelFactoryRegistry modelFactoryRegistry;
    private final ConfigDefinitionRepo configDefinitionRepo;
    private final ReloadListener reloadListener;
    private final ExecutorService bootstrapExecutor;
    private final ScheduledExecutorService checkForRemovedApplicationsService = new ScheduledThreadPoolExecutor(1, (ThreadFactory)new DaemonThreadFactory("check for removed applications"));
    private final Optional<Curator.DirectoryCache> directoryCache;

    @Inject
    public TenantRepository(HostRegistry hostRegistry, ConfigCurator configCurator, Metrics metrics, FlagSource flagSource, SecretStore secretStore, HostProvisionerProvider hostProvisionerProvider, ConfigserverConfig configserverConfig, ConfigServerDB configServerDB, Zone zone, ModelFactoryRegistry modelFactoryRegistry, ConfigDefinitionRepo configDefinitionRepo, ReloadListener reloadListener, TenantListener tenantListener) {
        this(hostRegistry, configCurator, metrics, (StripedExecutor<TenantName>)new StripedExecutor(), new FileDistributionFactory(configserverConfig), flagSource, Executors.newFixedThreadPool(1, ThreadFactoryFactory.getThreadFactory((String)TenantRepository.class.getName())), secretStore, hostProvisionerProvider, configserverConfig, configServerDB, zone, Clock.systemUTC(), modelFactoryRegistry, configDefinitionRepo, reloadListener, tenantListener);
    }

    public TenantRepository(HostRegistry hostRegistry, ConfigCurator configCurator, Metrics metrics, StripedExecutor<TenantName> zkWatcherExecutor, FileDistributionFactory fileDistributionFactory, FlagSource flagSource, ExecutorService zkCacheExecutor, SecretStore secretStore, HostProvisionerProvider hostProvisionerProvider, ConfigserverConfig configserverConfig, ConfigServerDB configServerDB, Zone zone, Clock clock, ModelFactoryRegistry modelFactoryRegistry, ConfigDefinitionRepo configDefinitionRepo, ReloadListener reloadListener, TenantListener tenantListener) {
        this.hostRegistry = hostRegistry;
        this.configserverConfig = configserverConfig;
        this.bootstrapExecutor = Executors.newFixedThreadPool(configserverConfig.numParallelTenantLoaders(), (ThreadFactory)new DaemonThreadFactory("bootstrap-tenant-"));
        this.curator = configCurator.curator();
        this.metrics = metrics;
        this.metricUpdater = metrics.getOrCreateMetricUpdater(Collections.emptyMap());
        this.zkCacheExecutor = zkCacheExecutor;
        this.zkWatcherExecutor = zkWatcherExecutor;
        this.fileDistributionFactory = fileDistributionFactory;
        this.flagSource = flagSource;
        this.secretStore = secretStore;
        this.hostProvisionerProvider = hostProvisionerProvider;
        this.configServerDB = configServerDB;
        this.zone = zone;
        this.clock = clock;
        this.modelFactoryRegistry = modelFactoryRegistry;
        this.configDefinitionRepo = configDefinitionRepo;
        this.reloadListener = reloadListener;
        this.tenantListener = tenantListener;
        this.configCurator = configCurator;
        this.curator.framework().getConnectionStateListenable().addListener(this::stateChanged);
        this.curator.create(tenantsPath);
        this.curator.create(locksPath);
        this.curator.create(barriersPath);
        this.createSystemTenants(configserverConfig);
        this.curator.create(vespaPath);
        this.directoryCache = Optional.of(this.curator.createDirectoryCache(tenantsPath.getAbsolute(), false, false, zkCacheExecutor));
        this.directoryCache.get().addListener(this::childEvent);
        this.directoryCache.get().start();
        this.bootstrapTenants();
        this.notifyTenantsLoaded();
        this.checkForRemovedApplicationsService.scheduleWithFixedDelay(this::removeUnusedApplications, checkForRemovedApplicationsInterval.getSeconds(), checkForRemovedApplicationsInterval.getSeconds(), TimeUnit.SECONDS);
    }

    private void notifyTenantsLoaded() {
        this.tenantListener.onTenantsLoaded();
    }

    public Tenant addTenant(TenantName tenantName) {
        try (Lock lock = this.tenantLocks.lock((Object)tenantName);){
            this.writeTenantPath(tenantName);
            Tenant tenant = this.createTenant(tenantName, this.clock.instant());
            return tenant;
        }
    }

    public void createAndWriteTenantMetaData(Tenant tenant) {
        this.createWriteTenantMetaDataTransaction(this.createMetaData(tenant)).commit();
    }

    public Transaction createWriteTenantMetaDataTransaction(TenantMetaData tenantMetaData) {
        return new CuratorTransaction(this.curator).add((Transaction.Operation)CuratorOperations.setData((String)TenantRepository.getTenantPath(tenantMetaData.tenantName()).getAbsolute(), (byte[])tenantMetaData.asJsonBytes()));
    }

    private TenantMetaData createMetaData(Tenant tenant) {
        Instant deployTime = tenant.getSessionRepository().clock().instant();
        Instant createdTime = this.getTenantMetaData(tenant).createdTimestamp();
        if (createdTime.equals(Instant.EPOCH)) {
            createdTime = deployTime;
        }
        return new TenantMetaData(tenant.getName(), deployTime, createdTime);
    }

    public TenantMetaData getTenantMetaData(Tenant tenant) {
        Optional<Object> metaData;
        Optional data = this.getCurator().getData(TenantRepository.getTenantPath(tenant.getName()));
        try {
            metaData = data.map(bytes -> TenantMetaData.fromJsonString(tenant.getName(), Utf8.toString((byte[])bytes)));
        }
        catch (IllegalArgumentException e) {
            metaData = Optional.empty();
        }
        return metaData.orElse(new TenantMetaData(tenant.getName(), tenant.getCreatedTime(), tenant.getCreatedTime()));
    }

    private static Set<TenantName> readTenantsFromZooKeeper(Curator curator) {
        return curator.getChildren(tenantsPath).stream().map(TenantName::from).collect(Collectors.toSet());
    }

    private void bootstrapTenants() {
        HashMap futures = new HashMap();
        TenantRepository.readTenantsFromZooKeeper(this.curator).forEach(t -> futures.put(t, this.bootstrapExecutor.submit(() -> this.bootstrapTenant((TenantName)t))));
        HashSet<TenantName> failed = new HashSet<TenantName>();
        for (Map.Entry f : futures.entrySet()) {
            TenantName tenantName = (TenantName)f.getKey();
            try {
                ((Future)f.getValue()).get();
            }
            catch (ExecutionException e) {
                log.log(Level.WARNING, "Failed to create tenant " + tenantName, e);
                failed.add(tenantName);
            }
            catch (InterruptedException e) {
                log.log(Level.WARNING, "Interrupted while creating tenant '" + tenantName + "'", e);
            }
        }
        if (failed.size() > 0) {
            throw new RuntimeException("Could not create all tenants when bootstrapping, failed to create: " + failed);
        }
        this.metricUpdater.setTenants(this.tenants.size());
        this.bootstrapExecutor.shutdown();
        try {
            this.bootstrapExecutor.awaitTermination(365L, TimeUnit.DAYS);
        }
        catch (InterruptedException e) {
            throw new RuntimeException("Executor for creating tenants did not terminate within timeout");
        }
    }

    protected void bootstrapTenant(TenantName tenantName) {
        try (Lock lock = this.tenantLocks.lock((Object)tenantName);){
            this.createTenant(tenantName, this.readCreatedTimeFromZooKeeper(tenantName));
        }
    }

    public Instant readCreatedTimeFromZooKeeper(TenantName tenantName) {
        Optional stat = this.curator.getStat(TenantRepository.getTenantPath(tenantName));
        if (stat.isPresent()) {
            return Instant.ofEpochMilli(((Stat)stat.get()).getCtime());
        }
        return this.clock.instant();
    }

    private Tenant createTenant(TenantName tenantName, Instant created) {
        if (this.tenants.containsKey(tenantName)) {
            return this.getTenant(tenantName);
        }
        Instant start = Instant.now();
        log.log(Level.FINE, "Adding tenant '" + tenantName);
        TenantApplications applicationRepo = new TenantApplications(tenantName, this.curator, this.zkWatcherExecutor, this.zkCacheExecutor, this.metrics, this.reloadListener, this.configserverConfig, this.hostRegistry, new TenantFileSystemDirs(this.configServerDB, tenantName), this.clock);
        PermanentApplicationPackage permanentApplicationPackage = new PermanentApplicationPackage(this.configserverConfig);
        SessionPreparer sessionPreparer = new SessionPreparer(this.modelFactoryRegistry, this.fileDistributionFactory, this.hostProvisionerProvider, permanentApplicationPackage, this.configserverConfig, this.configDefinitionRepo, this.curator, this.zone, this.flagSource, this.secretStore);
        SessionRepository sessionRepository = new SessionRepository(tenantName, applicationRepo, sessionPreparer, this.configCurator, this.metrics, this.zkWatcherExecutor, permanentApplicationPackage, this.flagSource, this.zkCacheExecutor, this.secretStore, this.hostProvisionerProvider, this.configserverConfig, this.configServerDB, this.zone, this.clock, this.modelFactoryRegistry, this.configDefinitionRepo, this.tenantListener);
        log.log(Level.INFO, "Adding tenant '" + tenantName + "', created " + created + ". Bootstrapping in " + Duration.between(start, Instant.now()));
        Tenant tenant = new Tenant(tenantName, sessionRepository, applicationRepo, created);
        this.createAndWriteTenantMetaData(tenant);
        this.tenants.putIfAbsent(tenantName, tenant);
        this.notifyNewTenant(tenant);
        return tenant;
    }

    public Tenant defaultTenant() {
        try (Lock lock = this.tenantLocks.lock((Object)DEFAULT_TENANT);){
            Tenant tenant = this.tenants.get(DEFAULT_TENANT);
            return tenant;
        }
    }

    private void removeUnusedApplications() {
        this.getAllTenants().forEach(tenant -> tenant.getApplicationRepo().removeUnusedApplications());
    }

    private void notifyNewTenant(Tenant tenant) {
        this.tenantListener.onTenantCreate(tenant);
    }

    private void notifyRemovedTenant(TenantName name) {
        this.tenantListener.onTenantDelete(name);
    }

    private void createSystemTenants(ConfigserverConfig configserverConfig) {
        ArrayList<TenantName> systemTenants = new ArrayList<TenantName>();
        systemTenants.add(DEFAULT_TENANT);
        if (configserverConfig.hostedVespa()) {
            systemTenants.add(HOSTED_VESPA_TENANT);
        }
        for (TenantName tenantName : systemTenants) {
            try {
                this.writeTenantPath(tenantName);
            }
            catch (RuntimeException e) {
                if (e.getCause().getClass() == KeeperException.NodeExistsException.class) continue;
                throw e;
            }
        }
    }

    private void writeTenantPath(TenantName name) {
        try (Lock lock = this.tenantLocks.lock((Object)name);){
            this.curator.createAtomically(new Path[]{TenantRepository.getTenantPath(name), TenantRepository.getSessionsPath(name), TenantRepository.getApplicationsPath(name), TenantRepository.getLocksPath(name)});
        }
    }

    public void deleteTenant(TenantName name) {
        if (name.equals((Object)DEFAULT_TENANT)) {
            throw new IllegalArgumentException("Deleting 'default' tenant is not allowed");
        }
        if (!this.tenants.containsKey(name)) {
            throw new IllegalArgumentException("Deleting '" + name + "' failed, tenant does not exist");
        }
        log.log(Level.INFO, "Deleting tenant '" + name + "'");
        try (Lock lock = this.tenantLocks.lock((Object)name);){
            Path path = this.tenants.get(name).getPath();
            this.closeTenant(name);
            this.curator.delete(path);
        }
    }

    private void closeTenant(TenantName name) {
        try (Lock lock = this.tenantLocks.lock((Object)name);){
            Tenant tenant = this.tenants.remove(name);
            if (tenant == null) {
                throw new IllegalArgumentException("Closing '" + name + "' failed, tenant does not exist");
            }
            log.log(Level.INFO, "Closing tenant '" + name + "'");
            this.notifyRemovedTenant(name);
            tenant.close();
        }
    }

    public static String logPre(ApplicationId app) {
        if (DEFAULT_TENANT.equals((Object)app.tenant())) {
            return "";
        }
        StringBuilder ret = new StringBuilder().append(TenantRepository.logPre(app.tenant())).append("app:" + app.application().value()).append(":" + app.instance().value()).append(" ");
        return ret.toString();
    }

    public static String logPre(TenantName tenant) {
        if (DEFAULT_TENANT.equals((Object)tenant)) {
            return "";
        }
        StringBuilder ret = new StringBuilder().append("tenant:" + tenant.value()).append(" ");
        return ret.toString();
    }

    private void stateChanged(CuratorFramework framework, ConnectionState connectionState) {
        switch (connectionState) {
            case CONNECTED: {
                this.metricUpdater.incZKConnected();
                break;
            }
            case SUSPENDED: {
                this.metricUpdater.incZKSuspended();
                break;
            }
            case RECONNECTED: {
                this.metricUpdater.incZKReconnected();
                break;
            }
            case LOST: {
                this.metricUpdater.incZKConnectionLost();
                break;
            }
        }
    }

    private void childEvent(CuratorFramework framework, PathChildrenCacheEvent event) {
        switch (event.getType()) {
            case CHILD_ADDED: {
                TenantName t1 = this.getTenantNameFromEvent(event);
                if (this.tenants.containsKey(t1)) break;
                this.zkWatcherExecutor.execute((Object)t1, () -> this.bootstrapTenant(t1));
                break;
            }
            case CHILD_REMOVED: {
                TenantName t2 = this.getTenantNameFromEvent(event);
                if (!this.tenants.containsKey(t2)) break;
                this.zkWatcherExecutor.execute((Object)t2, () -> this.deleteTenant(t2));
                break;
            }
        }
        this.metricUpdater.setTenants(this.tenants.size());
    }

    private TenantName getTenantNameFromEvent(PathChildrenCacheEvent event) {
        String path = event.getData().getPath();
        String[] pathElements = path.split("/");
        if (pathElements.length == 0) {
            throw new IllegalArgumentException("Path " + path + " does not contain a tenant name");
        }
        return TenantName.from((String)pathElements[pathElements.length - 1]);
    }

    public void close() {
        this.directoryCache.ifPresent(Curator.DirectoryCache::close);
        try {
            this.zkCacheExecutor.shutdown();
            this.checkForRemovedApplicationsService.shutdown();
            this.zkWatcherExecutor.shutdownAndWait();
            this.zkCacheExecutor.awaitTermination(50L, TimeUnit.SECONDS);
            this.checkForRemovedApplicationsService.awaitTermination(50L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            log.log(Level.WARNING, "Interrupted while shutting down.", e);
            Thread.currentThread().interrupt();
        }
    }

    public boolean checkThatTenantExists(TenantName tenant) {
        return this.tenants.containsKey(tenant);
    }

    public Tenant getTenant(TenantName tenantName) {
        return this.tenants.get(tenantName);
    }

    public Set<TenantName> getAllTenantNames() {
        return ImmutableSet.copyOf(this.tenants.keySet());
    }

    public Collection<Tenant> getAllTenants() {
        return ImmutableSet.copyOf(this.tenants.values());
    }

    public static Path getTenantPath(TenantName tenantName) {
        return tenantsPath.append(tenantName.value());
    }

    public static Path getSessionsPath(TenantName tenantName) {
        return TenantRepository.getTenantPath(tenantName).append("sessions");
    }

    public static Path getApplicationsPath(TenantName tenantName) {
        return TenantRepository.getTenantPath(tenantName).append("applications");
    }

    public static Path getLocksPath(TenantName tenantName) {
        return locksPath.append(tenantName.value());
    }

    public static Path getBarriersPath() {
        return barriersPath;
    }

    public Curator getCurator() {
        return this.curator;
    }
}

